package Cuitter;
use vars qw($VERSION);
$VERSION = 0.001003;

use Mouse;
use Net::Netrc;
use Net::Twitter::Lite;
use IO::Handle;
use Encode;
use POSIX qw( strftime );
use DateTime::Format::DateParse;
use Cache::FileCache;
use Data::Dumper;

with qw( MouseX::Getopt MouseX::Object::Pluggable );

has username => ( is => 'rw', isa => 'Str' );
has client => ( is => 'rw', isa => 'Object' );
has input => ( is => 'rw', isa => 'FileHandle' );
has output => ( is => 'rw', isa => 'FileHandle' );
has help => ( is => 'rw', isa => 'HashRef' );
has status => ( is => 'rw', isa => 'ArrayRef', clearer => 'clear_status' );
has rows => ( is => 'rw', isa => 'Int' );
has plugins => ( is => 'ro', isa => 'ArrayRef' );
has cache => ( is => 'rw', isa => 'Object' );

sub BUILD {
    my $self = shift;
    $self->help( {} );
    $| = 1;
    $self->load_plugins( 'OAuth' );
    if ( $self->plugins ) {
        $self->load_plugins( @{ $self->plugins } );
        for my $plugin ( @{ $self->plugins } ) {
            my $init_method = "_init_$plugin";
            $self->$init_method;
        }
    }
    $self->rows( 6 ) unless $self->rows;
    $self->input( IO::Handle->new );
    $self->input->fdopen( fileno( STDIN ), 'r' );
    $self->output( IO::Handle->new );
    $self->output->fdopen( fileno( STDOUT ), 'w' );
    $self->set_help( 'help', '( command ) show help of command.' );
    $self->set_help( 'exit', 'exit from cuitter' );
    $self->set_help( 'ft', 'show friends time-line.' );
    $self->cache( Cache::FileCache->new( { namespace => __PACKAGE__ } ) );
    $self->login;
    $self->loop if $self->username;
}

sub load_config {
    my $self = shift;
    my $config = LoadFile( $self->configfile ) if -e $self->configfile;
    for my $key ( keys %{ $config } ) {
        $self->$key( $config->{ "$key" } ) if $self->can( "$key" );
    }
}

sub login {
    my $self = shift;
    if ( ! Net::Netrc->lookup( 'twitter.com' ) ) {
        $self->_netrc_init;
        $self->output->print( "please restart cuitter.\n" );
    } 
    else {
        $self->_client_init(
            netrc => 1,
            clientname => __PACKAGE__,
            clientver => $VERSION,
            clienturl => 'http://cuitter.sourceforge.jp/',
            useragent => __PACKAGE__.'/'.$VERSION,
            source => 'cuitter'
        );
        $self->username( $self->client->{ username } ) unless $self->username;
        eval { $self->client->friends_timeline };
        if ( $@ ) {
            warn $@."\n";
            unlink $ENV{ HOME }.'/.netrc' if $@ =~ /^Could not authenticate you.$/;
            exit;
        }
    }
}

sub _client_init {
    my ( $self, %arg ) = @_;
    $self->client( Net::Twitter::Lite->new( %arg ) );
}

sub _netrc_init {
    my $self = shift;
    my ( $account, $password ) = @_;
    while ( !$account ) {
        $self->output->print( "What's your twitter-account? : " );
        $account = $self->input->getline;
        chomp $account;
    }
    while ( !$password ) {
        $self->output->print( "Password : " );
        $password = $self->input->getline;
        chomp $password;
    }
    my $netrc = $ENV{ HOME }. '/.netrc';
    open my $fh, '>>', $netrc;
    print $fh "machine twitter.com\nlogin $account\npassword $password\n";
    close $fh;
    chmod 0600, $netrc;
    $self->output->print( "saved to ". $ENV{ HOME }. '/.netrc'. "\n" );
}

sub prompt {
    my $self = shift;
    $self->output->print( '@'.$self->username.": " );
    my $line = $self->input->getline;
    chomp $line;
    $line = '/ft' unless $line;
    if ( $line =~ /^\// ) {
        $line =~ s/^\///;
        my ( $command, @arg ) = split / /, $line;
        my $exec = "com_$command";
        if ( $self->can( "$exec" ) ) {
            return $self->$exec( @arg );
        }
        else {
            $self->output->print( "$command ? I don't know it.\n" );
            return 1;
        }
    }
    else {
        $self->tweet( $line );
        $self->com_ft;
        return 1;
    }
}

sub loop {
    my $self = shift;
    while ( $self->prompt ) {};
}

sub tweet {
    my ( $self, $line ) = @_;
    $self->client->update( $line );
}

sub set_help {
    my ( $self, $name, $comment ) = @_;
    $self->help->{ "$name" } = $comment;
}

sub get_help {
    my ( $self, $name ) = @_;
    if ( $name ) {
        return "$name : ". $self->help->{ "$name" }."\n";
    }
    else {
        my $rtn = "";
        for my $key ( keys %{ $self->help } ) {
            $rtn .= "/$key : ".$self->help->{ "$key" }."\n";
        }
        return $rtn;
    }
}

sub com_help {
    my ( $self, $command ) = @_;
    $self->output->print( $self->get_help( $command ) );
    1;
}

sub com_exit {
    my $self = shift;
    $self->output->print( "Sayonara.\n" );
    return;
}
sub com_q { shift->com_exit }
sub com_quit { shift->com_exit }
sub com_bye { shift->com_exit }
sub com_sayonara { shift->com_exit }

sub com_ft {
    my ( $self, $rows ) = @_;
    $self->rows( $rows ) if $rows;
    $rows ||= $self->rows;
    $self->update_status( $rows );
    for my $tweet ( @{ $self->status } ) {
        $self->output->print( $tweet->{ display } );
    }
    $self->output->print( '--- Last Update: '. strftime( '%Y-%m-%d %H:%M:%S', localtime() )." ---\n" );
    1;
}

sub update_status {
    my ( $self, $rows ) = @_;
    my $id = 0;
    $self->clear_status;
    $self->status( [] );
    for my $tweet ( @{ $self->client->friends_timeline( { rows => $rows } ) } ) {
        $tweet->{ created_at } = DateTime::Format::DateParse->parse_datetime( $tweet->{ created_at } );
        my $delaysec = time() - $tweet->{ created_at }->epoch;
        $tweet->{ display } = $self->_build_display( $id, $delaysec, $tweet );
        push @{ $self->status }, $tweet;
        $id++;
        last if $id >= $rows;
    }
}

sub _build_display {
    my ( $self, $id, $delaysec, $tweet ) = @_;
    my $delaymin = int( $delaysec / 60 );
    my $delayhour = int( $delaymin / 60 );
    my $delayday = int( $delayhour / 24 );
    $delayhour -= $delayday * 24 if $delayday;
    $delaymin -= $delayhour * 60 if $delayhour;
    $delaysec -= $delaymin * 60 if $delaymin;
    my $delay = $delayday."Day ".sprintf("%02d",$delayhour).":".sprintf("%02d",$delaymin).":".sprintf("%02d",$delaysec);
    "---\n[$id] ". $delay. " ago\n". '@'.$tweet->{user}{screen_name}. ': '. encode( "utf8", $tweet->{text} ). "\n";
}

__PACKAGE__->meta->make_immutable;

no Mouse;

1;
__END__

=head1 NAME

Cuitter - Yet Another CUI Based Twitter Client

=head1 SYNOPSIS

On your shell, execute like following.

  $ cuitter [ --plugins (SomePluginNames) ]

=head1 DESCRIPTION

Cuitter is a CUI Based Twitter Client.

=head1 Look and Feel

=head2 Prompt

 @ytnobody:

=head2 Friend Timeline

 ---
 [0] 0Day 00:00:34 ago
 @ytnobody: なにのせるの？ RT @ne********: コンテナのような規格物であれば、すんなりシャシーに載せてフェリってもらえる。
 ---
 [1] 0Day 00:03:02 ago
 @ne********: コンテナのような規格物であれば、すんなりシャシーに載せてフェリってもらえる。
 ---
 [2] 0Day 00:04:03 ago
 @su******: とてもしょんぼりしたので、もう少しお酒を飲んで気分をごまかしたいと思います。バルス！
 ---
 [3] 0Day 00:06:28 ago
 @hi******: 『少子化 + 書籍離れ + ゲーム + などなど』で 2 桁の落ち込みはリーズナブルな数値では?
 ---
 [4] 0Day 00:06:41 ago
 @ne********: ナイスエアうどんw RT @tw***: そうそう、もちっとした食感、つるっとしたのどごし。２玉ぐらいならするする～っと！
 って。なにさせるんですか。T @ne********: エアうどんを2玉どうぞ #udon
 ---
 [5] 0Day 00:08:40 ago
 @ne********: 20ftのコンテナ2本くらい欲しいなぁ
 --- Last Update: 2009-11-21 01:01:46 ---

=head3 a tweet

For example, look following.

 ---
 [1] 0Day 00:03:02 ago
 @ne********: コンテナのような規格物であれば、すんなりシャシーに載せてフェリってもらえる。

status-label is

 [1]

tweeted at 

 0Day 00:03:02 ago

tweet by

 @ne********

tweet body 

 コンテナのような規格物であれば、すんなりシャシーに載せてフェリってもらえる。

=head1 Commands

=head2 /help [ command ]

show help

=head2 /exit /q /bye

exit from cuitter

=head2 /ft [ maxrows ]

show friends timeline

=head1 LICENSE

Artistic

=head1 AUTHOR

Satoshi Azuma E<lt>ytnobody@gmail.comE<gt>

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
