/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.util;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.trix.cuery.dom.stylesheets.StyleSheetImpl;
import org.trix.cuery.dom.stylesheets.StyleSheetListImpl;
import org.trix.cuery.filter.ClassFilter;
import org.trix.cuery.filter.ElementFilter;
import org.trix.cuery.filter.Filter;
import org.trix.cuery.filter.IDFilter;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.stylesheets.StyleSheetList;

/**
 * This class provides helper methods for operation and reference of DOM. The point where this class
 * is different from a general DOM utility class is that helper methods considers the special node
 * called as 'Vanishing Node'. The 'Vanishing Node' is a node to which it has been decided to be
 * going to be deleted in the future by a certain kind of node operation. To make a state of the
 * node 'Vanishing Node', you can use the method 'setVanishing(Node)' in this class.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: DOMUtil.java,v 1.05 2005/12/06 12:38:37 Teletha Exp $
 */
public final class DOMUtil {

    /** The empty elements. */
    public static final Element[] EMPTY_ELEMETS = new Element[0];

    /** The key for vanishing node. */
    private static final String KEY_VANISHING_NODE = DOMUtil.class.getName() + "Vanishing";

    /**
     * Avoid creating DOMUtil instance.
     */
    private DOMUtil() {
    }

    /**
     * Retrieve all descendant elements which are filtered by the class name.
     * 
     * @param source A source node.
     * @param className A class name.
     * @return All matched descendant elements.
     */
    public static Set getElementsByClassName(Node source, String className) {
        return retrieveElements(source, new ClassFilter(className), true);
    }

    /**
     * Retrieve a element which are filtered by the id. If there is no such element, return null.
     * 
     * @param source A source node.
     * @param id An id.
     * @return A matched element.
     */
    public static Element getElementById(Node source, String id) {
        Set result = retrieveElements(source, new IDFilter(id), true);

        if (result.size() == 1) {
            return (Element) result.iterator().next();
        } else {
            return null;
        }
    }

    /**
     * Retrieve all descendant elements which are filtered by the tag name. Tag name accepts null or
     * zero length value, and these are treated as an asterisk.
     * 
     * @param source A source node.
     * @param tagName A tag name.
     * @return All matched descendant elements.
     */
    public static Set getElementsByTagName(Node source, String tagName) {
        return retrieveElements(source, new ElementFilter(tagName), true);
    }

    /**
     * Return a parent element. If the target node has no parent, return null.
     * 
     * @param target A target node.
     * @return A parent element or null.
     */
    public static Element getParentElement(Node target) {
        // assert null
        if (target == null) {
            return null;
        }

        Node parent = target.getParentNode();

        if (isValidElement(parent)) {
            return (Element) parent;
        }
        return null;
    }

    /**
     * Retrieve a previous sibling element. If there is no such element, return null.
     * 
     * @param source A source node to start searching.
     * @return A matched element.
     */
    public static Element getPreviousElement(Node source) {
        // check null
        if (source == null) {
            return null;
        }

        Node previous = source.getPreviousSibling();

        if (isValidElement(previous)) {
            return (Element) previous;
        }
        return getPreviousElement(previous);
    }

    /**
     * Retrieve a next sibling element. If there is no such element, return null.
     * 
     * @param source A source node to start searching.
     * @return A matched element.
     */
    public static Element getNextElement(Node source) {
        // check null
        if (source == null) {
            return null;
        }

        Node next = source.getNextSibling();

        if (isValidElement(next)) {
            return (Element) next;
        }
        return getNextElement(next);
    }

    /**
     * Return stylesheet list from document.
     * 
     * @param document A target document.
     * @return A list of stylesheet.
     */
    public static StyleSheetList getStylesheets(Document document) {
        NodeList children = document.getChildNodes();
        StyleSheetListImpl list = new StyleSheetListImpl();

        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);

            if (child.getNodeType() == Node.PROCESSING_INSTRUCTION_NODE) {
                ProcessingInstruction pi = (ProcessingInstruction) child;
                list.addStyleSheet(new StyleSheetImpl(pi));
            }
        }
        return list;
    }

    /**
     * Return element position in parent element.
     * 
     * @param element A target element.
     * @return A position.
     */
    public static int getPosition(Element element) {
        // check null
        if (element == null) {
            return -1;
        }

        // check parent
        Node parent = element.getParentNode();

        if (!isValidElement(parent)) {
            return -1;
        }

        int position = 1;
        NodeList nodeList = parent.getChildNodes();

        for (int i = 0; i < nodeList.getLength(); i++) {
            Node current = nodeList.item(i);

            // check element
            if (!isValidElement(current)) {
                continue;
            }

            // cehck
            if (current == element) {
                return position;
            }
            position++;
        }
        return -1;
    }

    /**
     * Return typed element position in parent element.
     * 
     * @param element A target element.
     * @return A position.
     */
    public static int getTypedPosition(Element element) {
        // check null
        if (element == null) {
            return -1;
        }

        // check parent
        Node parent = element.getParentNode();

        if (!isValidElement(parent)) {
            return -1;
        }

        int position = 1;
        String ns = element.getNamespaceURI();
        String name = element.getLocalName();
        NodeList nodeList = parent.getChildNodes();

        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);

            // check element
            if (!isValidElement(node)) {
                continue;
            }

            Element current = (Element) node;

            // check namespace
            if (ns != null && !ns.equals(current.getNamespaceURI())) {
                continue;
            }

            // check local name
            if (!current.getLocalName().equals(name)) {
                continue;
            }

            // cehck
            if (current == element) {
                return position;
            }
            position++;
        }
        return -1;
    }

    /**
     * Check whether this element has a specified token attribute or not.
     * 
     * @param target A target element.
     * @param name A attribute name.
     * @param token A attribute value.
     * @return A result.
     */
    public static boolean hasToken(Element target, String name, String token) {
        // check null
        if (target == null || name == null || name.length() == 0 || token == null || token.length() == 0) {
            return false;
        }

        String attribute = target.getAttribute(name);

        if (attribute == null || attribute.length() < token.length()) {
            return false;
        }

        String[] values = attribute.split(" ");

        for (int i = 0; i < values.length; i++) {
            if (values[i].equals(token)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Retrieve all filtered elements.
     * 
     * @param source A source node.
     * @param filter A filter.
     * @param descendant A flag whether the descendant elements are parsed or not.
     * @return All matched elements.
     */
    public static Set retrieveElements(Node source, Filter filter, boolean descendant) {
        // check null
        if (source == null) {
            return Collections.EMPTY_SET;
        }

        // check node type
        if (source.getNodeType() == Node.DOCUMENT_NODE || source.getNodeType() == Node.ELEMENT_NODE) {
            Set container = new HashSet();
            retrieveElements(container, source, filter, descendant);
            return container;
        }
        return Collections.EMPTY_SET;
    }

    /**
     * Helper method to collect elements under some filters.
     * 
     * @param container A matched element container.
     * @param source Start searching from this position.
     * @param filter A catch-all filter.
     * @param descendant A flag whether the descendant elements are parsed or not.
     */
    private static void retrieveElements(Set container, Node source, Filter filter, boolean descendant) {
        // check child
        if (!source.hasChildNodes()) {
            return;
        }

        NodeList children = source.getChildNodes();

        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);

            if (isValidElement(child)) {
                Element element = (Element) child;

                // recursive collection
                if (descendant) {
                    retrieveElements(container, element, filter, descendant);
                }

                // filter
                if (!filter.accept(element)) {
                    continue;
                }
                container.add(element);
            }
        }
    }

    /**
     * Set the given node as vanishing node.
     * 
     * @param node A target node.
     */
    public static void setVanishing(Node node) {
        // assert null
        if (node == null) {
            return;
        }
        node.setUserData(KEY_VANISHING_NODE, KEY_VANISHING_NODE, null);
    }

    /**
     * Check whether the given node is valid element or not. The given node must not be Vanishing
     * node.
     * 
     * @param node A target node to check.
     * @return A result.
     */
    private static boolean isValidElement(Node node) {
        // assert null
        if (node == null) {
            return false;
        }

        // assert node type
        if (node.getNodeType() != Node.ELEMENT_NODE) {
            return false;
        }

        // assert vanishing node
        return node.getUserData(KEY_VANISHING_NODE) == null;
    }
}
