/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import org.trix.cuery.CSS;
import org.trix.cuery.util.I18nUtil;

import org.w3c.css.sac.CSSException;
import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: PseudoElementFilter.java,v 1.02 2005/12/25 09:26:47 Teletha Exp $
 */
public class PseudoElementFilter extends AbstractPseudoFilter {

    /** The name of pseudo elements in CSS2. */
    private static final String[] PSEUDO_ELEMENTS_CSS2 = {"after", "before", "first-letter", "first-line"};

    /** The name of pseudo elements in CCS3. */
    private static final String[] PSEUDO_ELEMENTS_CSS3 = {"alternate", "choices", "line-maker", "maker", "outside",
            "repeat-index", "repeat-item", "selection", "value"};

    /** The css level. */
    private int level;

    /** The pseudo name. */
    private String name;

    /**
     * Create PseudoElementFilter instance.
     * 
     * @param namespaceURI A namespace URI for CSS level.
     * @param name A pseudo name.
     */
    public PseudoElementFilter(String namespaceURI, String name) {
        // assert null
        if (namespaceURI == null || namespaceURI.length() == 0 || name == null || name.length() == 0) {
            throw new CSSException(I18nUtil.getText("parser.invalidPseudo", name));
        }

        // CSS Level2
        for (int i = 0; i < PSEUDO_ELEMENTS_CSS2.length; i++) {
            if (name.equals(PSEUDO_ELEMENTS_CSS2[i])) {
                level = 2;
                this.name = name;
                return;
            }
        }

        if (namespaceURI.equals(CSS.LEVEL2)) {
            throw new CSSException(I18nUtil.getText("parser.invalidPseudo", name));
        }

        // CSS Level3
        for (int i = 0; i < PSEUDO_ELEMENTS_CSS3.length; i++) {
            if (name.equals(PSEUDO_ELEMENTS_CSS3[i])) {
                level = 3;
                this.name = name;
                return;
            }
        }

        // Not defined
        throw new CSSException(I18nUtil.getText("parser.invalidPseudo", name));
    }

    /**
     * @see org.trix.cuery.filter.Filter#accept(org.w3c.dom.Element)
     */
    public boolean accept(Element target) {
        return false;
    }

    /**
     * @see org.trix.cuery.filter.AbstractPseudoFilter#getSpecificity()
     */
    public int getSpecificity() {
        return 0;
    }

    /**
     * @see org.trix.cuery.filter.AbstractPseudoFilter#toString()
     */
    public String toString() {
        if (level == 2) {
            return ":" + name;
        } else {
            return "::" + name;
        }
    }

    /**
     * @see org.trix.cuery.filter.AbstractPseudoFilter#getPseudoName()
     */
    protected String getPseudoName() {
        return name;
    }
}
