/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property;

import java.util.LinkedList;

import org.trix.cuery.util.CSSUtil;
import org.trix.cuery.value.CSSValue;

import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CascadableProperty.java,v 1.02 2005/08/19 14:12:41 Teletha Exp $
 */
public class CascadableProperty extends AbstractProperty {

    /** This is the UA's default stylesheet. */
    public static final int ORIGIN_AGENT = 1;

    /** This is the user's stylesheet. */
    public static final int ORIGIN_USER = 2;

    /** This is the author's stylesheet. */
    public static final int ORIGIN_AUTHOR = 3;

    /** The list of styles. */
    private LinkedList properties = new LinkedList();

    /** The position of a seperator between author and user. */
    private int author = 0;

    /** The position of a seperator between user and agent. */
    private int user = 0;

    /** The parent element. */
    private Element parent;

    /**
     * Create CascadableProperty instance.
     */
    public CascadableProperty() {
    }

    /**
     * Create CascadableProperty instance.
     * 
     * @param parent A parent element.
     */
    public CascadableProperty(Element parent) {
        this.parent = parent;
    }

    /**
     * @see org.trix.cuery.property.Property#getValue(java.lang.String)
     */
    public CSSValue getValue(String name) {
        PropertyDefinition definition = PropertyRegistry.getDefinition(name);
        CSSValue value = getSpecifiedValue(definition);

        Property parent = CSSUtil.getProperty(this.parent);

        if (parent == null) {
            return definition.getComputedValue(value, this, RootProperty.SINGLETON);
        } else {
            return definition.getComputedValue(value, this, parent);
        }
    }

    /**
     * Assign a specified value to a property. See also <a
     * href="http://www.w3.org/TR/REC-CSS2/cascade.html#specified-value">the mechanisms</a>.
     * 
     * @param definition A property definition.
     * @return A specified value.
     */
    private CSSValue getSpecifiedValue(PropertyDefinition definition) {
        String name = definition.getName();

        // If the cascade results in a value, use it.
        // search in user stylesheet with important priority
        for (int i = author; i < user; i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;

            if (property.isImportant(name)) {
                return property.getValue(name);
            }
        }

        // search in author stylesheet with important priority
        for (int i = 0; i < author; i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;

            if (property.isImportant(name)) {
                return property.getValue(name);
            }
        }

        // search in author stylesheet
        for (int i = 0; i < author; i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;
            CSSValue value = property.getValue(name);

            if (value != null) {
                return value;
            }
        }

        // search in user stylesheet
        for (int i = author; i < user; i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;
            CSSValue value = property.getValue(name);

            if (value != null) {
                return value;
            }
        }

        // search in agent stylesheet
        for (int i = user; i < properties.size(); i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;
            CSSValue value = property.getValue(name);

            if (value != null) {
                return value;
            }
        }

        // Otherwise, if the property is inherited, use the value of the parent element, generally
        // the computed value.
        if (parent != null && definition.isInheritable()) {
            return CSSUtil.getProperty(parent).getValue(name);
        }

        // Otherwise use the property's initial value. The initial value of each property is
        // indicated in the property's definition.
        return definition.getInitialValue();
    }

    /**
     * @see org.trix.cuery.property.Property#isImportant(java.lang.String)
     */
    public boolean isImportant(String name) {
        return false;
    }

    /**
     * Add property.
     * 
     * @param property A target property.
     * @param origin A origin type.
     * @param specificity A specificity of this property.
     * @param position A position of this property.
     */
    public void addProperty(Property property, int origin, int specificity, int position) {
        switch (origin) {
        case ORIGIN_AUTHOR:
            addProperty(property, specificity, position, 0, author);
            author++;
            user++;
            break;

        case ORIGIN_USER:
            addProperty(property, specificity, position, author, user);
            user++;
            break;

        case ORIGIN_AGENT:
            addProperty(property, specificity, position, user, properties.size());
            break;

        default:
            break;
        }
    }

    /**
     * Add property to a suitable position in list of properties.
     * 
     * @param property A target property.
     * @param specificity A specificity of this property.
     * @param position A position of this property.
     * @param start A start position for this property origin.
     * @param end A end position for this property origin.
     */
    private void addProperty(Property property, int specificity, int position, int start, int end) {
        PropertyWeightSet set = new PropertyWeightSet(property, specificity, position);

        for (int i = start; i < end; i++) {
            PropertyWeightSet target = (PropertyWeightSet) properties.get(i);

            // check specificity
            if (specificity > target.specificity) {
                properties.add(i, set);
                return;
            }

            // check position
            if (specificity == target.specificity && position > target.position) {
                properties.add(i, set);
                return;
            }
        }
        // lowest weight
        properties.add(end, set);
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: PropertyWeightSet.java,v 1.0 2005/08/25 16:42:35 Teletha Exp $
     */
    private class PropertyWeightSet {

        /** The actual property. */
        private Property property;

        /** The specificity of this property. */
        private int specificity;

        /** The position of this property. */
        private int position;

        /**
         * Create PropertyWeightSet instance.
         * 
         * @param property A target property.
         * @param specificity A specificity of this property.
         * @param position A position of this property.
         */
        public PropertyWeightSet(Property property, int specificity, int position) {
            this.property = property;
            this.specificity = specificity;
            this.position = position;
        }
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: RootProperty.java,v 1.0 2005/09/09 1:12:24 Teletha Exp $
     */
    private static class RootProperty extends AbstractProperty {

        /** The singleon instance. */
        private static final RootProperty SINGLETON = new RootProperty();

        /**
         * @see org.trix.cuery.property.Property#getValue(java.lang.String)
         */
        public CSSValue getValue(String name) {
            PropertyDefinition definition = PropertyRegistry.getDefinition(name);
            return definition.getComputedValue(definition.getInitialValue(), this, this);
        }

        /**
         * @see org.trix.cuery.property.Property#isImportant(java.lang.String)
         */
        public boolean isImportant(String name) {
            return false;
        }
    }
}
