/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import org.w3c.css.sac.ElementSelector;
import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: ElementFilter.java,v 1.1 2005/08/04 21:33:04 Teletha Exp $
 */
public class ElementFilter implements Filter, ElementSelector {

    /** The namespace prefix. */
    private String prefix;

    /** The namespace URI. */
    private String namespaceURI;

    /** The local name. */
    private String localName;

    /**
     * Create ElementFilter instance.
     * 
     * @param localName A local name.
     */
    public ElementFilter(String localName) {
        this(null, null, localName);
    }

    /**
     * Create ElementFilter instance.
     * 
     * @param prefix A namespace prefix.
     * @param namespaceURI A namespace URI.
     * @param localName A local name.
     */
    public ElementFilter(String prefix, String namespaceURI, String localName) {
        this.prefix = prefix;
        this.namespaceURI = namespaceURI;
        this.localName = localName;

        // check asterisk
        if (localName == null) {
            this.localName = "*";
        }
    }

    /**
     * @see org.trix.cuery.filter.Filter#accept(org.w3c.dom.Element)
     */
    public boolean accept(Element target) {
        if (namespaceURI != null && !namespaceURI.equals(target.getNamespaceURI())) {
            return false;
        }

        if (localName.equals("*")) {
            return true;
        }
        return localName.equals(target.getLocalName());
    }

    /**
     * @see org.trix.cuery.filter.Filter#getSpecificity()
     */
    public int getSpecificity() {
        if (localName.equals("*")) {
            return 0;
        } else {
            return 1;
        }
    }

    /**
     * @see org.w3c.css.sac.Condition#getConditionType()
     */
    public short getConditionType() {
        return SAC_NEGATIVE_CONDITION;
    }

    /**
     * @see org.w3c.css.sac.Selector#getSelectorType()
     */
    public short getSelectorType() {
        return SAC_ELEMENT_NODE_SELECTOR;
    }

    /**
     * @see org.w3c.css.sac.ElementSelector#getLocalName()
     */
    public String getLocalName() {
        return localName;
    }

    /**
     * @see org.w3c.css.sac.ElementSelector#getNamespaceURI()
     */
    public String getNamespaceURI() {
        return namespaceURI;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        if (prefix == null || prefix.length() == 0) {
            return localName;
        } else {
            return prefix + "|" + localName;
        }
    }

}
