/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.trix.cuery.css.AbstractCSSConsumer;
import org.trix.cuery.parser.CueryParser;
import org.trix.cuery.property.PropertyDefinition;
import org.trix.cuery.property.PropertyRegistry;
import org.trix.cuery.util.CSSUtil;
import org.trix.cuery.value.CSSValue;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.CSSParseException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CSSValidator.java,v 1.0 2005/08/29 4:30:15 Teletha Exp $
 */
public class CSSValidator {

    /** The css parser. */
    private static final CueryParser PARSER = new CueryParser();

    /**
     * Validate css.
     * 
     * @param source A target css.
     * @return A list of invalid points.
     * @throws IOException If this css has I/O error.
     */
    public CSSParseException[] validate(InputSource source) throws IOException {
        ValidatableConsumer consumer = new ValidatableConsumer();
        PARSER.setDocumentHandler(consumer);
        PARSER.parseStyleSheet(source);

        return (CSSParseException[]) consumer.container.toArray(new CSSParseException[consumer.container.size()]);
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: ValidatableConsumer.java,v 1.0 2005/08/29 4:42:44 Teletha Exp $
     */
    private static class ValidatableConsumer extends AbstractCSSConsumer {

        /** The error container. */
        private List container = new ArrayList();

        /**
         * @see org.w3c.css.sac.DocumentHandler#property(java.lang.String,
         *      org.w3c.css.sac.LexicalUnit, boolean)
         */
        public void property(String name, LexicalUnit value, boolean important) throws CSSException {
            PropertyDefinition definition = PropertyRegistry.getDefinition(name);

            // no definition
            if (definition == null) {
                addError("This property '" + name + "' is not defined.");
                return;
            }

            CSSValue cssValue = CSSUtil.convert(value);
            Map result = definition.parse(cssValue);

            if (result == null) {
                addError("This property '" + name + "' has a invalid value. '" + CSSUtil.toString(cssValue) + "'");
                return;
            }
        }

        /**
         * Add an error.
         * 
         * @param message An error message
         */
        private void addError(String message) {
            CSSParseException exception = new CSSParseException(message, PARSER.getLocator());
            container.add(exception);
        }
    }
}
