(*
 * Copyright (C) 2002 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobN.
 *
 * ÍobN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit gcryptUnit;

interface

uses
  SysUtils;

type
  EGCrypt = class(Exception)
  public
    constructor Create(ErrCode: Integer);
  end;

  GCRY_CTL_CMDS = (
    GCRYCTL_SET_KEY                   = 1,
    GCRYCTL_SET_IV                    = 2,
    GCRYCTL_CFB_SYNC                  = 3,
    GCRYCTL_RESET                     = 4,   (* e.g. for MDs *)
    GCRYCTL_FINALIZE                  = 5,
    GCRYCTL_GET_KEYLEN                = 6,
    GCRYCTL_GET_BLKLEN                = 7,
    GCRYCTL_TEST_ALGO                 = 8,
    GCRYCTL_IS_SECURE                 = 9,
    GCRYCTL_GET_ASNOID                = 10,
    GCRYCTL_ENABLE_ALGO               = 11,
    GCRYCTL_DISABLE_ALGO              = 12,
    GCRYCTL_DUMP_RANDOM_STATS         = 13,
    GCRYCTL_DUMP_SECMEM_STATS         = 14,
    GCRYCTL_GET_ALGO_NPKEY            = 15,
    GCRYCTL_GET_ALGO_NSKEY            = 16,
    GCRYCTL_GET_ALGO_NSIGN            = 17,
    GCRYCTL_GET_ALGO_NENCR            = 18,
    GCRYCTL_SET_VERBOSITY             = 19,
    GCRYCTL_SET_DEBUG_FLAGS           = 20,
    GCRYCTL_CLEAR_DEBUG_FLAGS         = 21,
    GCRYCTL_USE_SECURE_RNDPOOL        = 22,
    GCRYCTL_DUMP_MEMORY_STATS         = 23,
    GCRYCTL_INIT_SECMEM               = 24,
    GCRYCTL_TERM_SECMEM               = 25,
    GCRYCTL_DISABLE_SECMEM_WARN       = 27,
    GCRYCTL_SUSPEND_SECMEM_WARN       = 28,
    GCRYCTL_RESUME_SECMEM_WARN        = 29,
    GCRYCTL_DROP_PRIVS	              = 30,
    GCRYCTL_ENABLE_M_GUARD            = 31,
    GCRYCTL_START_DUMP                = 32,
    GCRYCTL_STOP_DUMP                 = 33,
    GCRYCTL_GET_ALGO_USAGE            = 34,
    GCRYCTL_IS_ALGO_ENABLED           = 35,
    GCRYCTL_DISABLE_INTERNAL_LOCKING  = 36,
    GCRYCTL_DISABLE_SECMEM            = 37,
    GCRYCTL_INITIALIZATION_FINISHED   = 38,
    GCRYCTL_INITIALIZATION_FINISHED_P = 39,
    GCRYCTL_ANY_INITIALIZATION_P      = 40
  );

function gcry_control(cmd: GCRY_CTL_CMDS): Integer; cdecl; varargs;

///////////////////////////////////////////////////////////////////////////////
//
// multi precision integer functions
//
///////////////////////////////////////////////////////////////////////////////
type
  GCRY_MPI_FORMAT = (
    GCRYMPI_FMT_NONE= 0,
    GCRYMPI_FMT_STD = 1,    (* twos complement stored without length *)
    GCRYMPI_FMT_PGP = 2,    (* As used by OpenPGP (only defined as unsigned)*)
    GCRYMPI_FMT_SSH = 3,    (* As used by SSH (same as 1 but with length)*)
    GCRYMPI_FMT_HEX = 4,    (* hex format *)
    GCRYMPI_FMT_USG = 5     (* like STD but this is an unsigned one *)
  );

  GCRY_MPI_FLAG = (
    GCRYMPI_FLAG_SECURE = 1,  (* Allocate the number in "secure" memory. *)
    GCRYMPI_FLAG_OPAQUE = 2   (* The number is not a real one but just a
                               way to store some bytes.  This is
                               useful for encrypted big integers. *)
  );

  PGCRY_MPI = ^GCRY_MPI;
  GCRY_MPI = Pointer;

  PPointer = ^Pointer;

function gcry_mpi_scan(ret_mpi: PGCRY_MPI; format: GCRY_MPI_FORMAT; const buffer: PByte; nbytes: PCardinal): Integer; cdecl;
function gcry_mpi_aprint(format: GCRY_MPI_FORMAT; buffer: PPointer; nbytes: PCardinal; const a: GCRY_MPI): Integer; cdecl;
procedure gcry_mpi_release(a: GCRY_MPI); cdecl;

///////////////////////////////////////////////////////////////////////////////
//
// S-expression management.
//
///////////////////////////////////////////////////////////////////////////////
type
  PGCRY_SEXP = ^GCRY_SEXP;
  GCRY_SEXP = Pointer;

  GCRY_SEXP_FORMAT = (
    GCRYSEXP_FMT_DEFAULT   = 0,
    GCRYSEXP_FMT_CANON	   = 1,
    GCRYSEXP_FMT_BASE64    = 2,
    GCRYSEXP_FMT_ADVANCED  = 3
  );

function gcry_sexp_build(retsexp: PGCRY_SEXP; erroff: PCardinal; const format: PChar): Integer; cdecl; varargs;
procedure gcry_sexp_release(sexp: GCRY_SEXP); cdecl;
procedure gcry_sexp_dump(const a: GCRY_SEXP); cdecl;
function gcry_sexp_find_token(list: GCRY_SEXP; const tok: PChar; toklen: Cardinal): GCRY_SEXP; cdecl;
function gcry_sexp_cadr(const a: GCRY_SEXP): GCRY_SEXP; cdecl;
function gcry_sexp_nth_mpi(list: GCRY_SEXP; number: Integer; mpifmt: GCRY_MPI_FORMAT): GCRY_MPI; cdecl

///////////////////////////////////////////////////////////////////////////////
//
// symmetric cipher functions
//
///////////////////////////////////////////////////////////////////////////////
type
  GCRY_CIPHER_ALGOS = (
    GCRY_CIPHER_NONE	    = 0,
    GCRY_CIPHER_IDEA	    = 1,
    GCRY_CIPHER_3DES	    = 2,
    GCRY_CIPHER_CAST5	    = 3,
    GCRY_CIPHER_BLOWFISH    = 4,
    GCRY_CIPHER_SAFER_SK128 = 5,
    GCRY_CIPHER_DES_SK	    = 6,
    GCRY_CIPHER_AES         = 7,
    GCRY_CIPHER_AES192      = 8,
    GCRY_CIPHER_AES256      = 9,
    GCRY_CIPHER_TWOFISH     = 10,
    (* other cipher numbers are above 300 for OpenPGP reasons. *)
    (* GCRY_CIPHER_ARCFOUR     = 301, *)
    GCRY_CIPHER_AES128      = GCRY_CIPHER_AES,
    GCRY_CIPHER_RIJNDAEL    = GCRY_CIPHER_AES,
    GCRY_CIPHER_RIJNDAEL128 = GCRY_CIPHER_AES128,
    GCRY_CIPHER_RIJNDAEL192 = GCRY_CIPHER_AES192,
    GCRY_CIPHER_RIJNDAEL256 = GCRY_CIPHER_AES256
  );

  GCRY_CIPHER_MODES = (
    GCRY_CIPHER_MODE_NONE   = 0,  (* Not yet specified. *)
    GCRY_CIPHER_MODE_ECB    = 1,  (* Electronic codebook. *)
    GCRY_CIPHER_MODE_CFB    = 2,  (* Cipher feedback. *)
    GCRY_CIPHER_MODE_CBC    = 3,  (* Cipher block chaining. *)
    GCRY_CIPHER_MODE_STREAM = 4,  (* Used with stream ciphers. *)
    GCRY_CIPHER_MODE_OFB    = 5   (* Outer feedback. *)
  );

  GCRY_CIPHER_FLAGS = (
    GCRY_CIPHER_SECURE	    = 1,  (* Allocate in secure memory. *)
    GCRY_CIPHER_ENABLE_SYNC = 2   (* Enable CFB sync mode. *)
  );

  GCRY_CIPHER_HD = Pointer;

function gcry_cipher_open(algo: GCRY_CIPHER_ALGOS; mode: GCRY_CIPHER_MODES; flags: GCRY_CIPHER_FLAGS): GCRY_CIPHER_HD;
procedure gcry_cipher_close(h: GCRY_CIPHER_HD); cdecl;
function gcry_cipher_ctl(h: GCRY_CIPHER_HD; cmd: GCRY_CTL_CMDS; buffer: Pointer; buflen: Cardinal): Integer;
function gcry_cipher_algo_info(algo: GCRY_CIPHER_ALGOS; what: GCRY_CTL_CMDS; buffer: Pointer; nbytes: PCardinal): Integer;
function gcry_cipher_encrypt(h: GCRY_CIPHER_HD; o: PByte; outsize: Cardinal; i: PByte; inlen: Cardinal): Integer; cdecl;
function gcry_cipher_decrypt(h: GCRY_CIPHER_HD; o: PByte; outsize: Cardinal; i: PByte; inlen: Cardinal): Integer; cdecl;

function gcry_cipher_setkey(h: GCRY_CIPHER_HD; k: PByte; l: Cardinal): Integer;
function gcry_cipher_setiv(h: GCRY_CIPHER_HD; k: PByte; l: Cardinal): Integer;
function gcry_cipher_sync(h: GCRY_CIPHER_HD): Integer;

function gcry_cipher_get_algo_keylen(a: GCRY_CIPHER_ALGOS): Integer;
function gcry_cipher_get_algo_blklen(a: GCRY_CIPHER_ALGOS): Integer;


function _gcry_cipher_open(algo, mode: Integer; flags: Cardinal): GCRY_CIPHER_HD; cdecl;
function _gcry_cipher_ctl(h: GCRY_CIPHER_HD; cmd: Integer; buffer: Pointer; buflen: Cardinal): Integer; cdecl;
function _gcry_cipher_algo_info(algo, what: Integer; buffer: Pointer; nbytes: PCardinal): Integer; cdecl;

///////////////////////////////////////////////////////////////////////////////
//
// asymmetric cipher functions
//
///////////////////////////////////////////////////////////////////////////////
type
  GCRY_PK_ALGOS = (
    GCRY_PK_NONE  = 0,
    GCRY_PK_RSA   = 1,
    GCRY_PK_RSA_E = 2,	    (* deprecated *)
    GCRY_PK_RSA_S = 3,	    (* deprecated *)
    GCRY_PK_ELG_E = 16,     (* use only for OpenPGP *)
    GCRY_PK_DSA   = 17,
    GCRY_PK_ELG   = 20
  );

const
  GCRY_PK_USAGE_SIGN = 1;
  GCRY_PK_USAGE_ENCR = 2;

function gcry_pk_encrypt(result: PGCRY_SEXP; data, pkey: GCRY_SEXP): Integer; cdecl;
function gcry_pk_decrypt(result: PGCRY_SEXP; data, skey: GCRY_SEXP): Integer; cdecl;
function gcry_pk_sign(result: PGCRY_SEXP; data, skey: GCRY_SEXP): Integer; cdecl;
function gcry_pk_verify(sigval, data, pkey: GCRY_SEXP): Integer; cdecl;
function gcry_pk_genkey(r_key: PGCRY_SEXP; s_params: GCRY_SEXP): Integer; cdecl;
function gcry_pk_get_nbits(key: GCRY_SEXP): Cardinal; cdecl;
function gcry_pk_algo_info(algo: GCRY_PK_ALGOS; what: GCRY_CTL_CMDS; buffer: Pointer; nbytes: PCardinal): Integer;
function gcry_pk_algo_name(algo: GCRY_PK_ALGOS): PChar;

function _gcry_pk_algo_info(algo, what: Integer; buffer: Pointer; nbytes: PCardinal): Integer; cdecl;
function _gcry_pk_algo_name(algo: Integer): PChar; cdecl;

///////////////////////////////////////////////////////////////////////////////
//
// cryptograhic hash functions
//
///////////////////////////////////////////////////////////////////////////////
type
  GCRY_MD_ALGOS = (
    GCRY_MD_NONE    = 0,
    GCRY_MD_MD5     = 1,
    GCRY_MD_SHA1    = 2,
    GCRY_MD_RMD160  = 3,
    GCRY_MD_MD2     = 5,
    GCRY_MD_TIGER   = 6,   (* TIGER/192. *)
    GCRY_MD_HAVAL   = 7,   (* HAVAL, 5 pass, 160 bit. *)
    GCRY_MD_SHA256  = 8,
    GCRY_MD_SHA384  = 9,
    GCRY_MD_SHA512  = 10
  );

  (* Flags used with the open function. *)
  GCRY_MD_FLAGS = (
    GCRY_MD_FLAG_NONE   = 0,
    GCRY_MD_FLAG_SECURE = 1,  (* Allocate all buffers in "secure" memory *)
    GCRY_MD_FLAG_HMAC	= 2   (* Make an HMAC out of this algorithm. *)
  );

  GCRY_MD_HD = Pointer;

function gcry_md_open(algo: GCRY_MD_ALGOS; flags: GCRY_MD_FLAGS): GCRY_MD_HD;
procedure gcry_md_close(hd: GCRY_MD_HD); cdecl;
procedure gcry_md_reset(hd: GCRY_MD_HD); cdecl;
function gcry_md_ctl(hd: GCRY_MD_HD; cmd: GCRY_CTL_CMDS; buffer: PByte; buflen: Cardinal): Integer;
procedure gcry_md_write(hd: GCRY_MD_HD; const buffer: PByte; length: Cardinal); cdecl;
function gcry_md_read(hd: GCRY_MD_HD; algo: GCRY_MD_ALGOS): PByteArray;
function gcry_md_get_algo_dlen(algo: GCRY_MD_ALGOS): Cardinal;
function gcry_md_final(hd: GCRY_MD_HD): Integer;

function _gcry_md_open(algo: Integer; flags: Cardinal): GCRY_MD_HD; cdecl;
function _gcry_md_ctl(hd: GCRY_MD_HD; cmd: Integer; buffer: PByte; buflen: Cardinal): Integer; cdecl;
function _gcry_md_read(hd: GCRY_MD_HD; algo: Integer): PByteArray; cdecl;
function _gcry_md_get_algo_dlen(algo: Integer): Cardinal; cdecl;

///////////////////////////////////////////////////////////////////////////////
//
// random generating functions
//
///////////////////////////////////////////////////////////////////////////////
type
  GCRY_RANDOME_LEVEL = (
    GCRY_WEAK_RANDOM        = 0,
    GCRY_STRONG_RANDOM      = 1,
    GCRY_VERY_STRONG_RANDOM = 2
  );

procedure gcry_randomize(buffer: PByte; length: Cardinal; level: GCRY_RANDOME_LEVEL); cdecl;
function gcry_random_bytes_secure(nbytes: Cardinal; level: GCRY_RANDOME_LEVEL): Pointer; cdecl;

///////////////////////////////////////////////////////////////////////////////
//
// miscellaneous stuff
//
///////////////////////////////////////////////////////////////////////////////
procedure gcry_free(a: Pointer); cdecl;

implementation

constructor EGcrypt.Create(ErrCode: Integer);
begin
  inherited CreateFmt('GCrypt Error: %d', [ErrCode]);
end;

function gcry_cipher_setkey(h: GCRY_CIPHER_HD; k: PByte; l: Cardinal): Integer;
begin
  Result := gcry_cipher_ctl(h, GCRYCTL_SET_KEY, k, l);
end;

function gcry_cipher_setiv(h: GCRY_CIPHER_HD; k: PByte; l: Cardinal): Integer;
begin
  Result := gcry_cipher_ctl(h, GCRYCTL_SET_IV, k, l);
end;

function gcry_cipher_sync(h: GCRY_CIPHER_HD): Integer;
begin
  Result := gcry_cipher_ctl(h, GCRYCTL_CFB_SYNC, nil, 0);
end;

function gcry_cipher_get_algo_keylen(a: GCRY_CIPHER_ALGOS): Integer;
begin
  Result := gcry_cipher_algo_info(a, GCRYCTL_GET_KEYLEN, nil, nil);
end;

function gcry_cipher_get_algo_blklen(a: GCRY_CIPHER_ALGOS): Integer;
begin
  Result := gcry_cipher_algo_info(a, GCRYCTL_GET_BLKLEN, nil, nil);
end;

function gcry_cipher_open(algo: GCRY_CIPHER_ALGOS; mode: GCRY_CIPHER_MODES; flags: GCRY_CIPHER_FLAGS): GCRY_CIPHER_HD;
begin
  Result := _gcry_cipher_open(Ord(algo), Ord(mode), Ord(flags));
end;

function gcry_cipher_ctl(h: GCRY_CIPHER_HD; cmd: GCRY_CTL_CMDS; buffer: Pointer; buflen: Cardinal): Integer;
begin
  Result := _gcry_cipher_ctl(h, Ord(cmd), buffer, buflen);
end;

function gcry_cipher_algo_info(algo: GCRY_CIPHER_ALGOS; what: GCRY_CTL_CMDS; buffer: Pointer; nbytes: PCardinal): Integer;
begin
  Result := _gcry_cipher_algo_info(Ord(algo), Ord(what), buffer, nbytes); 
end;

function gcry_pk_algo_info(algo: GCRY_PK_ALGOS; what: GCRY_CTL_CMDS; buffer: Pointer; nbytes: PCardinal): Integer;
begin
  Result := _gcry_pk_algo_info(Ord(algo), Ord(what), buffer, nbytes);
end;

function gcry_pk_algo_name(algo: GCRY_PK_ALGOS): PChar;
begin
  Result := _gcry_pk_algo_name(Ord(algo));
end;

function gcry_md_open(algo: GCRY_MD_ALGOS; flags: GCRY_MD_FLAGS): GCRY_MD_HD;
begin
  Result := _gcry_md_open(Ord(algo), Ord(flags));
end;

function gcry_md_ctl(hd: GCRY_MD_HD; cmd: GCRY_CTL_CMDS; buffer: PByte; buflen: Cardinal): Integer;
begin
  Result := _gcry_md_ctl(hd, Ord(cmd), buffer, buflen)
end;

function gcry_md_read(hd: GCRY_MD_HD; algo: GCRY_MD_ALGOS): PByteArray;
begin
  Result := _gcry_md_read(hd, Ord(algo));
end;

function gcry_md_get_algo_dlen(algo: GCRY_MD_ALGOS): Cardinal;
begin
  Result := _gcry_md_get_algo_dlen(Ord(algo))
end;

function gcry_md_final(hd: GCRY_MD_HD): Integer;
begin
  Result := gcry_md_ctl(hd, GCRYCTL_FINALIZE, nil, 0)
end;

function gcry_control; external 'LIBGCRYPT.DLL';

function gcry_mpi_scan; external 'LIBGCRYPT.DLL';
function gcry_mpi_aprint; external 'LIBGCRYPT.DLL';
procedure gcry_mpi_release; external 'LIBGCRYPT.DLL';

function gcry_sexp_build; external 'LIBGCRYPT.DLL';
procedure gcry_sexp_release; external 'LIBGCRYPT.DLL';
procedure gcry_sexp_dump; external 'LIBGCRYPT.DLL';
function gcry_sexp_find_token; external 'LIBGCRYPT.DLL';
function gcry_sexp_cadr; external 'LIBGCRYPT.DLL';
function gcry_sexp_nth_mpi; external 'LIBGCRYPT.DLL';

function _gcry_cipher_open; external 'LIBGCRYPT.DLL' name 'gcry_cipher_open';
procedure gcry_cipher_close; external 'LIBGCRYPT.DLL';
function _gcry_cipher_ctl; external 'LIBGCRYPT.DLL' name 'gcry_cipher_ctl';
function _gcry_cipher_algo_info; external 'LIBGCRYPT.DLL' name 'gcry_cipher_algo_info';
function gcry_cipher_encrypt; external 'LIBGCRYPT.DLL';
function gcry_cipher_decrypt; external 'LIBGCRYPT.DLL';

function gcry_pk_encrypt; external 'LIBGCRYPT.DLL';
function gcry_pk_decrypt; external 'LIBGCRYPT.DLL';
function gcry_pk_sign; external 'LIBGCRYPT.DLL';
function gcry_pk_verify; external 'LIBGCRYPT.DLL';
function gcry_pk_genkey; external 'LIBGCRYPT.DLL';
function gcry_pk_get_nbits; external 'LIBGCRYPT.DLL';
function _gcry_pk_algo_info; external 'LIBGCRYPT.DLL' name 'gcry_pk_algo_info';
function _gcry_pk_algo_name; external 'LIBGCRYPT.DLL' name 'gcry_pk_algo_name';

function _gcry_md_open; external 'LIBGCRYPT.DLL' name 'gcry_md_open';
procedure gcry_md_close; external 'LIBGCRYPT.DLL';
procedure gcry_md_reset; external 'LIBGCRYPT.DLL';
function _gcry_md_ctl; external 'LIBGCRYPT.DLL' name 'gcry_md_ctl';
procedure gcry_md_write; external 'LIBGCRYPT.DLL';
function _gcry_md_read; external 'LIBGCRYPT.DLL' name 'gcry_md_read';
function _gcry_md_get_algo_dlen; external 'LIBGCRYPT.DLL' name 'gcry_md_get_algo_dlen';

procedure gcry_randomize; external 'LIBGCRYPT.DLL';
function gcry_random_bytes_secure; external 'LIBGCRYPT.DLL';

procedure gcry_free; external 'LIBGCRYPT.DLL';

end.
