package jp.cssj.sakae.opentype;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * TTCまたはTTFファイルへアクセスするためのクラスです。 このクラスはTTC/TTFファイルの判定を行います。
 * 
 * @author <a href="mailto:miyabe@info.port4">MIYABE Tatsuhiko</a>
 * @version $Id: TTFile.java,v 1.2 2007-05-06 04:27:03 miyabe Exp $
 */
public class FontFile {
	private static Map fileToRaf = new HashMap();
	private static Map fileToFontFile = new HashMap();

	private final List fonts;

	private final OpenTypeFont font;

	public static synchronized FontFile getFontFile(File file)
			throws IOException {
		file = file.getCanonicalFile();
		RandomAccessFile raf = (RandomAccessFile) fileToRaf.get(file);
		if (raf == null) {
			raf = new RandomAccessFile(file, "r");
			fileToRaf.put(file, raf);
		}
		FontFile ttFile = (FontFile) fileToFontFile.get(file);
		if (ttFile == null) {
			ttFile = new FontFile(raf);
			fileToFontFile.put(file, ttFile);
		}
		return ttFile;
	}

	/**
	 * 
	 * @param file
	 *            TTCまたはTTFファイル。
	 * @throws IOException
	 */
	private FontFile(RandomAccessFile raf) throws IOException {
		// TTCファイルの判定
		byte[] tagBytes = new byte[4];
		raf.read(tagBytes);
		String tag = new String(tagBytes, "ISO-8859-1");
		if ("ttcf".equals(tag)) {
			ArrayList fonts = new ArrayList();
			raf.skipBytes(4);

			int numFonts = raf.readInt();
			long[] dirOffsets = new long[numFonts];
			for (int i = 0; i < numFonts; i++) {
				dirOffsets[i] = raf.readInt();
			}
			for (int i = 0; i < numFonts; i++) {
				raf.seek(dirOffsets[i]);
				fonts.add(new OpenTypeFont(raf));
			}
			this.fonts = Collections.unmodifiableList(fonts);
			this.font = null;
		} else {
			raf.seek(0);
			this.fonts = null;
			this.font = new OpenTypeFont(raf);
		}
	}

	/**
	 * TTCファイルであればtrueを返します。
	 * 
	 * @return TTCファイルであればtrueです。
	 */
	public boolean isFontCollection() {
		return this.fonts != null;
	}

	/**
	 * TTCに含まれるフォントの数を返します。 TTFの場合は常に1を返します。
	 * 
	 * @return フォントの数です。
	 */
	public int getNumFonts() {
		if (this.fonts == null) {
			return 1;
		}
		return this.fonts.size();
	}

	/**
	 * TTCに含まれるフォントを返します。 TTFの場合、iは常に0でなければなりません。
	 * 
	 * @param i
	 * @return １つのフォントです。
	 */
	public OpenTypeFont getFont(int i) {
		if (this.fonts == null) {
			if (i != 0) {
				throw new ArrayIndexOutOfBoundsException(i);
			}
			return this.font;
		}
		return (OpenTypeFont) this.fonts.get(i);
	}

	/**
	 * 最初のフォントを返します。 TTFの場合は、唯一のフォントを返します。
	 * 
	 * @return １つのフォントです。
	 */
	public OpenTypeFont getFont() {
		if (this.fonts != null) {
			return this.getFont(0);
		}
		return this.font;
	}
}
