/* ------------------------------------------------------------------------- */
/*
 *  utility.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Tue 16 Jun 2009 13:35:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_UTILITY_H
#define CLX_UTILITY_H

#include <cmath>
#include <limits>
#include <istream>
#include "endian.h"
#include "memory.h"

#include "bitmask.h"
#include "combine.h"

/* ------------------------------------------------------------------------- */
/*
 *  STATIC_CHECK
 *
 *  This implementation is based on STATIC_CHECK macro described in
 *  A. Alexandrescu, ``Modern C++ Design''.
 */
/* ------------------------------------------------------------------------- */
template <bool> struct compile_time_checker;
template <> struct compile_time_checker<true> {
	void check_done() {}
};

#define STATIC_CHECK(expr, msg)					\
	if (false) {								\
		compile_time_checker<(expr) != 0>		\
		error_##msg;							\
		(error_##msg).check_done();				\
	}

namespace clx {
	/* --------------------------------------------------------------------- */
	//  fixed_cast
	/* --------------------------------------------------------------------- */
	template <int DecimalBits, class Type>
	inline double fixed_cast(Type x) {
		double dest = (x >> DecimalBits);
		
		Type mask = 1 << (DecimalBits - 1);
		double digit = 1 / 2.0;
		for (int i = 0; i < DecimalBits; ++i) {
			if (x & mask) dest += digit;
			mask >>= 1;
			digit /= 2.0;
		}
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  float_cast
	/* --------------------------------------------------------------------- */
	template <class Type>
	inline double float_cast(Type x) {
		if (x == 0) return 0.0;
		
		bool sign = (x & bitmask<31>::upper) ? true : false;
		Type src = x & bitmask<31>::lower;
		Type exp = (src & bitmask<23>::upper) >> 23;
		Type frac = src & bitmask<23>::lower;
		
		if (src == 0xff) {
			if (frac != 0) return std::numeric_limits<double>::quiet_NaN();
			else if (sign) return -std::numeric_limits<double>::infinity();
			return std::numeric_limits<double>::infinity();
		}
		
		double dest = 1.0 / std::pow(2, static_cast<double>(127 - exp)) * (1 + fixed_cast<23>(frac));
		
		return sign ? -dest : dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  read
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class Type, std::size_t N>
	inline bool read(std::basic_istream<Ch, Tr>& in, Type (&dest)[N]) {
		in.read(reinterpret_cast<Ch*>(dest), N * sizeof(Type) / sizeof(Ch));
		return !in.fail();
	}
	
	/* --------------------------------------------------------------------- */
	//  read
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class Container>
	inline bool read(std::basic_istream<Ch, Tr>& in, Container& dest) {
		while (1) {
			typename Container::value_type tmp[65536];
			if (!clx::read(in, tmp) || in.gcount() <= 0) break;
			dest.insert(dest.end(), tmp, tmp + in.gcount());
		}
		return !dest.empty();
	}
	
	/* --------------------------------------------------------------------- */
	//  read
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr>
	inline bool read(std::basic_istream<Ch, Tr>& in, std::basic_string<Ch, Tr>& dest) {
		while (1) {
			Ch tmp[65536];
			if (!clx::read(in, tmp) || in.gcount() <= 0) break;
			dest.insert(dest.size(), tmp, in.gcount());
		}
		return !dest.empty();
	}
	
	/* --------------------------------------------------------------------- */
	//  get
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class Type>
	inline bool get(std::basic_istream<Ch, Tr>& in, Type& dest, std::size_t which = endian::little) {
		in.read(reinterpret_cast<char*>(&dest), sizeof(dest));
		if (in.fail()) return false;
		if (which == endian::big) reverse(dest);
		return true;
	}
}

#endif // CLX_UTILITY_H
