/* ------------------------------------------------------------------------- */
/*
 *  combine.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Wed 23 Dec 2009 14:40:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_COMBINE_H
#define CLX_COMBINE_H

#include <climits>
#include "bitmask.h"
#include "range.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  combine
	/* --------------------------------------------------------------------- */
	template <class Container>
	inline std::size_t combine(const Container& x) {
		std::size_t dest = 0;
		
		const std::size_t bits = sizeof(typename range_value<const Container>::type) * CHAR_BIT;
		typename range_iterator<const Container>::type pos = begin(x);
		typename range_iterator<const Container>::type last = end(x);
		for (; pos != last; ++pos) {
			std::size_t tmp = static_cast<std::size_t>(*pos) & bitmask<bits>::lower;
			dest <<= bits;
			dest |= tmp;
		}
		
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  combine
	/* --------------------------------------------------------------------- */
	template <class T1, class T2>
	inline std::size_t combine(const T1& x1, const T2& x2) {
		std::size_t tmp = 0, dest = 0;
		
		tmp = static_cast<std::size_t>(x1) & bitmask<sizeof(T1) * CHAR_BIT>::lower;
		dest |= tmp << (sizeof(T2) * CHAR_BIT);
		tmp = static_cast<std::size_t>(x2) & bitmask<sizeof(T2) * CHAR_BIT>::lower;
		dest |= tmp;
		
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  combine
	/* --------------------------------------------------------------------- */
	template <class T1, class T2, class T3>
	inline std::size_t combine(const T1& x1, const T2& x2, const T3& x3) {
		std::size_t tmp = 0, dest = 0;
		
		tmp = combine(x1, x2);
		dest |= tmp << (sizeof(T3) * CHAR_BIT);
		tmp = static_cast<std::size_t>(x3) & bitmask<sizeof(T3) * CHAR_BIT>::lower;
		dest |= tmp;
		
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  combine
	/* --------------------------------------------------------------------- */
	template <class T1, class T2, class T3, class T4>
	inline std::size_t combine(const T1& x1, const T2& x2, const T3& x3, const T4& x4) {
		std::size_t tmp = 0, dest = 0;
		
		tmp = combine(x1, x2, x3);
		dest |= tmp << (sizeof(T4) * CHAR_BIT);
		tmp = static_cast<std::size_t>(x4) & bitmask<sizeof(T4) * CHAR_BIT>::lower;
		dest |= tmp;
		
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  combine2_
	/* --------------------------------------------------------------------- */
	template <std::size_t X1, unsigned char X2>
	struct combine2_ {
		static const std::size_t value = (X1 << CHAR_BIT) | X2;
	};
	
	/* --------------------------------------------------------------------- */
	//  combine3_
	/* --------------------------------------------------------------------- */
	template <std::size_t X1, unsigned char X2, unsigned char X3>
	struct combine3_ {
		static const std::size_t value = combine2_<combine2_<X1, X2>::value, X3>::value;
	};
	
	/* --------------------------------------------------------------------- */
	//  combine4_
	/* --------------------------------------------------------------------- */
	template <unsigned char X1, unsigned char X2, unsigned char X3, unsigned char X4>
	struct combine4_ {
		static const std::size_t value = combine2_<combine3_<X1, X2, X3>::value, X4>::value;
	};
}

#endif // CLX_COMBINE_H
