/* ------------------------------------------------------------------------- */
/*
 *  uri.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Mon 06 Apr 2009 18:43:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_URI_H
#define CLX_URI_H

#include <string>
#include <sstream>
#include <iterator>
#include <limits>
#include "literal.h"
#include "lexical_cast.h"
#include "predicate.h"

namespace clx {
	namespace detail {
		/* ----------------------------------------------------------------- */
		//  is_reserved
		/* ----------------------------------------------------------------- */
		template <class CharT>
		inline bool is_reserved(CharT c) {
			const CharT words[] = LITERAL("!#$&'()*+,-./:;=?@_~");
			if (is_alnum()(c) || is_any_of(words)(c)) return true;
			return false;
		}
	};
	
	namespace uri {
		/* ----------------------------------------------------------------- */
		//  encode
		/* ----------------------------------------------------------------- */
		template <class Ch, class Tr>
		inline std::basic_string<Ch, Tr> encode(const std::basic_string<Ch, Tr>& s) {
			static const int width = (sizeof(Ch) > 1) ? 4 : 2;
			static const int shift = ((sizeof(unsigned long) - sizeof(Ch)) * 8);
			static const unsigned long mask = std::numeric_limits<unsigned long>::max() >> shift;
			
			std::basic_ostringstream<Ch, Tr> ss;
			for (size_t i = 0; i < s.size(); ++i) {
				if (!clx::detail::is_reserved(s.at(i))) {
					ss << LITERAL('%');
					if (sizeof(Ch) > 1) ss << LITERAL('u');
					ss << std::hex << std::setw(width) << std::setfill(LITERAL('0'))
					   << static_cast<int>(s.at(i) & mask);
				}
				else ss << s.at(i);
			}
			return ss.str();
		}
		
		/* ----------------------------------------------------------------- */
		//  encode
		/* ----------------------------------------------------------------- */
		template <class CharT>
		inline std::basic_string<CharT> encode(const CharT* s) {
			std::basic_string<CharT> tmp(s);
			return encode(tmp);
		}
		
		/* ----------------------------------------------------------------- */
		//  decode
		/* ----------------------------------------------------------------- */
		template <class Ch, class Tr>
		inline std::basic_string<Ch, Tr> decode(const std::basic_string<Ch, Tr>& s) {
			std::basic_ostringstream<Ch, Tr> ss;
			
			size_t i = 0;
			while (i < s.size()) {
				if (s.at(i) == LITERAL('%')) {
					++i;
					size_t len = 2;
					if (s.at(i) == LITERAL('u')) {
						++i;
						len += 4;
					}
					if (i + len - 1 >= s.size()) break;
					int c = lexical_cast<int>(s.substr(i, len), std::ios::hex);
					ss << static_cast<Ch>(c);
					i += len;
				}
				else {
					ss << s.at(i);
					++i;
				}
			}
			
			return ss.str();
		}
		
		/* ----------------------------------------------------------------- */
		//  decode
		/* ----------------------------------------------------------------- */
		template <class CharT>
		inline std::basic_string<CharT> decode(const CharT* s) {
			std::basic_string<CharT> tmp(s);
			return decode(tmp);
		}
	};
}

#endif
