/* ------------------------------------------------------------------------- */
/*
 *  uri.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Mon 06 Apr 2009 18:43:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_URI_H
#define CLX_URI_H

#include <string>
#include <sstream>
#include <iterator>
#include <limits>
#include "lexical_cast.h"
#include "predicate.h"

namespace clx {
	namespace detail {
		/* ----------------------------------------------------------------- */
		//  is_reserved
		/* ----------------------------------------------------------------- */
		inline bool is_reserved(char c) {
			if (is_alnum()(c) || is_any_of("!#$&'()*+,-./:;=?@_~")(c)) return true;
			return false;
		}
	};
	
	namespace uri {
		/* ----------------------------------------------------------------- */
		//  encode
		/* ----------------------------------------------------------------- */
		inline std::basic_string<char> encode(const std::basic_string<char>& s) {
			std::basic_ostringstream<char> ss;
			for (unsigned int i = 0; i < s.size(); ++i) {
				if (!clx::detail::is_reserved(s.at(i))) {
					ss << '%' << std::hex << std::setw(2) << std::setfill('0')
					   << static_cast<int>(s.at(i) & 0xff);
				}
				else ss << s.at(i);
			}
			return ss.str();
		}
		
		/* ----------------------------------------------------------------- */
		//  encode
		/* ----------------------------------------------------------------- */
		inline std::basic_string<char> encode(const char* s) {
			std::basic_string<char> tmp(s);
			return encode(tmp);
		}
		
		/* ----------------------------------------------------------------- */
		//  decode
		/* ----------------------------------------------------------------- */
		inline std::basic_string<char> decode(const std::basic_string<char>& s) {
			std::basic_ostringstream<char> ss;
			
			size_t i = 0;
			while (i < s.size()) {
				if (s.at(i) == '%') {
					++i;
					if (i + 1 >= s.size()) break;
					int c = lexical_cast<int>(s.substr(i, 2), std::ios::hex);
					ss << static_cast<char>(c);
					i += 2;
				}
				else {
					ss << s.at(i);
					++i;
				}
			}
			
			return ss.str();
		}
		
		/* ----------------------------------------------------------------- */
		//  decode
		/* ----------------------------------------------------------------- */
		inline std::basic_string<char> decode(const char* s) {
			std::basic_string<char> tmp(s);
			return decode(tmp);
		}
	};
}

#endif
