/* ------------------------------------------------------------------------- */
/*
 *  time_duration.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Thu 14 Jun 2007 14:38:02 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_TIME_DURATION_H
#define CLX_TIME_DURATION_H

namespace clx {
	typedef long time_value;
	
	namespace time_unit {
		enum {
			minute = 60,			// 60 sec
			hour   = 3600,			// 60 * 60 sec
			day    = 86400,			// 60 * 60 * 24 sec
			week   = 604800			// 60 * 60 * 24 * 7 sec
		};
		
		enum {
			millisec = 1000,		// 1 / (10^3) sec
			microsec = 1000000,		// 1 / (10^6) sec
			nanosec  = 1000000000	// 1 / (10^9) sec
		};
	};
	
	/* --------------------------------------------------------------------- */
	//  date_duration
	/* --------------------------------------------------------------------- */
	class date_duration {
	public:
		typedef long value_type;
		
		explicit date_duration(value_type y = 0, value_type m = 0, value_type d = 0) :
			years_(y), months_(m), days_(d) {}
		
		date_duration& operator+=(const date_duration& t) {
			years_ += t.years_;
			months_ += t.months_;
			days_ += t.days_;
			return *this;
		}
		
		date_duration& operator-=(const date_duration& t) {
			years_ -= t.years_;
			months_ -= t.months_;
			days_ -= t.days_;
			return *this;
		}
		
		// binary operator
		friend date_duration operator+(date_duration x, const date_duration& y) { return x += y; }
		friend date_duration operator-(date_duration x, const date_duration& y) { return x -= y; }
		friend date_duration operator+(const date_duration& y, date_duration x) { return x += y; }
		friend date_duration operator-(const date_duration& y, date_duration x) { return x -= y; }
		
		// accessor
		value_type years() const { return years_; }
		value_type months() const { return months_; }
		value_type days() const { return days_; }
		
	private:
		value_type years_;
		value_type months_;
		value_type days_;
	};
	
	/* --------------------------------------------------------------------- */
	//  helper_functions
	/* --------------------------------------------------------------------- */
	date_duration years(time_value t) { return date_duration(t); }
	date_duration months(time_value t) { return date_duration(0, t); }
	
	time_value days(time_value t) { return t * time_unit::day; }
	time_value weeks(time_value t) { return t * time_unit::week; }
	time_value hours(time_value t) { return t * time_unit::hour; }
	time_value minutes(time_value t) { return t * time_unit::minute; }
	time_value seconds(time_value t) { return t; }
	
	double millisec(time_value t) { return t / static_cast<double>(time_unit::millisec); }
	double mircosec(time_value t) { return t / static_cast<double>(time_unit::microsec); }
	double nanosec(time_value t) { return t / static_cast<double>(time_unit::nanosec); }
}

#endif // CLX_TIME_DURATION_H
