/* ------------------------------------------------------------------------- */
/*
 *  win32/mutex.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Tue 12 Aug 2008 14:01:23 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_WIN32_MUTEX_H
#define CLX_WIN32_MUTEX_H

#include <windows.h>
#include "thread_error.h"

namespace clx {
	template <class Mutex> class unique_lock;
	
	/* --------------------------------------------------------------------- */
	//  mutex
	/* --------------------------------------------------------------------- */
	class mutex {
	public:
		typedef unique_lock<mutex> scoped_lock;
		typedef HANDLE handle_pointer;
		
		mutex() {
			m_ = CreateMutex(NULL, false, NULL);
			if (m_ == NULL) throw thread_error("CreateMutex");
		}
		
		virtual ~mutex() { CloseHandle(m_); }
		
		void lock() { WaitForSingleObject(m_, INFINITE); }
		void unlock() { ReleaseMutex(m_); }
		handle_pointer native_handle() { return m_; }
		
	private:
		handle_pointer m_;
		
		// non-copyable
		mutex(const mutex& cp);
		mutex& operator=(const mutex& cp);
	};
	
	/* --------------------------------------------------------------------- */
	//  recursive_mutex
	/* --------------------------------------------------------------------- */
	class recursive_mutex {
	public:
		typedef unique_lock<recursive_mutex> scoped_lock;
		typedef HANDLE handle_pointer;
		
		recursive_mutex() : id_(), count_(0) {
			m_ = CreateMutex(NULL, false, NULL);
			if (m_ == NULL) throw thread_error("CreateMutex");
		}
		
		virtual ~recursive_mutex() { CloseHandle(m_); }
		
		void lock() {
			if (count_ > 0 && id_ == GetCurrentThreadId()) {
				count_++;
				return;
			}
			WaitForSingleObject(m_, INFINITE);
			id_ = GetCurrentThreadId();
			count_ = 1;
		}
		
		void unlock() {
			if (count_ > 1 && id_ == GetCurrentThreadId()) {
				count_--;
				return;
			}
			ReleaseMutex(m_);
			count_ = 0;
		}
		
		handle_pointer native_handle() { return m_; }
		
	private:
		typedef DWORD thread_id_type;
		typedef size_t size_type;
		
		handle_pointer m_;
		thread_id_type id_;
		size_type count_;
	};
}

#endif // CLX_WIN32_MUTEX_H
