/* ------------------------------------------------------------------------- */
/*
 *  format.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Sat 10 Mar 2007 20:58:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_FORMAT_H
#define CLX_FORMAT_H

#include <cstdlib>
#include <cctype>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <ostream>
#include <string>
#include "lexical_cast.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  basic_format
	/* --------------------------------------------------------------------- */
	template <
		class CharT,
		class Traits = std::char_traits<CharT>
	>
	class basic_format {
	public:
		typedef CharT char_type;
		typedef unsigned int size_type;
		typedef typename std::basic_string<CharT, Traits> string_type;
		
		// constructor and destructor
		basic_format() : str_(), fmt_() {}
		
		explicit basic_format(const char_type* fmt) :
			str_(), fmt_(fmt) { this->reset(); }
		
		explicit basic_format(const string_type& fmt) :
			str_(), fmt_(fmt) { this->reset(); }
		
		// operator
		template <class ValueT>
		basic_format& operator%(const ValueT& x) {
			if (*pos_ == rep && pos_ + 1 != fmt_.end() && *(pos_ + 1) != rep) {
				string_type flag;
				while (++pos_ != fmt_.end() && types_.find(*pos_) == string_type::npos) flag += *pos_;
				if (pos_ == fmt_.end()) return *this; // uncorrect format
				str_ += this->put(x, *pos_++, flag);
			}
			while (pos_ != fmt_.end() && *pos_ != rep) str_ += *pos_++;
			return *this;
		}
		
		template <class Ch, class Tr>
		friend std::basic_ostream<Ch, Tr>& operator<<(
			std::basic_ostream<Ch, Tr>& sout, const basic_format<Ch, Tr>& f) {
			sout << f.str_;
			return sout;
		}
		
	private:
		typedef typename string_type::const_iterator const_iterator;
		typedef typename std::basic_stringstream<CharT, Traits> internal_stream;
		
		static const char_type rep = '%';
		static const char_type lpos = '-';
		static const char_type ws = ' ';
		static const char_type sign = '+';
		static const char_type zero = '0';
		static const char_type prefix = '#';
		static const char_type dot = '.';
		
		string_type types_;
		string_type str_;
		string_type fmt_;
		const_iterator pos_;
		
		void reset() {
			str_.clear();
			pos_ = fmt_.begin();
			types_ = "diouxXeEfFgGcs";
			while (pos_ != fmt_.end() && *pos_ != rep) str_ += *pos_++;
		}
		
		template <class ValueT>
		string_type put(const ValueT& x, char_type c, const string_type& flag) {
			internal_stream ss;
			
			this->setopt(ss, flag);
			this->settype(ss, c);
			ss << x;
			
			return ss.str();
		}
		
		void setopt(internal_stream& ss, const string_type& flag) {
			bool lp = false;
			for (const_iterator pos = flag.begin(); pos != flag.end(); pos++) {
				switch (*pos) {
				case lpos:
					lp = true;
					ss << std::setfill(ws);
					ss << std::left;
					break;
				case prefix:
					ss << std::showbase;
					break;
				case sign:
					ss << std::showpos;
					break;
				case ws:
					ss << std::setfill(*pos);
					break;
				case zero:
					if (!lp) ss << std::setfill(*pos);
					break;
				case dot:
					this->setprecision(ss, ++pos, flag.end());
					if (pos == flag.end()) return;
					break;
				default:
					if (isdigit(*pos)) this->setwidth(ss, pos, flag.end());
					if (pos == flag.end()) return;
					break; /* ignore unknown flags */
				}
			}
		}
		
		void setwidth(internal_stream& ss, const_iterator& cur, const_iterator last) {
			std::string digit;
			for (; cur != last && isdigit(*cur); cur++) digit += *cur;
			if (!digit.empty()) ss << std::setw(lexical_cast<int>(digit));
			if (cur != last && *cur == dot) this->setprecision(ss, ++cur, last);
		}
		
		void setprecision(internal_stream& ss, const_iterator& cur, const_iterator last) {
			std::string digit;
			for (; cur != last && isdigit(*cur); cur++) digit += *cur;
			if (!digit.empty()) ss << std::setprecision(lexical_cast<int>(digit));
		}
		
		void settype(internal_stream& ss, char_type c) {
			char_type ctype;
			if (isupper(c)) ss << std::uppercase;
			ctype = tolower(c);
			
			switch (ctype) {
			case 'o':
				ss << std::oct;
				break;
			case 'x':
				ss << std::hex;
				break;
			case 'e':
				ss << std::scientific;
				break;
			case 'f':
				ss << std::setiosflags(std::ios::fixed);
				break;
			default:
				break;
			}
		}
	};
	
	typedef basic_format<char> format;
}
#endif // CLX_FORMAT_H
