/*
    ChibiOS - Copyright (C) 2006..2018 Giovanni Di Sirio

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

/**
 * @file    SAMA5D2x/hal_st_lld.c
 * @brief   ST Driver subsystem low level driver code.
 *
 * @addtogroup ST
 * @{
 */

#include "hal.h"

/*===========================================================================*/
/* Driver local definitions.                                                 */
/*===========================================================================*/

/**
 * @brief   Periodic Interrupt Timer frequency.
 */
#define SAMA_PIT  (SAMA_MCK / 16 / SAMA_H64MX_H32MX_RATIO)

#if (SAMA_ST_USE_TC0 == TRUE) || (SAMA_ST_USE_TC1 == TRUE)
/**
 * @brief   Enable write protection on TC registers block.
 *
 * @param[in] tc    pointer to a TC
 *
 * @notapi
 */
#define tcEnableWP(tc) {                                                     \
  tc->TC_WPMR = TC_WPMR_WPKEY_PASSWD | TC_WPMR_WPEN;                         \
}

/**
 * @brief   Disable write protection on TC registers block.
 *
 * @param[in] tc    pointer to a TC
 *
 * @notapi
 */
#define tcDisableWP(tc) {                                                     \
  tc->TC_WPMR = TC_WPMR_WPKEY_PASSWD;                                         \
}
#endif

#if OSAL_ST_MODE == OSAL_ST_MODE_FREERUNNING

#if SAMA_ST_USE_PIT
#error "PIT timer doesn't support tick-less mode"
#endif

#if SAMA_ST_USE_TC0
#if ((SAMA_TC0CLK) / (OSAL_ST_FREQUENCY) != 32)
#error "Bad OSAL_ST_FREQUENCY value in configuration. It must be set to TC0_periph_clk / 32"
#endif
#endif

#if SAMA_ST_USE_TC1
#if (((SAMA_TC1CLK) / (OSAL_ST_FREQUENCY) != 32) || ((SAMA_TC1CLK) % (OSAL_ST_FREQUENCY)) != 0)
#error "Bad OSAL_ST_FREQUENCY value in configuration. It must be set to TC1_periph_clk / 32"
#endif
#endif

#endif

/*===========================================================================*/
/* Driver exported variables.                                                */
/*===========================================================================*/

/*===========================================================================*/
/* Driver local types.                                                       */
/*===========================================================================*/

/*===========================================================================*/
/* Driver local variables and types.                                         */
/*===========================================================================*/

#if SAMA_ST_USE_TC0 || SAMA_ST_USE_TC1
static Tc *tcp;
#endif

/*===========================================================================*/
/* Driver local functions.                                                   */
/*===========================================================================*/

/*===========================================================================*/
/* Driver interrupt handlers.                                                */
/*===========================================================================*/

#if (SAMA_ST_USE_TC0 || SAMA_ST_USE_TC1) || defined(__DOXYGEN__)
/**
 * @brief   System Timer vector.
 * @details This interrupt is used both in periodic or free running
 *          mode, generated by TCx timer.
 *
 * @isr
 */
OSAL_IRQ_HANDLER(SAMA_ST_TC_HANDLER) {

  OSAL_IRQ_PROLOGUE();
  (void)tcp->TC_CHANNEL[0].TC_SR; /* acknowledge TC interrupt */
  osalSysLockFromISR();
  osalOsTimerHandlerI();
  osalSysUnlockFromISR();
  aicAckInt();
  OSAL_IRQ_EPILOGUE();
}
#endif

#if (SAMA_ST_USE_PIT)  || defined(__DOXYGEN__)
/**
 * @brief   System Timer vector.
 * @details This interrupt is used for system tick in periodic mode.
 *
 * @isr
 */
OSAL_IRQ_HANDLER(PIT_Handler) {
  uint32_t ivr;
  OSAL_IRQ_PROLOGUE();

  osalSysLockFromISR();
  ivr = PIT->PIT_PIVR;    /* acknowledge PIT interrupt */
  osalDbgAssert((ivr & PIT_PIVR_PICNT_Msk) == (1 << PIT_PIVR_PICNT_Pos),
      "check for lost tick");
  osalOsTimerHandlerI();
  osalSysUnlockFromISR();
  aicAckInt();
  OSAL_IRQ_EPILOGUE();
}
#endif /* SAMA_ST_USE_PIT == TRUE */

/*===========================================================================*/
/* Driver exported functions.                                                */
/*===========================================================================*/

/**
 * @brief   Low level ST driver initialization.
 *
 * @notapi
 */
void st_lld_init(void) {

#if SAMA_ST_USE_TC0 || SAMA_ST_USE_TC1

#if SAMA_ST_USE_TC0
  tcp = TC0;
  uint32_t rc = (SAMA_TC0CLK) / (OSAL_ST_FREQUENCY);

#if SAMA_HAL_IS_SECURE
  mtxConfigPeriphSecurity(MATRIX1, ID_TC0, SECURE_PER);
#endif /* SAMA_HAL_IS_SECURE */

  pmcEnableTC0();
  aicSetSourcePriority(ID_TC0, SAMA_TC0_IRQ_PRIORITY);
  aicSetSourceHandler(ID_TC0, SAMA_ST_TC_HANDLER);
  aicEnableInt(ID_TC0);
#endif

#if SAMA_ST_USE_TC1
  tcp = TC1;
  uint32_t rc = (SAMA_TC1CLK) / (OSAL_ST_FREQUENCY);

#if SAMA_HAL_IS_SECURE
  mtxConfigPeriphSecurity(MATRIX1, ID_TC1, SECURE_PER);
#endif /* SAMA_HAL_IS_SECURE */

  pmcEnableTC1();
  aicSetSourcePriority(ID_TC1, SAMA_TC1_IRQ_PRIORITY);
  aicSetSourceHandler(ID_TC1, SAMA_ST_TC_HANDLER);
  aicEnableInt(ID_TC1);
#endif

  tcDisableWP(tcp);
#if OSAL_ST_MODE == OSAL_ST_MODE_FREERUNNING

  /* Initializing the timer counter in free running mode.
   * The clock source is the bus clock divided by 32.*/
  (void)rc;
  tcp->TC_CHANNEL[0].TC_EMR = 0;
  tcp->TC_CHANNEL[0].TC_CMR = TC_CMR_WAVE | TC_CMR_WAVSEL_UP |
                              TC_CMR_TCCLKS(TC_CMR_TCCLKS_TIMER_CLOCK3);
  tcp->TC_CHANNEL[0].TC_RC  = 0;
  tcp->TC_CHANNEL[0].TC_CCR = TC_CCR_CLKEN | TC_CCR_SWTRG;
  tcp->TC_CHANNEL[0].TC_IDR = 0xFFFFFFFF;         /* Disable IRQs.                */
  tcp->TC_CHANNEL[0].TC_SR;                       /* Clear pending IRQs.          */
#endif

#if OSAL_ST_MODE == OSAL_ST_MODE_PERIODIC
  tcp->TC_CHANNEL[0].TC_EMR = TC_EMR_NODIVCLK;
  tcp->TC_CHANNEL[0].TC_CMR = TC_CMR_WAVE | TC_CMR_WAVSEL_UP_RC;
  tcp->TC_CHANNEL[0].TC_RC  = TC_RC_RC(rc);
  tcp->TC_CHANNEL[0].TC_CCR = TC_CCR_CLKEN | TC_CCR_SWTRG;;
  tcp->TC_CHANNEL[0].TC_SR;                       /* Clear pending IRQs.          */
  tcp->TC_CHANNEL[0].TC_IER = TC_IER_CPCS;
#endif

  tcEnableWP(tcp);
#endif /* SAMA_ST_USE_TC0 || SAMA_ST_USE_TC1 */

#if (SAMA_ST_USE_PIT == TRUE)
#if SAMA_HAL_IS_SECURE
  mtxConfigPeriphSecurity(MATRIX1, ID_PIT, SECURE_PER);
#endif /* SAMA_HAL_IS_SECURE */

  /* Enable PIT.*/
  pmcEnablePIT();
  PIT->PIT_MR = PIT_MR_PIV((SAMA_PIT / OSAL_ST_FREQUENCY) - 1);
  PIT->PIT_MR |= PIT_MR_PITEN | PIT_MR_PITIEN;
  (void) PIT->PIT_PIVR;    /* reset PIT PICNT counter */

  /* Enable IRQ.*/
  aicSetSourcePriority(ID_PIT, SAMA_ST_IRQ_PRIORITY);
  aicSetSourceHandler(ID_PIT, PIT_Handler);
  aicEnableInt(ID_PIT);
#endif /* SAMA_ST_USE_PIT */

}

/** @} */
