module CGIKit

  # Utilitis is a module wihch collects utility methods based on cgi.rb.
  module Utilities
    CR            = "\015"
    LF            = "\012"
    EOL           = CR + LF
    RFC822_DAYS   = %w[ Sun Mon Tue Wed Thu Fri Sat ]
    RFC822_MONTHS = %w[ Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ]
    MEM_CONTENT_LENGTH = 10240

    def query_from_headers( headers, input )
      if ("POST" == headers['REQUEST_METHOD']) and
          %r|\Amultipart/form-data.*boundary=\"?([^\";,]+)\"?|n.match(headers['CONTENT_TYPE'])
        boundary = $1.dup
        query_from_multipart(headers, input,
                             boundary, Integer(headers['CONTENT_LENGTH']))
      else
        case headers['REQUEST_METHOD']
        when "HEAD" then query = query_string_from_head(headers)
        when "GET"  then query = query_string_from_get(headers)
        when "POST" then query = query_string_from_post(headers, input)
        else             query = query_string_from_shell end
        Request.parse_query_string(query)
      end
    end

    def query_string_from_get( headers )
      headers['QUERY_STRING'] or ""
    end

    alias :query_string_from_head :query_string_from_get

    def query_string_from_post( headers, input )
      input.binmode
      input.read(Integer(headers['CONTENT_LENGTH'])) or ''
    end

    def query_string_from_shell
      require "shellwords"
      msg = %|(offline mode: enter name=value pairs on standard input)\n|
        string = unless ARGV.empty?
                   ARGV.join(' ')
                 else
                   if STDIN.tty?
                     STDERR.print(msg)
                   end
                   readlines.join(' ').gsub(/\n/n, '')
                 end.gsub(/\\=/n, '%3D').gsub(/\\&/n, '%26')
      words = Shellwords.shellwords(string)

      if words.find{|x| /=/n.match(x) }
        words.join('&')
      else
        words.join('+')
      end
    end

    def query_from_multipart( headers, input, boundary, content_length )
      params   = Hash.new([])
      boundary = "--" + boundary
      buf      = ""
      bufsize  = 10 * 1024

      # start multipart/form-data
      input.binmode
      boundary_size   = boundary.size + EOL.size
      content_length -= boundary_size
      status          = input.read boundary_size
      if status == nil then
        raise EOFError, "no content body"
      elsif boundary + EOL != status
        raise EOFError, "bad content body"
      end

      until content_length == -1
        head = nil
        tmp = nil
        if content_length > MEM_CONTENT_LENGTH
          require "tempfile"
          tmp = Tempfile.new("CGIKit")
          tmp.binmode
          data = TempfileByteData.new(tmp)
        else
          data = ByteData.new
        end

        until head and /#{boundary}(?:#{EOL}|--)/n.match(buf)
            if (not head) and /#{EOL}#{EOL}/n.match(buf)
                buf = buf.sub(/\A((?:.|\n)*?#{EOL})#{EOL}/n) do
              head = $1.dup
              ""
            end
              next
            end

          if head and ( (EOL + boundary + EOL).size < buf.size )
            data << buf[0 ... (buf.size - (EOL + boundary + EOL).size)]
            buf[0 ... (buf.size - (EOL + boundary + EOL).size)] = ""
          end

          if bufsize < content_length then
            c = input.read(bufsize) or ''
          else
            c = input.read(content_length) or ''
          end
          buf += c
          content_length -= c.size
        end

        buf = buf.sub(/\A((?:.|\n)*?)(?:#{EOL})?#{boundary}(#{EOL}|--)/n) do
          data << $1
          if "--" == $2
            content_length = -1
          end
          ""
        end

        /Content-Disposition:.* filename="?([^\";]*)"?/ni.match(head)
        filename = ($1 || "").dup
        if /Mac/ni.match(headers['HTTP_USER_AGENT']) and
            /Mozilla/ni.match(headers['HTTP_USER_AGENT']) and
            (not /MSIE/ni.match(headers['HTTP_USER_AGENT']))
          filename = Utilities.unescape_url filename
        end
        data.path = filename

        /Content-Type: ([^\r\n]*)/ni.match(head)
        if $1 then
          data.content_type = $1.dup
        else
          data = data.to_s
        end

        /Content-Disposition:.* name="?([^\";]*)"?/ni.match(head)
        name = $1.dup

        if params.has_key? name then
          params[name].push data
        else
          params[name] = [data]
        end
      end

      params
    end

    # Returns an encoded string for URL.
    def escape_url( string )
      string.gsub( /([^ a-zA-Z0-9_.-]+)/n ) do
        '%' + $1.unpack( 'H2' * $1.size ).join( '%' ).upcase
      end.tr( ' ', '+' )
    end

    # Returns a string decoded from URL.
    def unescape_url( string )
      string.tr( '+', ' ' ).gsub( /((?:%[0-9a-fA-F]{2})+)/n ) do
        [ $1.delete( '%' ) ].pack( 'H*' )
      end
    end

    # Escapes HTML control characters.
    def escape_html( string )
      string.gsub(/&/n, '&amp;').
        gsub(/\"/n, '&quot;').
        gsub(/>/n, '&gt;').
        gsub(/</n, '&lt;').
        gsub(/'/n, '&#39;')
    end

    # Unescapes HTML control characters.
    def unescape_html( string )
      string.gsub(/&(.*?);/n) do
        match = $1.dup
        case match
        when /\Aamp\z/ni           then '&'
        when /\Aquot\z/ni          then '"'
        when /\Agt\z/ni            then '>'
        when /\Alt\z/ni            then '<'
        when /\A#0*(\d+)\z/n       then
          if Integer($1) < 256
            Integer($1).chr
          else
            if Integer($1) < 65536 and \
              ($KCODE[0] == ?u or $KCODE[0] == ?U)
              [Integer($1)].pack("U")
            else
              "&##{$1};"
            end
          end
        when /\A#x([0-9a-f]+)\z/ni then
          if $1.hex < 256
            $1.hex.chr
          else
            if $1.hex < 65536 and \
              ($KCODE[0] == ?u or $KCODE[0] == ?U)
              [$1.hex].pack("U")
            else
              "&#x#{$1};"
            end
          end
        else
          "&#{match};"
        end
      end
    end

    # Formats Time object in RFC1123.
    # For example, "Sat, 1 Jan 2000 00:00:00 GMT".
    def date( time )
      t = time.clone.gmtime
      return format("%s, %.2d %s %.4d %.2d:%.2d:%.2d GMT",
                    RFC822_DAYS[t.wday], t.day, RFC822_MONTHS[t.month-1], t.year,
                    t.hour, t.min, t.sec)
    end

    def class_named_from( name, spaces = [CGIKit, Object] )
      spaces.each do |klass|
        base = klass
        name.split("::").each do |i|
          if i.empty? then
            i = :Object
          end
          if base.const_defined?(i) then
            base = base.const_get(i)
          else
            base = nil
            break
          end
        end
        return base if base
      end
      raise "No such '#{name}' class from #{spaces.inspect}."
    end

    module_function :query_from_headers
    module_function :query_string_from_get
    module_function :query_string_from_post
    module_function :query_string_from_head
    module_function :query_string_from_shell
    module_function :query_from_multipart
    module_function :escape_url
    module_function :unescape_url
    module_function :escape_html
    module_function :unescape_html
    module_function :date
    module_function :class_named_from

  end


  # FileLock is for locking files.
  class FileLock
    # Creates a shared file lock on a file.
    def self.shared_lock( filename, mode = 'r' )
      File.open( filename, mode ) do | io |
        io.flock File::LOCK_SH
        yield io
        io.flock File::LOCK_UN
      end
    end

    # Creates a exclusive file lock on a file.
    def self.exclusive_lock( filename, mode = 'w' )
      File.open( filename, mode ) do | io |
        io.flock File::LOCK_EX
        yield io
        io.flock File::LOCK_UN
      end
    end
  end


  class DummyLogger
    def fatal( msg = nil, &block ); end
    def warn( msg = nil, &block ); end
    def error( msg = nil, &block ); end
    def info( msg = nil, &block ); end
    def debug( msg = nil, &block ); end
    def method_missing( *args ); p args;end
  end


  module Logging

    def logger
      @application.logger
    end

    def fatal( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def error( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def warn( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def info( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def debug( msg = nil, method = false, &block )
      _log(msg, method, :debug, &block)
    end

    def _log( msg = nil, method = false, level = :info, &block )
      msg = _log_message(msg) if method or msg.nil?
      msg = "<Thread:#{Thread.current.object_id}> #{msg}"
      if block_given? then
        logger().__send__(level, msg, block)
      else
        logger().__send__(level, msg)
      end
    end

    private

    def _log_message( msg )
      method = _method(caller[2])
      log = "#{self.class}##{method}"
      msg ? "#{log}: #{msg}" : log
    end

    def _method( at )
      if /^(.+?):(\d+)(?::in `(.*)')?/ =~ at then
        $3
      end
    end

  end

end
