module CGIKit

  # ResourceManager class manages resources of an application.
  class ResourceManager
    include Logging

    MIME = {
      'ez'      => 'application/andrew-inset',
      'hqx'     => 'application/mac-binhex40',
      'cpt'     => 'application/mac-compactpro',
      'doc'     => 'application/msword',
      'bin'     => 'application/octet-stream',
      'dms'     => 'application/octet-stream',
      'lha'     => 'application/octet-stream',
      'lzh'     => 'application/octet-stream',
      'exe'     => 'application/octet-stream',
      'class'   => 'application/octet-stream',
      'so'      => 'application/octet-stream',
      'dll'     => 'application/octet-stream',
      'oda'     => 'application/oda',
      'pdf'     => 'application/pdf',
      'ai'      => 'application/postscript',
      'eps'     => 'application/postscript',
      'ps'      => 'application/postscript',
      'smi'     => 'application/smil',
      'smil'    => 'application/smil',
      'mif'     => 'application/vnd.mif',
      'xls'     => 'application/vnd.ms-excel',
      'ppt'     => 'application/vnd.ms-powerpoint',
      'wbxml'   => 'application/vnd.wap.wbxml',
      'wmlc'    => 'application/vnd.wap.wmlc',
      'wmlsc'   => 'application/vnd.wap.wmlscriptc',
      'bcpio'   => 'application/x-bcpio',
      'vcd'     => 'application/x-cdlink',
      'pgn'     => 'application/x-chess-pgn',
      'cpio'    => 'application/x-cpio',
      'csh'     => 'application/x-csh',
      'dcr'     => 'application/x-director',
      'dir'     => 'application/x-director',
      'dxr'     => 'application/x-director',
      'dvi'     => 'application/x-dvi',
      'spl'     => 'application/x-futuresplash',
      'gtar'    => 'application/x-gtar',
      'hdf'     => 'application/x-hdf',
      'js'      => 'application/x-javascript',
      'skp'     => 'application/x-koan',
      'skd'     => 'application/x-koan',
      'skt'     => 'application/x-koan',
      'skm'     => 'application/x-koan',
      'latex'   => 'application/x-latex',
      'nc'      => 'application/x-netcdf',
      'cdf'     => 'application/x-netcdf',
      'sh'      => 'application/x-sh',
      'shar'    => 'application/x-shar',
      'swf'     => 'application/x-shockwave-flash',
      'sit'     => 'application/x-stuffit',
      'sv4cpio' => 'application/x-sv4cpio',
      'sv4crc'  => 'application/x-sv4crc',
      'tar'     => 'application/x-tar',
      'tcl'     => 'application/x-tcl',
      'tex'     => 'application/x-tex',
      'texinfo' => 'application/x-texinfo',
      'texi'    => 'application/x-texinfo',
      't'       => 'application/x-troff',
      'tr'      => 'application/x-troff',
      'roff'    => 'application/x-troff',
      'man'     => 'application/x-troff-man',
      'me'      => 'application/x-troff-me',
      'ms'      => 'application/x-troff-ms',
      'ustar'   => 'application/x-ustar',
      'src'     => 'application/x-wais-source',
      'xhtml'   => 'application/xhtml+xml',
      'xht'     => 'application/xhtml+xml',
      'zip'     => 'application/zip',
      'au'      => 'audio/basic',
      'snd'     => 'audio/basic',
      'mid'     => 'audio/midi',
      'midi'    => 'audio/midi',
      'kar'     => 'audio/midi',
      'mpga'    => 'audio/mpeg',
      'mp2'     => 'audio/mpeg',
      'mp3'     => 'audio/mpeg',
      'aif'     => 'audio/x-aiff',
      'aiff'    => 'audio/x-aiff',
      'aifc'    => 'audio/x-aiff',
      'm3u'     => 'audio/x-mpegurl',
      'ram'     => 'audio/x-pn-realaudio',
      'rm'      => 'audio/x-pn-realaudio',
      'rpm'     => 'audio/x-pn-realaudio-plugin',
      'ra'      => 'audio/x-realaudio',
      'wav'     => 'audio/x-wav',
      'pdb'     => 'chemical/x-pdb',
      'xyz'     => 'chemical/x-xyz',
      'bmp'     => 'image/bmp',
      'gif'     => 'image/gif',
      'ief'     => 'image/ief',
      'jpeg'    => 'image/jpeg',
      'jpg'     => 'image/jpeg',
      'jpe'     => 'image/jpeg',
      'png'     => 'image/png',
      'tiff'    => 'image/tiff',
      'tif'     => 'image/tiff',
      'djvu'    => 'image/vnd.djvu',
      'djv'     => 'image/vnd.djvu',
      'wbmp'    => 'image/vnd.wap.wbmp',
      'ras'     => 'image/x-cmu-raster',
      'pnm'     => 'image/x-portable-anymap',
      'pbm'     => 'image/x-portable-bitmap',
      'pgm'     => 'image/x-portable-graymap',
      'ppm'     => 'image/x-portable-pixmap',
      'rgb'     => 'image/x-rgb',
      'xbm'     => 'image/x-xbitmap',
      'xpm'     => 'image/x-xpixmap',
      'xwd'     => 'image/x-xwindowdump',
      'igs'     => 'model/iges',
      'iges'    => 'model/iges',
      'msh'     => 'model/mesh',
      'mesh'    => 'model/mesh',
      'silo'    => 'model/mesh',
      'wrl'     => 'model/vrml',
      'vrml'    => 'model/vrml',
      'css'     => 'text/css',
      'html'    => 'text/html',
      'htm'     => 'text/html',
      'asc'     => 'text/plain',
      'txt'     => 'text/plain',
      'rtx'     => 'text/richtext',
      'rtf'     => 'text/rtf',
      'sgml'    => 'text/sgml',
      'sgm'     => 'text/sgml',
      'tsv'     => 'text/tab-separated-values',
      'wml'     => 'text/vnd.wap.wml',
      'wmls'    => 'text/vnd.wap.wmlscript',
      'etx'     => 'text/x-setext',
      'xml'     => 'text/xml',
      'xsl'     => 'text/xml',
      'mpeg'    => 'video/mpeg',
      'mpg'     => 'video/mpeg',
      'mpe'     => 'video/mpeg',
      'qt'      => 'video/quicktime',
      'mov'     => 'video/quicktime',
      'mxu'     => 'video/vnd.mpegurl',
      'avi'     => 'video/x-msvideo',
      'movie'   => 'video/x-sgi-movie',
      'ice'     => 'x-conference/x-cooltalk'
    }

    DEFAULT_TMP_DATA_KEY_FIGURES = 16
    RESOURCE_PATH                = 'resources'
    WEB_SERVER_RESOURCE_PATH     = 'www'
    CGIKIT_PACKAGE               = 'CGIKit'

    class << self
      def create_tmp_data_key
        md5 = Digest::MD5::new
        md5.update Time.now.to_s
        md5.update rand(0).to_s
        md5.update $$.to_s
        md5.hexdigest[0, tmp_data_key_figures]
      end

      def tmp_data_key_figures
        DEFAULT_TMP_DATA_KEY_FIGURES
      end

      def mime( extension )
        MIME[extension]
      end

    end

    attr_reader :packages, :main_package

    def initialize( application, main_package_options = {} )
      @application          = application
      @resource_store       = @application.resource_store
      @resources            = @application.resources
      @web_server_resources = @application.web_server_resources
      @document_root        = @application.document_root || '.'
      @keys                 = {}
      @packages             = {}
      load_package(CGIKIT_PACKAGE)
      @main_package = create_main_package(main_package_options)
      load_packages()
      @packages[@main_package.name] = @main_package
    end

    def create_main_package( options = {} )
      main = Package.new(File.dirname(@application.path), options)
      main.lib_path = nil
      if @application.component_path then
        main.component_path = @application.component_path
      end
      if @application.resource_path then
        main.resource_path = @application.resource_path
      end
      if @application.web_server_resource_path then
        main.web_server_resource_path = @application.web_server_resource_path
      end
      main.load
      info("load main package: #{main.name}")
      main
    end

    def load_package( name )
      info("load package: #{name}")
      @application.package_paths.each do |path|
        fullpath = File.join(path, name)
        if FileTest.exist?(fullpath) then
          package = Package.new(fullpath)
          @packages[package.name] = package
          if @application.database then
            @application.database.add_models(package.paths_for_model)
          end
          return
        end
      end
      raise "#{name}: No such package"
    end

    def load_packages
      package_path = Application::PACKAGE_PATH
      if FileTest.exist?(package_path) then
        Dir.foreach(package_path) do |name|
          path = File.join(Application::PACKAGE_PATH, name)
          if /\A[^.]/ === name and FileTest.directory?(path) then
            load_package(name)
          end
        end
      end
    end

    def package( klass )
      name = klass.to_s.split('::').first
      @packages[name] || @main_package
    end


    #
    # path accessing
    #

    # Returns the public URL for the specified resource
    # when it is under the web server resources directory.
    # Otherwise returns nil.
    def url( name, package_name = nil, languages = [], request = nil )
      url = @resource_store.url(name, request)
      unless url then
        package_name ||= @main_package.name
        package = @packages[package_name]
        path = package.path_for_web_server_resource(@document_root, name, languages)
        if /\A#{File.expand_path(@document_root)}/ === path then
          url = path.sub(File.expand_path(@document_root), '')
        end
      end
      url
    end

    private

    def _resource_url( base, file, request )
      key = file.sub(base, '')
      key = Utilities.escape_url(key)
      @application.resource_request_handler.resource_url(key, request)
    end

    public

    # Returns the file path of the specified resource.
    def path( name, package_name = nil, languages = [] )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.path_for_resource(name, languages) ||
        package.path_for_web_server_resource(@document_root, name, languages)
    end

    # Returns a ByteData object for the specified resource.
    def bytedata( name, package_name = nil, languages = [] )
      data = @resource_store.bytedata(name)
      if data.nil? and filepath = path(name) then
        data = ByteData.new_with_file(filepath)
        data.content_type = content_type(filepath)
      end
      data
    end

    # Finds the content type for extension of the specified path.
    # If the path don't have extension, returns nil.
    def content_type( path )
      # for ruby 1.6
      base = File.basename path
      type = nil
      if base =~ /\.([^\.]+)$/ then
        type = MIME[$1]
      end
      type
    end

    def set_data( data, key, mime )
      data.tmp = key.nil?
      key ||= ResourceManager.create_tmp_data_key
      @resource_store.set_data(data, key, mime)
    end

    def key( data )
      @resource_store.key(data)
    end

    def remove_data( key )
      @resource_store.remove_data( key )
    end

    def flush
      @resource_store.flush
    end

    def message( key, name, package_name, languages )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.message(key, name, languages)
    end

   def nmessage( key, plural_key, n, name, package_name, languages )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.nmessage(key, plural_key, n, name, languages)
   end

  end


  class MemoryResourceStore

    def initialize( application )
      @application = application
      @keys = {}
    end

    def key( data )
      @keys.index(data)
    end

    def url( name, request )
      if @keys.key?(name) then
        @application.resource_request_handler.resource_url(name, request)
      else
        nil
      end
    end

    def bytedata( name )
      @keys[name]
    end

    def set_data( keys, key, mime )
      cache = ByteData.new(keys.to_s)
      cache.content_type = mime
      @keys[key] = cache
    end

    def remove_keys( key )
      @keys.delete(key)
    end

    def flush
      @keys.clear
    end

  end


  class FileResourceStore < MemoryResourceStore

    TMPDIR = 'resource'

    def initialize( application )
      super
      @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
    end

    def url( name, request )
      if FileTest.exist?(tmpfile(name)) then
        @application.resource_request_handler.resource_url(name, request)
      else
        nil
      end
    end

    def bytedata( name )
      unless data = super then
        path = File.join(@tmpdir, name)
        if FileTest.exist?(path) then
          cache = nil
          FileLock.shared_lock(path) do |file|
            cache = Marshal.load(file)
          end
          data = ByteData.new(cache[:data])
          data.path = path
          data.content_type = cache[:mime]
        end
      end
      data
    end

    def set_data( data, key, mime )
      super
      unless FileTest.directory? @tmpdir
        require 'ftools'
        File.makedirs @tmpdir
      end
      cache = {}
      cache[:data] = data
      cache[:key]  = key
      cache[:mime] = mime
      FileLock.exclusive_lock(tmpfile(key)) do |file|
        Marshal.dump(cache, file)
      end
    end

    def remove_data( key )
      super
      path = tmpfile(key)
      if FileTest.exist?(path)
        File.delete(path)
      end
    end

    def tmpfile( filename )
      File.join(@tmpdir, filename).untaint
    end

    def flush
      super
      Dir.foreach(@tmpdir) do |file|
        unless /\A\./ === file
          path = File.join(@tmpdir, file)
          File.delete(path)
        end
      end
    end

  end


  module ResourceLoadable

    def caching_url( file, package, data, key, mime, request )
      rm = @application.resource_manager
      url = nil
      if file then
        url = rm.url(file, package, request.languages, request)
        unless url then
          data = rm.bytedata(file)
        end
      end

      if data and url.nil? then
        if String === data then
          data = ByteData.new(data)
        end
        mime = data.content_type || mime
        rm.set_data(data, key, mime)
        key ||= rm.key(data)
        url = rm.url(key, package, request.languages, request)
      end
      url
    end

    def resource_urls( files, package, request )
      urls = []
      unless Array === files then
        files = [files]
      end
      rm = @application.resource_manager
      files.each do |file|
        if url = rm.url(file, package, request.languages, request) then
          urls << url
        end
      end
      urls
    end

    def cgikit_resource_urls( files )
      urls = []
      files.each do |file|
        if url = cgikit_resource_url(file) then
          urls << url
        end
      end
      urls
    end

    def cgikit_resource_url( file )
      @application.resource_manager.url(file, ResourceManager::CGIKIT_PACKAGE)
    end

  end

end

