require 'uconv'

module CGIKit::Project

  class Template
    attr_accessor :project

    def initialize( project )
      @project = project
    end

    def write
      if @project.japanese? then
        str = template_ja
      else
        str = template_en
      end
      File.open(path, 'w') do |f| f.write(str) end
    end

  end


  class ApplicationTemplate < Template

    DEFAULT_PORT = 8080

    def path
      File.join(@project.lib_path, 'application.rb')
    end

    def class_name
      "#{@project.name}::Application"
    end

  end


  class SessionTemplate < Template

    def path
      File.join(@project.lib_path, 'session.rb')
    end

    def class_name
      "#{@project.name}::Session"
    end

  end


  class DirectActionTemplate < Template

    def path
      File.join(@project.lib_path, 'directaction.rb')
    end

    def class_name
      "#{@project.name}::DirectAction"
    end

  end


  class RSSActionTemplate < Template

    def path
      File.join(@project.lib_path, 'rssaction.rb')
    end

    def class_name
      "#{@project.name}::RSSAction"
    end

  end


  class AjaxActionTemplate < Template

    def path
      File.join(@project.lib_path, 'ajaxaction.rb')
    end

    def class_name
      "#{@project.name}::AjaxAction"
    end

  end


  class ComponentTemplate < Template

    COMPONENT_LIB = './components'

    attr_accessor :name, :project_name, :superclass_name

    def initialize( project, name, superclass_name = nil )
      super(project)
      if /\A([\w:]+)::([\w]+)\Z/ === name then
        @project_name = $1
        @name = $2
      else
        @name = name
      end
      @superclass_name = superclass_name || 'CGIKit::Component'
    end

    def class_name
      "#{@project.name}::#@name"
    end

    def template_path
      name = "#@name.html"
      if @project_name then
        name
      else
        File.join(@project.component_path, name)
      end
    end

    def declaration_path
      name = "#@name.ckd"
      if @project_name then
        name
      else
        File.join(@project.component_path, name)
      end
    end

    def script_path
      name = "#@name.rb"
      if @project_name then
        name
      else
        File.join(@project.component_path, name)
      end
    end

    def write
      write_template
      write_declaration
      write_script
    end

    def write_template
      if @superclass_name and @superclass_name != 'CGIKit::Component' then
        unless str = superclass_template('html') then
          raise "#@superclass_name: Can't find template file"
        end
      elsif @project.japanese? then
        str = template_template_ja
      else
        str = template_template_en
      end
      File.open(template_path, 'w') do |f| f.write(str) end
    end

    def superclass_template( ext )
      basename = @superclass_name.split('::').last
      $LOAD_PATH.each do |loadpath|
        filename = "#{basename}.#{ext}"
        cgikit = 'cgikit/components'
        paths = [File.join(loadpath, cgikit, filename),
          File.join(loadpath, cgikit, basename, filename)]
        paths.each do |path|
          path.untaint
          if FileTest.exist?(path) then
            content = nil
            File.open(path) do |f| content = f.read end
            return content
          end
        end
      end
      nil
    end

    def write_declaration
      if @superclass_name and @superclass_name != 'CGIKit::Component' then
        unless str = superclass_template('ckd') then
          raise "#@superclass_name: Can't find binding file"
        end
      else
        str = "{\n}\n"
      end
      File.open(declaration_path, 'w') do |f| f.write(str) end
    end

    def write_script
      if @project.japanese? then
        str = script_template_ja
      else
        str = script_template_en
      end
      File.open(script_path, 'w') do |f| f.write(str) end
    end

    def project_name
      @project_name || @project.name
    end

  end


  class ConfigurationTemplate < Template

    CONF_FILE = './cgikitconf.rb'

    def path
      File.join(@project.project_path, 'cgikitconf.rb')
    end

  end


  class ResourceTemplate < Template

    def path
      File.join(@project.project_path, 'resources')
    end

    def write
      FileUtils.makedirs(path)
    end

  end


  class WebServerResourceTemplate < Template

    def path
      File.join(@project.project_path, 'www')
    end

    def write
      FileUtils.makedirs(path)
    end

  end


  class PackageTemplate < Template

    def path
      File.join(@project.project_path, 'packages')
    end

    def write
      FileUtils.makedirs(path)
    end

  end


  class ScriptTemplate < Template

    def path
      File.join(@project.project_path, "#{@project.name}.#{self.class.suffix}")
    end

  end


  class CGIScriptTemplate < ScriptTemplate

    def self.suffix
      "cgi"
    end

  end


  class ModRubyScriptTemplate < ScriptTemplate

    def self.suffix
      "rbx"
    end

    def content
    end

  end


  class WEBrickScriptTemplate < ScriptTemplate

    def self.suffix
      "rb"
    end

    def content
    end

  end


  class FastCGIScriptTemplate < ScriptTemplate

    def self.suffix
      "fcgi"
    end

    def content
    end

  end

end
