require 'strscan'

module CGIKit; end
module CGIKit::Project

  class ComponentConverter

    def initialize( files, ckid = true )
      @files = files
      @ckid = ckid
      @converters = []
      @files.each do |file|
        file = File.expand_path(file)
        if result = convert(file) then
          @converters.concat(result)
        end
      end
    end

    def convert( file )
      if /\A\./ === file then
        # do nothing
      elsif FileTest.directory?(file) then
        convert_dir(file)
      elsif TemplateConverter.convert?(file) then
        [TemplateConverter.new(file, @ckid)]
      elsif DeclarationConverter.convert?(file) then
        [DeclarationConverter.new(file)]
      elsif ScriptConverter.convert?(file) then
        [ScriptConverter.new(file)]
      end
    end

    def convert_dir( path )
      converter = []
      Dir.foreach(path) do |file|
        unless /\A\./ === file then
          if result = convert(File.join(path, file)) then
            converter.concat(result)
          end
        end
      end
      converter
    end

    def write( path )
      @converters.each do |c|
        c.write(path)
      end
    end

  end


  class TemplateConverter

    def self.convert?( path )
      /\.html\Z/ === path
    end

    def initialize( path, ckid = true )
      @ckid = ckid
      @path = path
    end

    def write( path )
      content = convert(@path)
      out = File.join(path, File.basename(@path))
      File.open(out, 'w') do |f| f.write(content) end
    end

    def convert( path )
      content = ''
      File.open(path) do |f|
        content = f.read
      end
      scanner = StringScanner.new(content, false)

      content.gsub!(/<\!---([^-](.*?)[^-])--->/mi) do "<!-- ck #$1 -->" end
      content.gsub!(/<\/cgikit\s*>/i, "</span>")
      content.gsub!(/<cgikit\s+name\s*=\s*([^\/>]+)\/>/i) do
        name = $1
        name.tr!("\"'", '')
        if @ckid then
          "<span ckid=\"#$1\"/>"
        else
          "<span id=\"ck:#$1\"/>"
        end
      end
      content.gsub!(/<cgikit\s+name\s*=\s*([^\/>]+)>/i) do
        name = $1
        name.tr!("\"'", '')
        if @ckid then
          "<span ckid=\"#$1\">"
        else
          "<span id=\"ck:#$1\">"
        end
      end
      content
    end

  end


  class DeclarationConverter

    def self.convert?( path )
      /\.ckd\Z/ === path
    end

    def initialize( path )
      @path = path
    end

    def write( path )
      defs = parse_ckd_file(@path)
      content = format(defs)
      File.open(File.join(path, File.basename(@path)), 'w') do |f|
        f.write(content)
      end
    end

    def parse_ckd_file(filename)
      string = nil

      File.open(filename) do |f|
        string = f.read
      end

      parse(string, filename)
    end

    def parse(string, filename = '')
      scanner = StringScanner.new( string, false )

      # The way to use these local variables is bad.      
      definitions     = []
      in_attrs        = false
      line_num        = 1      
      current_element = nil
      element_name    = nil
      element_type    = nil
      term            = "(;|\n)"

      # in the future, we may have to change this Proc object.
      pos_proc = proc do 
        "#{filename}:#{line_num}: "
      end
      
      while scanner.rest?
        # line break
        if scanner.skip(/(\r)?\n/) then
          line_num += 1
          next
        end

        # skip commnet or space
        if scanner.skip(/(?:#.*$|[ \r\f\t]+)/) then
          next
        end

        if in_attrs then
          # attribute
          if scanner.skip(/([a-zA-Z0-9_]+[\!\?]?)[ \r\f\t]*=[ \r\f\t]*/)
            key = scanner[1]

            # array and hash literal
            if scanner.skip(/^([^"'][^"';\n]*)\[/) then
              list   = nil
              object = scanner[1]

              if scanner.match?(/^(["'])/) then
                # string literal
                close = scanner[1]
                if     close == '"' and scanner.skip(/"(.*?)"\]/)
                elsif  close == "'" and scanner.skip(/'(.*?)'\]/)
                else
                  raise CKDParseError, \
                  pos_proc.call << scanner.peek(60).inspect << \
                  "\nString literal error. Or, \";\" doesn't exist."
                end
                list = "\"#{scanner[1]}\""
              else
                if scanner.skip(/([a-zA-Z0-9_\.\!\^?]+)\][ \r\t\f]*#{term}/)
                  list = scanner[1]
                else
                  raise CKDParseError, pos_proc.call << \
                  scanner.peek(60).inspect << \
                  "\nArray or Hash literal error. Or, \";\" doesn't exist.\n"
                end
              end
              value = "#{object}[#{list}]"
              line_num += value.count("\n")

              # value of attribute
            elsif scanner.match?(/^([\"\'])/) then
              # string lietral
              close = scanner[1]
              if     close == '"' and scanner.skip(/"(.*?)"[ \r\f\t]*#{term}/m)
              elsif  close == "'" and scanner.skip(/'(.*?)'[ \r\f\t]*#{term}/m)
              else
                raise CKDParseError, \
                pos_proc.call << scanner.peek(60).inspect << \
                "\nString literal error. Or, \";\" doesn't exist."
              end

              value = "\"#{scanner[1]}\""
              line_num += value.count("\n")
            else
              # others
              #
              # This regexp is not accurate.
              if scanner.skip(/([a-zA-Z0-9_\.\!\^?]+)[ \r\t\f]*#{term}/)
                value = ":\"#{scanner[1]}\""
              else
                raise CKDParseError, pos_proc.call << \
                scanner.peek(60).inspect << \
                "\nMethod name error. Or, \";\" doesn't exist." "\n" 
              end
            end

            current_element[key] = value      
            next
          end

          # end of definition
          if scanner.skip(/\s*\}/)
            definitions << current_element
            current_element = nil
            in_attrs = false
            next
          end

          # skips space and separator.
          #
          # (ex.)
          # foo : CKString{;}
          if scanner.skip(/[ \r\t\f]*#{term}/)
            next
          end
        else
          # extracts name of the element and its class
          # class's regexp is not precise.
          if scanner.skip(/([a-zA-Z0-9_]+)\s*:\s*([a-zA-Z0-9_]+)\s*\{/)
            element_name = scanner[1]
            element_type = scanner[2]
            current_element = {}
            current_element['oid'] = element_name
            current_element['element'] = element_type

            definitions.each do |definition|
              if definition['oid'] == element_name then
                raise CKDParseError, pos_proc.call << \
                scanner.peek(60).inspect << \
                "\n'#{element_name}' definition is already existed." "\n" 
              end
            end
            
            in_attrs = true
            next
          end
        end

        def_str = _pretty_print(definitions)
        raise CKDParseError, pos_proc.call << scanner.peek(60).inspect << \
        "\nNot match any rule.\n\n" << def_str << "\n" 
      end

      # Now, there is no data to be parsed.      
      if in_attrs 
        raise CKDParseError, pos_proc.call << \
        "The last element is not enclosed.\n\n"
      end
      
      definitions
    end

    def pretty_print(defs)
      print _pretty_print(defs)
    end

    def _pretty_print(defs)
      s = ''
      indent = '    '

      defs.keys.sort.each do |name|
        attrs = defs[name]
        s << name
        s << ":\n"
        
        attrs.keys.sort.each do |k|
          v = attrs[k]
          s << indent 
          s << k 
          s << ': '
          s << v.inspect
          s << "\n"
        end

        s << "\n"
      end

      s
    end

    def format( defs )
      string = "{\n"
      defs.each do |d|
        string << "  :#{d['oid']} => {\n"
        element = element_name(d['element'])
        string << "    :element => #{element},\n"
        d.each do |key, value|
          if (key != 'oid') and (key != 'element') then
            key, value = key_value_for_element(element, key, value)
            string << "    :#{key} => #{value},\n" if key
          end
        end
        string.chop!
        string.chop!
        string << "\n  },\n\n"
      end
      string.chomp!
      string << '}'
      string
    end

    def element_name( old )
      case old
      when 'CKString' then 'String'
      when 'CKHyperlink' then 'Link'
      when 'CKImage' then 'Image'
      when 'CKForm' then 'Form'
      when 'CKTextField' then 'TextField'
      when 'CKText' then 'Text'
      when 'CKBrowser' then 'Browser'
      when 'CKPopUpButton' then 'Popup'
      when 'CKCheckbox' then 'Checkbox'
      when 'CKRadioButton' then 'Radio'
      when 'CKSubmitButton' then 'Submit'
      when 'CKResetButton' then 'Reset'
      when 'CKFileUpload' then 'Upload'
      when 'CKConditional' then 'Conditional'
      when 'CKRepetition' then 'Repetition'
      when 'CKFrame' then 'Frame'
      when 'CKContent' then 'Content'
      when 'CKGenericElement' then 'GenericElement'
      else
        old
      end
    end

    def key_value_for_element( element, key, value )
      case key
      when 'page' then value = class_name_from(value)
      end

      case element
      when 'GenericElement'
        case key
        when 'enabled' then key = 'displayed'
        when 'string' then key = nil
        when 'option' then key = nil
        end
      when 'Form'
        case key
        when 'fileupload' then key = 'upload'
        end
      when 'Popup'
        case key
        when 'values' then key = 'value'
        when 'selected' then key = 'selection'
        end
      when 'Browser'
        case key
        when 'values' then key = 'value'
        when 'selected' then key = 'selections'
        end
      when 'Upload'
        case key
        when 'data' then key = 'value'
        when 'file' then key = nil
        end
      end
      [key, value]
    end

    def class_name_from( str )
      str.tr("\"'", '')
    end

  end

  
  class ScriptConverter

    def self.convert?( path )
      /\.rb\Z/ === path
    end

    def initialize( path )
      @path = path
    end

    def write( path )
      content = nil
      open(@path) do |f| content = f.read end

      content.gsub!('CKComponent', 'CGIKit::Component')
      content.gsub!('CKByteData', 'CGIKit::ByteData')
      content.gsub!('CKCookie', 'CGIKit::Cookie')
      content.gsub!('CKRequest', 'CGIKit::Request')
      content.gsub!('CKResponse', 'CGIKit::Response')
      content.gsub!('CKSession', 'CGIKit::Session')

      path = File.join(path, File.basename(@path))
      open(path, 'w') do |f| f.write(content) end
    end

  end

end
