module CGIKit

  module Delegatable

    attr_accessor :delegate

    def delegate?( method )
      @delegate and @delegate.respond_to?(method)
    end

    def ask( method, context = nil )
      if delegate?(method)
        @delegate.context = context if context
        yield @delegate
      end
    end

  end


  class Delegate

    SUPPORT_CLASSES = []

    attr_accessor :element, :component, :context, :bindings

    class << self

      def support_classes
        SUPPORT_CLASSES
      end

      def support?( klass )
        klasses = support_classes()
        klasses.empty? or klasses.include?(klass)
      end

    end

    def value( name, does_resolve = true )
      value = @bindings[name]
      if does_resolve and Symbol === value then
        component.value_for_key(@bindings[name])
      else
        value
      end
    end

    def bool( name, does_resolve = true )
      Association.adapt_to_bool(value(name, does_resolve))
    end

    def set_value( name, value )
      component.take_value_for_key(@bindings[name], value)
    end

    def has_binding?( name )
      @bindings.key?(name)
    end

  end


  class HTMLTag

    class << self

      def a( attributes = {}, other = '', content = '' )
        new(:a, attributes, other, content)
      end

      def img( attributes = {}, other = '' )
        new(:img, attributes, other)
      end

      def form( attributes = {}, other = '' )
        new(:form, attributes, other)
      end

      def textarea( attributes = {}, other = '', content = '' )
        new(:textarea, attributes, other, content)
      end

      def input( attributes = {}, other = '' )
        new(:input, attributes, other)
      end

      def textfield( attributes = {}, other = '' )
        attributes[:type] ||= 'text'
        new(:input, attributes, other)
      end

      def radio( attributes = {}, other = '' )
        attributes[:type] = 'radio'
        new(:input, attributes, other)
      end

      def checkbox( attributes = {}, other = '' )
        attributes[:type] = 'checkbox'
        new(:input, attributes, other)
      end

      def submit( attributes = {}, other = '' )
        attributes[:type] = 'submit'
        new(:input, attributes, other)
      end

      def reset( attributes = {}, other = '' )
        attributes[:type] = 'reset'
        new(:input, attributes, other)
      end

      def upload( attributes = {}, other = '' )
        attributes[:type] = 'upload'
        new(:input, attributes, other)
      end

      def select( attributes = {}, other = '' )
        new(:select, attributes, other)
      end

      def option( attributes = {}, other = '', content = '' )
        new(:option, attributes, other, content)
      end

    end

    attr_accessor :name, :attributes, :other, :content

    def initialize( name = nil, attributes = {}, other = '', content = '' )
      @name = name
      @attributes = attributes
      @other = other
      @content = content
    end

    def []( key )
      @attributes[key]
    end

    def []=( key, value )
      @attributes[key] = value
    end

    def open_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << ">"
      s
    end

    def close_tag
      "</#@name>"
    end

    def empty_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << " />"
      s
    end

    def container_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << ">#@content</#@name>"
      s
    end

    def attributes_string
      s = ''
      @attributes.each do |key, value|
        if value then
          case key
          when :checked
            s << "checked=\"checked\" "
          when :selected
            s << "selected=\"selected\" "
          when :disabled
            s << "disabled=\"disabled\" "
          else
            s << "#{key}=\"#{value}\" "
          end
        end
      end
      s.chop!
      s
    end

  end

end
