module CGIKit

class Component < Element
  include KeyValueCoding

  CACHE_TEMPLATE_DIR = 'cache'

  # Parent component.
  attr_reader :parent

  attr_accessor :declaration_store, :context

  attr_accessor :context_id, :subcomponents, :application, :declaration_name

  def initialize( context, *args )
    @access_attributes = true
    @subcomponents = []
    @associations = nil
    load_files(context)
    __send__(:init, *args)
  end


  #
  # hook
  #

  # Invoked to initialize component.
  # Form data is not setted when the method is called.
  def init; end

  # Invoked when the component parses template.
  # A string returned by the method is used instead of template file.
  def will_parse_template; end

  # Invoked when the component parses declarations.
  # A hash returned by the method is used instead of declaration file.
  def will_parse_declarations; end

  # Invoked when saving the component.
  # If the component has data can't be saved or marshaled with FileSessionStore,
  # you must clear the data in this method.
  def will_save_page; end

  # Invoked after restoring the component.
  def did_restore_page; end


  #
  # accessing
  #

  def root
    component = self
    while (component.root? == false)
      component = component.parent
    end
    component
  end

  def []( key )
    value_for_key(key)
  end

  def []=( key, value )
    take_value_for_key(key, value)
  end

  # Returns base name of the component in name space package.
  def base_name
    self.class.to_s.split('::').last
  end

  # Creates a specified page component.
  def page( name, *args )
    @application.page(name, @context, *args)
  end

  def parent=( parent )
    @parent = parent
    @parent.subcomponents << self
  end

  # Path for the component.
  def path
    @path ||= package.path_for_component(base_name(), request.languages)
  end


  #
  # testing
  #

  def root?
    @parent.nil?
  end

  def subcomponent?
    @associations and @parent
  end

  # Return true if parent component syncronizes bound attribute to self.
  # Override this method returns false to create a non-synchronizing component.
  def sync?
    true
  end

  def stateless?
    false
  end

  def has_session?
    @context.has_session?
  end

  def loaded?
    @loaded
  end

  def has_binding?( name )
    @associations.key?(name)
  end

  def can_set_binding_value?( name )
    has_binding?(name) and Symbol === binding_value(name, false)
  end

  def can_get_binding_value?( name )
    has_binding?(name)
  end


  #
  # loading
  #

  # loads template, merges declaration hash, loads declaration file
  def load_files( context )
    @context = context
    @application = context.application

    template = @application.template_store.checkout(self)
    @template_node = template.template_node
    @declaration_store = template.declaration_store

    load_associations_from_parent_declaration
    validate_api
    @loaded = true
  end

  def load_associations_from_parent_declaration
    if parent and @declaration_name then
      @associations = parent.declaration_store[@declaration_name]
    end
  end

  def validate_api
    if application.validate_api? then
      @declaration_store.validate_api(self.class)
    end
  end


  #
  # request-response loop
  #

  def begin_context( context )
    increment_and_record_if_subcomponent(context)
    init_context_id(context)
  end

  def increment_and_record_if_subcomponent( context )
    if subcomponent? then
      context.increment
      @parent_context_id = context.context_id
    end
  end

  def init_context_id( context )
    context.delete_all
    if context.has_session? then
      context.init_component_id_with_session(self)
    else
      context.init_component_id_without_session(self)
    end
    context.append_zero
  end

  def end_context( context )
    if subcomponent? then
      context.component = parent
      context.context_id = @parent_context_id
    end
  end

  def sync( context, &block )
    unless loaded? then
      awake_from_restoration(context)
    end
    if subcomponent? and sync? then
      pull_values_from_parent
      result = block.call if block_given?
      push_values_to_parent
      result
    elsif block_given?
      block.call
    end
  end

  def take_values_from_request( request, context )
    sync(context) do
      @template_node.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    sync(context) do
      @template_node.invoke_action(request, context)
    end
  end

  def append_to_response( response, context )
    sync(context) do
      if !stateless? and context.has_session? then
        context.session.save_page(self) 
      end
      @template_node.append_to_response(response, context)
    end
  end

  def generate_response
    info()
    response = Response.new_with_response(response())
    handler = application.component_request_handler
    handler.append_to_response(self, response, context)
    response.component = self
    response
  end

  def to_html
    generate_response.content
  end


  #
  # marshaling
  #

  def dup
    component = super
    component.context = @context
    component.declaration_store = @declaration_store
    component.node = @template_node
    component
  end

  def marshal_dump
    will_save_page
    not_includes = ['@context', '@declaration_store', '@template_node',
      '@node', '@application', '@loaded']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  # Invoked after the component is restored to initialize.
  # This method invokes deletage method did_restore_page() finally.
  def awake_from_restoration( context )
    @context = context
    @application = context.application
    debug(object_id(), true)
    load_files(context)
    @subcomponents.each do |sub|
      sub.awake_from_restoration(context)
    end
    did_restore_page
  end

  def add_all_components( session, permanently = false )
    unless stateless? then
      session.add_component(self, @context, permanently)
      @subcomponents.each do |sub|
        sub.add_all_components(session, permanently)
      end
    end
  end


  #
  # aliases
  #

  # Returns a request object of the context.
  def request; @context.request; end

  # Returns a response object of the context.
  def response; @context.response; end

  # Returns the session object.
  # If the session isn't existed, returns a new session.
  def session; @context.session; end

  # Returns a resource manager object.
  def resource_manager; @application.resource_manager; end


  #
  # template paths
  #

  def package
    @package ||= resource_manager().package(self.class)
  end

  # Returns name of the template file.
  def template_file
    package().path_for_component("#{base_name()}.html", request().languages)
  end

  # Returns name of the declaration( binding ) file.
  def declaration_file
    package().path_for_component("#{base_name()}.ckd", request().languages)
  end


  #
  # Synchronizing components
  #

  def next_subcomponent( context_id )
    if @context_id == context_id then
      return @subcomponents.first
    end
    @subcomponents.each_with_index do |component, index|
      if context_id == component.context_id then
        return @subcomponents[index]
      end
    end
    nil
  end

  # Allows a subcomponent to invoke an action method of its parent component
  # bound to the child component.
  def perform_parent_action( name )
    @parent.value_for_key(name) if sync?
  end

  def pull_values_from_parent
    @associations.each do |key, as|
      if @api and @api.has_binding?(key) and as.settable_in_component?(self) then
        take_value_for_key(key, as.value)
      else
        take_value_for_key(key, as.value(@parent))
      end
    end
  end

  def push_values_to_parent
    @associations.each do |key, as|
      if (key != Declaration::ELEMENT_KEY) and @api.nil? or \
        (@api and @api.has_binding?(key) and \
         (as.settable_in_component?(self) == false)) then
        @parent.take_value_for_key(key, value_for_key(key))
      end
    end
  end

  def binding_value( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.value(@parent)
      else
        as.value
      end
    else
      nil
    end
  end

  def set_binding_value( name, value )
    if as = @associations[name] then
      as.set_value(value, @parent)
    end
  end


  #
  # message accessing
  #

  def message( key, name = nil, package_name = nil, languages = nil )
    package_name ||= package().name
    languages ||= request().languages
    resource_manager().message(key, name, package_name, languages)
  end
  alias _ message

  def nmessage( key, plural_key, n, name = nil, package_name = nil, languages = nil )
    package_name ||= package().name
    languages ||= request().languages
    resource_manager().nmessage(key, plural_key, n, name, package_name, languages)
  end
  alias n_ nmessage

end


class StatelessComponent < Component

  def sync?
    false
  end

  def stateless?
    true
  end

end


  # backward compatibility
  class ErrorPage < Component; end
  CKErrorPage = ErrorPage

end
