if __FILE__ == $0
	$LOAD_PATH.unshift '../lib' 
	require 'test/unit'
	require 'test/unit/ui/console/testrunner'
	require 'cgikit'
	require 'cgikit/command'
	require 'TestApp/app'
	require 'TestApp/SessionPage/SessionPage'
	require 'test_common'
end


# session - fetch, store, remove
# auth - timeout, auth(agent/addr)
# terminate?
# cache - cache, permanent
# save, restore
# component - add, add_for_ids, cid, component
# marshal
class TestSession < Test::Unit::TestCase
	include CGIKit, TestCommon

	def setup
		@app = testapp()
		@app.main = SessionPage
		@session = Session.new
		@session.awake_from_restoration(@app)
		@request = Request.new
		@context = Context.new(@request, @app)
		@context.session = @session
		@session.context = @context
	end

	def component( context_id = '0.0' )
		cmp = @app.run.component
		ctxt = Context.new
		ctxt.context_id = context_id
		cmp.context = ctxt
		cmp		
	end


	#
	# session values
	#

	def test_values_to_access_and_remove
		expected = {:key=>:value}
		expected.each do |key, value|
			@session[key] = value
			assert_equal(value, @session[key])
			@session.remove(key)
			assert_nil(@session[key])
		end
	end


	#
	# terminate
	#

	def test_terminate_session
		assert(!@session.terminate?)
		@session.terminate
		assert(@session.terminate?)
	end

	def test_terminate_session_by_timeout
		assert(!@session.terminate?)
		@session.last_accessed_time = Time.new - Session::DEFAULT_TIMEOUT * 2
		assert(@session.terminate?)
	end


	#
	# authorization
	#

	def test_timeout
		assert(!@session.timeout?)
		@session.last_accessed_time = Time.new - Session::DEFAULT_TIMEOUT * 2
		assert(@session.timeout?)
	end

	def test_no_timeout
		@session.timeout = 0
		assert(!@session.timeout?)
		@session.last_accessed_time = Time.new - Session::DEFAULT_TIMEOUT * 2
		assert(!@session.timeout?)
	end

	def test_remote_addr
		@session.auth_by_remote_addr = true
		@session.remote_addr = '127.0.0.1'
		assert(!@session.remote_addr?('127.0.0.2'))
	end

	def test_user_agent
		@session.auth_by_user_agent = true
		@session.user_agent = 'agentA'
		assert(!@session.user_agent?('agentB'))
	end


	#
	# component
	#

	def test_add_component_for_ids
		cmp = @app.run.component
		cmp_id = 10
		ctxt_id = 'test'
		@session.add_component_for_ids(cmp, cmp_id, ctxt_id, false)
		assert_equal(cmp, @session.caches[cmp_id])
		@session.add_component_for_ids(cmp, cmp_id, ctxt_id, true)
		assert_equal(cmp, @session.permanent_caches[cmp_id])
	end

	def test_add_component_with_new
		cmp = @app.run.component
		cmp_id = @session.add_component(cmp)
		assert_equal(0, cmp_id)
		assert_equal(cmp, @session.component('0.0'))
	end

	def test_add_component_serial
		cmp1 = @app.run.component
		cmp2 = @app.run.component
		cmp3 = @app.run.component
		assert_equal(0, @session.add_component(cmp1))
		assert_equal(1, @session.add_component(cmp2))
		assert_equal(2, @session.add_component(cmp3))
	end

	def test_add_component_twice
		cmp = @app.run.component
		cmp_id1 = @session.add_component(cmp)
		cmp_id2 = @session.add_component(cmp)
		assert_equal(cmp_id1, cmp_id2)
	end

	def test_add_component_with_context
		cmp = @app.run.component
		ctxt = Context.new
		ctxt.context_id = '1.1.1'
		@session.add_component(cmp, ctxt, false)
		assert_equal(cmp, @session.component(ctxt.context_id))
	end

	def test_add_component_to_cache
		cmp = @app.run.component
		ctxt = Context.new
		ctxt.context_id = 'test'
		cmp_id = @session.add_component(cmp, ctxt, false)
		assert_equal(cmp, @session.caches[cmp_id])
	end

	def test_add_component_to_permanent_cache
		cmp = @app.run.component
		ctxt = Context.new
		ctxt.context_id = 'test'
		cmp_id = @session.add_component(cmp, ctxt, true)
		assert_equal(cmp, @session.permanent_caches[cmp_id])
	end

	def test_component_id
		cmp = component()
		@session.save_page(cmp)
		assert_not_nil(@session.component_id(cmp))
	end

	def test_component_for_component_id
		cmp = component()
		@session.save_page(cmp)
		cid = @session.component_id(cmp)
		assert_equal(cmp, @session.component_for_component_id(cid))
	end

	def test_component_for_context_id
		ctxtid = '0.1.2'
		cmp = component(ctxtid)
		@session.save_page(cmp)
		assert_equal(cmp, @session.component(ctxtid))
	end


	#
	# saving and restoring
	#

	def test_save_page
		cid = '0.1.2'
		cmp = component(cid)
		@session.save_page(cmp)
		assert_not_nil(@session.component_id(cmp))
		assert_equal(cmp, @session.component(cid))
	end

	def test_save_page_permanently
		cid = '0.1.2'
		cmp = component(cid)
		@session.save_page(cmp, true)
		cmpid = @session.component_id(cmp)
		assert_not_nil(cmpid)
		assert_equal(cmp, @session.component(cid))
		assert_equal(cmp, @session.permanent_caches[cmpid])
	end

	def test_save_page_to_remove_cache
		@app.page_cache_size = 1
		cid1 = '0.1.1'
		cid2 = '0.1.2'
		cmp1 = component(cid1)
		cmp2 = component(cid2)
		@session.save_page(cmp1)
		@session.save_page(cmp2)
		assert_nil(@session.component(cid1))
		assert_not_nil(@session.component(cid2))
	end

	def test_save_page_to_remove_permanent_cache
		@app.permanent_page_cache_size = 1
		cid1 = '0.1.1'
		cid2 = '0.1.2'
		cmp1 = component(cid1)
		cmp2 = component(cid2)
		@session.save_page(cmp1, true)
		@session.save_page(cmp2, true)
		assert_nil(@session.component(cid1))
		assert_not_nil(@session.component(cid2))
	end

	# restore_page only invokes component().
	# def test_restore_page; end


	#
	# marshaling
	#

	# values, cache, permanent, frame, user_agent, remote_addr, timeout, session_id
	def test_marshal
		path = 'TestSession.test_mershal'
		@session[:key] = :value
		@session.caches[:cmpid] = :cmp
		@session.permanent_caches[:cmpid] = :cmp
		@session.frame_components[:cmpid] = :cmp
		dump_session(@session, path)

		loaded = load_session(path)
		assert_equal(@session.values, loaded.values)
		assert_equal(@session.user_agent, loaded.user_agent)
		assert_equal(@session.remote_addr, loaded.remote_addr)
		assert_equal(@session.session_id, loaded.session_id)
		assert_equal(@session.last_accessed_time, loaded.last_accessed_time)
		assert_equal(@session.caches, loaded.caches)
		assert_equal(@session.permanent_caches, loaded.permanent_caches)
		assert_equal(@session.frame_components, loaded.frame_components)

		File.delete(path)
	end

	def dump_session( session, path )
		FileLock.exclusive_lock(path) do |file|
			Marshal.dump(session, file)
		end
	end

	def load_session( path )
		FileLock.shared_lock(path) do | file |
			return Marshal.load(file)
		end
	end


	#
	# validating
	#

	def test_validate_timeout
		@session.last_accessed_time = Time.new - Session::DEFAULT_TIMEOUT * 2
		assert_raises(Application::SessionTimeoutError) do
			@session.validate_timeout
		end
	end

	def test_validate_authorization_by_user_agent
		@session.auth_by_user_agent = true
		@session.user_agent = 'agentA'
		@request.user_agent = 'agentB'
		assert_raises(Application::SessionAuthorizationError) do
			@session.validate_authorization
		end
	end

	def test_validate_authorization_by_remote_addr
		@session.auth_by_remote_addr = true
		@session.remote_addr = '192.168.0.1'
		@request.remote_addr = '192.168.0.2'
		assert_raises(Application::SessionAuthorizationError) do
			@session.validate_authorization
		end
	end

	def test_validate_authorization_by_both
		@session.auth_by_user_agent = true
		@session.user_agent = 'agentA'
		@request.user_agent = 'agentB'
		@session.auth_by_remote_addr = true
		@session.remote_addr = '192.168.0.1'
		@request.remote_addr = '192.168.0.2'
		assert_raises(Application::SessionAuthorizationError) do
			@session.validate_authorization
		end
	end


	#
	# managing cookie
	#

	def test_set_cookie
		@session.set_cookie
		assert(!@context.response.cookies.empty?)
		cookie = @context.response.cookies[0]
		assert_equal(@session.session_key, cookie.name)
		assert_equal(@session.session_id, cookie.value)
		expires = Time.new + @session.cookie_expires
		assert_equal(expires.to_s, cookie.expires.to_s)
	end

	def test_remove_cookie
		@session.remove_cookie
		assert(!@context.response.cookies.empty?)
		cookie = @context.response.cookies[0]
		assert_equal(@session.session_key, cookie.name)
		assert_nil(cookie.value)
		assert(Time.new > cookie.expires)
	end

end


if __FILE__ == $0 then
	suite = TestSession.suite
	runner = Test::Unit::UI::Console::TestRunner.new suite
	runner.start
end
