require 'cgikit'
require 'test/unit'

class CGIKitParserTest < Test::Unit::TestCase
  
  def setup
    @pr = CGIKit::HTMLParser::HTMLParser.new(nil)
  end
  
  def test_ckid
    s = <<-EOF
    <html>
    <div ckid='foo'></div>
    </html>
    EOF
    @pr.parse(s)
    
    assert_nothing_raised{
      @pr.node.component
    }
    assert_equal(3, @pr.node.size)
    assert_equal("<html>\n    ", @pr.node[0].content)
    assert_equal(:foo, @pr.node[1].ckid)
    assert_equal(0, @pr.node[1].node.size)
    
    
    assert_equal(1, @pr.declarations.size)
    assert_equal([:foo], @pr.declarations.keys)
    assert_equal([{}],@pr.declarations.values)
  end
  
  def test_id2ckid
    s = <<-EOF
    <hoge></hoge>
    <div id='ck:foo'/>
    EOF
    @pr.parse(s)
    
    assert_equal(3, @pr.node.size)
    assert_equal(:foo, @pr.node[1].ckid)
    assert_equal(2, @pr.node[1].attributes.keys.size)
    assert_equal('foo', @pr.node[1].attributes[:ckid])
    assert_equal('ck:foo', @pr.node[1].attributes[:id])
    
    assert_equal(1, @pr.declarations.size)
    assert_equal([:foo], @pr.declarations.keys)
    assert_equal([{:id => 'ck:foo'}],@pr.declarations.values)
  end
  
  def test_empty_tag
    s = <<-EOF
    <a></a>
    EOF
    
    @pr.parse(s)
    
    assert_equal(1, @pr.node.size)
    assert_equal("<a/>\n", @pr.node[0].content)
  end
  
  def test_xmldec
    s = <<-EOF
    <?xml version="1.0" encoding="shift_jis"?>
    <html>
    </html>
    EOF
    
    @pr.parse(s)
    
    assert_equal(s.split("\n")[0].strip + '<html>', @pr.node[0].content.split("\n")[0].strip)
    assert_equal('shift_jis', @pr.encoding)
  end
  
  def test_encoding
    s = <<-EOF
    <?xml version='1.0' encoding='shift_jis'?>
    <html> قق e_lX</html>
    EOF
    
    assert_nothing_raised {
      @pr.parse(s)
    }
    
    assert_equal('shift_jis'.downcase, @pr.encoding)
    
    # compare Japanese String
    assert_equal(s.split("\n")[1].strip, @pr.node.node[0].content.split('?>')[1].strip)
  end
  
  def test_pi
    s = <<-EOF
    <?xsl hoge foo?>
    EOF
    
    @pr.parse(s)
    
    assert_equal(s.strip, @pr.node.node[0].content.strip)
  end
  
  def test_cdata
    s = <<-EOF
    <html><![CDATA[ <a><b><c><d><e><f>g</f></e></d></c></b></a> ]]></html>
    EOF
    
    @pr.parse(s)
    assert_equal(s.strip, @pr.node[0].content.strip)
  end
  
  def test_doctype
    s = <<-EOF
    <?xml version="1.0" encoding="euc-jp"?>
    <!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"><html><head> </head></html>
    EOF
    
    @pr.parse(s)
    
    s1, s2 = *s.split("\n", 2) 
    assert_equal(s1.strip, @pr.node[0].content.strip)
    assert_equal(s2.strip, @pr.node[1].content.strip)
  end
  
  def test_comment
    s = <<-EOF
    <!-- <span ckid='hoge'/> -->
    EOF
    
    @pr.parse(s)
    
    assert_equal(s.strip, @pr.node[0].content.strip)
  end
  
  def test_comment2
    s = <<-EOF
    <!--ck <span ckid='hoge'/> -->
    EOF
    
    @pr.parse(s)
    
    assert_equal(3, @pr.node.node.size)
    assert_equal('<!--', @pr.node.node[0].content)
    assert_equal(:hoge, @pr.node.node[1].ckid)
  end
  
  def test_to_s
    s = <<-EOF
    <html>
    <head><span ckid="hoge"/></head>
    <body>
    <div ckid="foo">
    hoge <span ckid="foo"> </span>
    </div>
    </body>
    </html>
    EOF
    
    @pr.parse(s)
    t = @pr.node.to_s
    
    a = s.split.collect{|l| l.strip}
    b = t.split.collect{|l| l.strip}
    
    assert_equal(a.size, b.size)
    (0..(a.size-1)).each do |i|
      assert_equal(a[i], b[i])
    end    
  end
end


