module CGIKit

# A response object that is sent to a browser by a Adapter object.
#
# == Getting a response object
# You can get a response object by Application#response or Component#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class Response < Message
  # Status code in HTTP. Default status is 200 ( OK ).
  attr_accessor :status

  attr_accessor :component

  STATUS = { 
    100 => 'Continue',
    101 => 'Switching Protocols',
    200 => 'OK',
    201 => 'Created',
    202 => 'Accepted',
    203 => 'Non-Authoritative Information',
    204 => 'No Content',
    205 => 'Reset Content',
    206 => 'Partial Content',
    300 => 'Multiple Choices',
    301 => 'Moved Permanently',
    302 => 'Found',
    303 => 'See Other',
    304 => 'Not Modified',
    305 => 'Use Proxy',
    307 => 'Temporary Redirect',
    400 => 'Bad Request',
    401 => 'Unauthorized',
    402 => 'Payment Required',
    403 => 'Forbidden',
    404 => 'Not Found',
    405 => 'Method Not Allowed',
    406 => 'Not Acceptable',
    407 => 'Proxy Authentication Required',
    408 => 'Request Timeout',
    409 => 'Conflict',
    410 => 'Gone',
    411 => 'Length Required',
    412 => 'Precondition Failed',
    413 => 'Request Entity Too Large',
    414 => 'Request-URI Too Long',
    415 => 'Unsupported Media Type',
    416 => 'Requested Range Not Satisfiable',
    417 => 'Expectation Failed',
    500 => 'Internal Server Error',
    501 => 'Not Implemented',
    502 => 'Bad Gateway',
    503 => 'Service Unavailable',
    504 => 'Gateway Timeout',
    505 => 'HTTP Version Not Supported'
  }

  def initialize( headers = nil )
    super
    @status = 200
    @headers['Content-Type'] = 'text/html'
  end

  def header
    response = ''
    response << ( __cookie || '' )
    response << ( _general_header  || '' )
    response << ( _response_header || '' )
    response << _entity_header
    response << EOL
    response
  end

  def to_s
    response = ''
    response << (self.content || '') unless redirect?
    response
  end
  private

  def _general_header
    header = ''
    header << ( __header('Cache-Control') || '' )
    header << ( __header('Connection') || '' )
    header << ( __header('Date') || '' )
    header << ( __header('Pragma') || '' )
    header << ( __header('Trailer') || '' )
    header << ( __header('Transfet-Encoding') || '' )
    header << ( __header('Upgrade') || '' )
    header << ( __header('Via') || '' )
    header << ( __header('Warning') || '' )
  end

  def _response_header
    header = ''
    header << ( __header('Accept-Ranges') || '' )
    header << ( __header('Age') || '' )
    header << ( __header('ETag') || '' )
    header << ( __header('Location') || '' )
    header << ( __header('Proxy-Authenticate') || '' )
    header << ( __header('Retry-After') || '' )
    header << ( __header('Server') || '' )
    header << ( __header('Vary') || '' )
    header << ( __header('WWW-Authenticate') || '' )
  end

  def _entity_header
    header = ''
    header << ( __header('Allow') || '' )
    header << ( __header('Content-Encoding') || '' )
    header << ( __header('Content-Language') || '' )
    header << ( __header('Content-Length') || '' )
    header << ( __header('Content-Location') || '' )
    header << ( __header('Content-MD5') || '' )
    header << ( __header('Content-Range') || '' )
    header << __content_type
    header << ( __header('Content-Disposition') || '' )
    header << ( __header('Expires') || '' )
    header << ( __header('Last-Modified') || '' )
  end

  def __header( header )
    "#{header}: #{self.headers[header]}#{EOL}" if self.headers[header]
  end

  def __content_type
    header = "Content-Type: #{self.headers['Content-Type']}"
    header << "; charset=#{self.encoding}" if self.encoding
    header << EOL
  end

  def __cookie
    return if @cookies.empty?

    header = ''
    @cookies.each { | cookie |
      header << "Set-Cookie: #{cookie.to_s}"
      header << EOL
    }
    header
  end

  public

  def status_line
    "#{http_version_line} #@status #{STATUS[@status]}#{EOL}"
  end

  # Sends a temporary redirect response to the client using the specified URL.
  def set_redirect( url )
    @status = 302
    @headers['Location'] = url
  end

  # Returns true if the response is setted redirect.
  def redirect?
    (@status == 302) or (@status == 307)
  end

  # Returns self. This method is invoked to display results in direct action.
  def generate_response
    self
  end
end

end
