module CGIKit

  class RequestHandler
    attr_reader :application

    def initialize( application )
      @application = application
    end


    #
    # parsing session IDs, context IDs and request handler key from URLs
    #

    # Returns a session ID parsed from the request.
    # If you customize URL format including session ID,
    # override the method by subclasses.
    def session_id( request ); end

    def session_id_from_cookie( request )
      sid = nil
      cookie = request.cookie(@application.session_key)
      if cookie then
        sid = cookie.value
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def session_id_from_url( request )
      sid = nil
      if path = request.request_handler_path then
        sid = id_by_separating(path, false)
      end
      sid
    end

    def id_by_separating( path, context = true )
      path = path.reverse.chop.reverse
      separated = path.split('/')
      separated.delete_at(0)
      id = nil
      case separated.size
      when 1
        if context then
          if separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        else
          if !separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        end
      when 2
        if context then
          id = separated.last
        else
          id = separated.first
        end
      end
      if id and !context and !Session.session_id?(id) then
        id = nil
      end
      id
    end

    # Returns a context ID parsed from the request.
    # If you customize URL format including session ID,
    # override the method by subclasses.
    def context_id( request ); end

    def request_handler_key( request )
      key = nil
      if info = request.request_handler_path then
        info = info.reverse.chop.reverse
        separated = info.split('/')
        key = separated.first
      end
      key
    end


    #
    # request-response loop
    #

    # Abstract method. Returns a response object.
    def handle_request( request )
      request.session_id = session_id(request)
      request.context_id = context_id(request)
    end

    def transaction( element, context, &block )
      element.begin_context(context)
      block.call
      element.end_context(context)
    end

    def take_values_from_request( element, request, context )
      transaction(element, context) do
        element.take_values_from_request(request, context)
      end
    end

    def invoke_action( element, request, context )
      result = nil
      transaction(element, context) do
        result = element.invoke_action(request, context)
      end

      if (Component === result) or (Response === result) then
        result
      else
        nil
      end
    end

    def append_to_response( element, response, context, result = nil )
      transaction(element, context) do
        element.append_to_response(response, context)
      end
    end


    #
    # generating URLs
    #

    def url( context, path, query, is_secure, port = nil ); end

    def application_path( request, is_secure = false, port = nil )
      protocol = nil
      if is_secure == true then
        protocol = 'https://'
      else
        protocol = 'http://'
      end

      domain = request.server_name || 'localhost'

      if port.nil? and request.server_port then
        port = request.server_port.to_i
      end
      if port == 80 then
        port = nil
      end

      script = (request.script_name || @application.path).dup
      script.sub!(/\A\//, '')

      if port then
        path = "#{protocol}#{domain}:#{port}/#{script}"
        path.gsub!(/\/\Z/, '')
      else
        path = "#{protocol}#{domain}/#{script}"
      end
      path
    end

    def query_string( hash = {} )
      str = ''
      keys = hash.keys.sort do |a, b|
        a.to_s <=> b.to_s
      end
      keys.each do |key|
        value = hash[key]
        if Array === value then
          value.each do |item|
            str << query_association(key, item)
          end
        else
          str << query_association(key, value)
        end
      end
      str.chop!
      str
    end

    def query_association( key, value )
      unless value.nil? then
        "#{key}=#{Utilities.escape_url(value.to_s)};"
      else
        "#{key};"
      end
    end
  end


  class ComponentRequestHandler < RequestHandler
    def handle_request( request )
      super
      context = @application.create_context(request)
      context.request_handler_key = @application.component_request_handler_key

      transaction(context.component, context) do
        @application.take_values_from_request(request, context)
      end

      result = nil
      context.delete_all
      transaction(context.component, context) do
        result = @application.invoke_action(request, context)
      end

      if Response === result then
        context.response = result
      else
        if (Component === result) and (context.component != result) then
          result.awake_from_restoration(context)
          context.component = result
        else
          result = context.component.root
        end
        context.delete_all
        transaction(result, context) do
          @application.append_to_response(context.response, context)
        end
        context.response.component = result
      end

      @application.save_session(context)
      context.response
    end

    def url( context, path = nil, query = {}, is_secure = false, port = nil )
      str = application_path(context.request, is_secure, port)
      str << "/#{@application.component_request_handler_key}"
      if context.session.store_in_url then
        str << "/#{context.session.session_id}"
      end
      str << "/#{context.context_id}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def session_id( request )
      sid_url = session_id_from_url(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_url then
        sid_url
      else
        nil
      end
    end

    def context_id( request )
      id = nil
      if path = request.request_handler_path then
        id = id_by_separating(path, true)
      end
      id
    end

  end


  class DirectActionRequestHandler < RequestHandler

    class ActionResultError < StandardError #:nodoc:
    end

    def session_id( request )
      sid_query = session_id_from_query(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_query then
        sid_query
      else
        nil
      end
    end

    def session_id_from_query( request )
      if sid = request[@application.direct_action_session_key] then
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def handle_request( request )
      super
      direct_action, action_name = direct_action_and_action_name(request)
      result = direct_action.perform_action(action_name)

      unless result.respond_to?(:generate_response) then
        raise ActionResultError, \
          "Direct action must return an object has generate_response()" +
          " and not nil. - #{direct_action.class}##{action_name}"
      end

      response = result.generate_response
      if Component === result then
        response.component = result
        @application.save_session(result.context)
      end
      response
    end

    def direct_action_and_action_name( request )
      klass, action_name = direct_action_class_and_action_name(request)

      direct_action = klass.new(@application, request)
      if action_name.nil? then
        action_name = direct_action.default_action_name
      end
      if !direct_action.direct_action?(action_name) then
        klass = @application.direct_action_class
        direct_action = klass.new(@application, request)
        action_name = direct_action.default_action_name
      end
      [direct_action, action_name]
    end

    def direct_action_class_and_action_name( request )
      klass = @application.direct_action_class
      action_name = nil
      if path = request.request_handler_path then
        path = path.dup
        path.gsub!(/\A\//, '')
        path.gsub!(/\?(.*)/, '')
        key, class_name, action_name = path.split('/')
        begin
          klass = Object
          class_name.split('::').each do |name|
            klass = klass.const_get(name)
          end
        rescue Exception => e
          klass = @application.direct_action_class
          unless action_name then
            action_name = class_name
          end
        end
      end
      unless klass <= DirectAction then
        klass = @application.direct_action_class
        action_name = nil
      end
      [klass, action_name]
    end

    def url( context, path = nil, query = {}, is_secure = false, port = 80 )
      str = application_path(context.request, is_secure, port)
      str << "/#{@application.direct_action_request_handler_key}"
      str << "/#{path}" if path
      if context.has_session? and context.session.store_in_url then
        query[@application.direct_action_session_key] = context.session.session_id
      end
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def direct_action_url( context, action_class, action_name, query )
      action_class = @application.direct_action_class unless action_class
      key = @application.direct_action_request_handler_key
      path = direct_action_path(action_class, action_name)
      url(context, path, query, false, nil)
    end

    def direct_action_path( action_class, action_name )
      if action_name == DirectAction::DEFAULT_ACTION_NAME then
        action_name = ''
      end
      if @application.direct_action_class.to_s == action_class.to_s then
        action_name
      elsif action_name.empty?
        action_class.to_s
      else
        "#{action_class}/#{action_name}"
      end
    end

  end


  class ResourceRequestHandler < RequestHandler

    RESOURCE_KEY = 'data'

    def handle_request( request )
      super
      response = Response.new
      key = request.form_value(RESOURCE_KEY)
      if data = @application.resource_manager.bytedata(key) then
        response.headers['Content-Type'] = data.content_type
        response.content = data.to_s
        @application.resource_manager.remove_data(key)
      else
        response.status = 404
      end
      response
    end

    def resource_url( name, request )
      str = application_path(request)
      str << "/#{@application.resource_request_handler_key}"
      str << "?#{RESOURCE_KEY}=#{name}"
      str
    end

  end

end
