module CGIKit

class Element
  class UnknownElementError < StandardError; end #:nodoc:
  class AttributeError      < StandardError; end #:nodoc:

  @@apis = {}

  attr_accessor :name, :node, :associations

  class << self
    include APIUtilities

    def api
      unless @@apis[self] then
        @@apis[self] = create_api
      end
      @@apis[self]
    end

    # Implemented by subclasses. Returns a new API object for the element.
    def create_api; end
  end

  def api
    self.class.api
  end

  def begin_context( context )
    context.increment
  end

  def end_context( context ); end

  def take_values_from_request( request, context ); end
  def invoke_action( request, context ); end
  def append_to_response( response, context ); end

  def empty?
    @node.nil? or @node.empty?
  end
end


# The super class of dynamic element classes.
# Dynamic elements convert themselves to HTML.
# These are very important in development with CGIKit.
#
# == Paths for searching elements and components
# Element objects, including components, are usually instantiated
# by Element.instance. This method loads elements/components and creates
# element objects. In this method, some paths are searched to require files.
#
# The searched paths:
# 1. CKApplicaton#component_path
# 2. ($LOAD_PATH)/cgikit/elements
# 3. ($LOAD_PATH)/cgikit/components
#
# The latter two paths are for extention elements or components.
# It is recommended that your own elements or components are installed
# in the 1st path.
class DynamicElement < Element

  attr_accessor :name, :root, :values, :context_id, :application, :parent

  class << self
    def keys_to_delete_from_associations
      []
    end
  end

  def initialize( name, associations, root )
    super()
    @name = name
    @associations = associations
    @root = root
    @values = {}
    @once = {}
    @application = @root.application
    self.class.keys_to_delete_from_associations.each do |key|
      @associations.delete(key)
    end
    init
  end


  #
  # request-response loop
  #

  def take_values_from_request( request, context )
    @node.take_values_from_request(request, context) if @node
  end

  def invoke_action( request, context )
    @node.invoke_action(request, context) if @node
  end

  def append_to_response( response, context ); end

  # Returns value from the request if the context is continued.
  # If not be continued, returns nil.
  def value_from_request( request, context )
    values_from_request(request, context)[0]
  end

  def values_from_request( request, context )
    values = nil
    if (values = request.form_values[context.context_id]) and \
      (values.empty? == false) then
    elsif @values[:name] and context.in_form? and \
      (values = request.form_values[@values[:name].to_s]) and \
      (values.empty? == false) then
    end
    values ||= []

    if !(ByteData === values) and (values.empty? == false) and \
      (encoding = application.encoding) then
      values = encode_strings(values, encoding)
    end
    values
  end

  def encode_strings( strings, encoding )
    encoded = []
    strings.each do |string|
      begin
        encoded << (encode_string(string, encoding) || string)
      rescue Exception => e
        encoded << (failed_encode_string(string, encoding, e) || string)
      end
    end
    encoded
  end


  #
  # take values for binding keys
  #

  def take_value_once( name, does_resolve = true )
    unless @once[name] then
      @values[name] = value(name, does_resolve)
      @once[name] = true
    end
  end

  def take_value( name, does_resolve = true )
    @values[name] = value(name, does_resolve)
  end

  def take_bool( name, does_resolve = true )
    @values[name] = bool(name, does_resolve)
  end

  def value( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.value(root)
      else
        as.value
      end
    else
      if api and api[name] then
        api[name].default(root)
      else
        nil
      end
    end
  end

  def bool( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.bool(root)
      else
        as.bool
      end
    else
      if api and api[name] then
        Association.adapt_to_bool(api[name].default(root))
      else
        nil
      end
    end
  end

  def set_value( name, value )
    if as = @associations[name] then
      as.set_value(value, root)
    end
  end

  def name_value( context )
    escaped_string(@values[:name] || context.context_id)
  end

  def declared?( name )
    @associations.key?(name)
  end

  def direct_action?
    declared?(:direct_action) or declared?(:action_class)
  end


  #
  # escaping
  #

  def escaped_string( string, escape = true )
    if escape then
      string = Utilities.escape_html(string.to_s)
    end
    string
  end


  #
  # creating attributes' string
  #

  def other
    api.other(root, associations)
  end

  def enabled
    api.enabled(root, associations)
  end

  def checked
    " checked=\"checked\""
  end

  def selected
    " selected=\"selected\""
  end


  #
  # validating
  #

  def validate( name )
    take_value(name)
    take_value(Declaration::VALIDATE_KEY, false)
    take_value(Declaration::PASS_KEY, false)
    method = @values[Declaration::VALIDATE_KEY]
    pass = @values[Declaration::PASS_KEY]
    if method and pass then
      result = root.__send__(method, @values[name])
      set_value(Declaration::PASS_KEY, result)
    end
  end


  #
  # hook
  #

  def init; end

  # Convert character code for the form values.
  def encode_string( string, encoding ); end

  # Invoked when encode_string() raised an error.
  # This method should be return string.
  def failed_encode_string( string, encoding, error ); end

end

end


class Array

  def take_values_from_request( request, context )
    each do |item|
      item.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    result = nil
    each do |item|
      if action_result = item.invoke_action(request, context) then
        result = action_result
      end
    end
    result
  end
  
  def append_to_response( response, context )
    each do |item|
      item.append_to_response(response, context)
    end
  end

end

