module CGIKit

  VERSION = '2.0.0'

  class CGIKitError < StandardError
  end


  class Application

    include KeyValueCoding

    class SessionCreationError < StandardError #:nodoc:
    end
    class SessionRestorationError < StandardError #:nodoc:
    end
    class SessionAuthorizationError < SessionRestorationError #:nodoc:
    end
    class SessionTimeoutError < SessionRestorationError #:nodoc:
    end
    class PageRestorationError < StandardError #:nodoc:
    end

    # Main component. If session ID or context ID aren't specified,
    # this component is shown. The default value is MainPage.
    attr_accessor :main

    # Locale of an application in a transaction. If the locale is specified,
    # CGIKit change a template for a component. The name of the template
    # includes the component name and the locale name. Also, the template name
    # is seprated by underscore("_"). For example, if the locale is "ja" and
    # the component name is "MainPage", the template name is "MainPage_ja.html".
    attr_accessor :locale

    # Main locale of an application. If the value is equal to
    # locale specified in request, components use templates whose name
    # doesn't include the locale name. If the value is
    # "ja", requested locale is "ja" and the component name is "MainPage",
    # the application uses templated named "MainPage.html".
    attr_accessor :master_locale

    # Document root directory.
    attr_accessor :document_root

    # The application URL based on SCRIPT_NAME.
    attr_accessor :baseurl

    # The file system path of the application.
    attr_accessor :path

    # The file system paths for components. Components are searched under it.
    attr_accessor :component_paths

    # Resource directory.
    # This directory includes files to be used by the application,
    attr_accessor :resources

    # Web server resources directory.
    # This directory includes files to be displayed to browser.
    # The files are used by Image element, etc.
    attr_accessor :web_server_resources

    # ResourceManager object.
    attr_accessor :resource_manager

    # Adapter object.
    attr_accessor :adapter

    # Adapter class. The default value is CGI class.
    attr_accessor :adapter_class

    # Name or class of an error page component to show caught errors.
    attr_accessor :error_page

    # Temporary directory to be used by the framework.
    # The framework uses this to store sessions and template caches.
    attr_accessor :tmpdir

    # Session key. This key is used in cookie.
    attr_accessor :session_key

    # Session key in direct action.
    # This key is used in hidden fields of form and URL when using direct action.
    attr_accessor :direct_action_session_key

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Expiry date of cookie for session. If you set the value to nil,
    # session cookies will be invalid when closing browser.
    attr_accessor :session_cookie_expires

    # Enables or disables the use of URLs for storing session IDs.
    attr_accessor :store_in_url

    # Enables or disables the use of cookies for storing session IDs.
    attr_accessor :store_in_cookie

    # Enables or disables session authorization by browsers.
    # If you set the value to true, the application raises error
    # when an user accesses it with browser that is different from
    # one registered session.
    attr_accessor :auth_by_user_agent

    # Enables or disables session authorization by IP addresses.
    # If you set the value to true, the application raises error
    # when an user accesses it with IP address that is different from
    # one registered session.
    attr_accessor :auth_by_remote_addr

    # Encoding to encode character code of form data.
    # The default implementation uses Kconv to encode Japanese character codes.
    # Then specify constant values of Kconv; Kconv::JIS, Kconv::SJIS, Kconv::EUC, etc.
    # If the value is nil, form data is not encoded. The default value is nil.
    attr_accessor :encoding

    # Request handler key to display components.
    attr_accessor :component_request_handler_key

    # Request handler key to invoke direct actions.
    attr_accessor :direct_action_request_handler_key

    # Request handler key to display resource files.
    attr_accessor :resource_request_handler_key

    # Default request handler key. Default key is component request handler key.
    attr_accessor :default_request_handler

    # Request handler for components.
    attr_accessor :component_request_handler

    # Request handler for direct actions.
    attr_accessor :direct_action_request_handler

    # Request handler for resources.
    attr_accessor :resource_request_handler

    # Session class. Default class is CGIKit::Session.
    attr_accessor :session_class

    # Whether or not validates setting of attributes for each elements.
    # If wrong attribute name or combination are found, raises error.
    attr_accessor :validate_api
    alias validate_api? validate_api

    # HTML parser class. Default class is CGIKit::HTMLParser::HTMLParser.
    attr_accessor :htmlparser_class

    # Size to cache components permanently in session.
    # Newly generated page is cached automatically.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :page_cache_size

    # Size to cache components permanently in session.
    # Permanent page cache is cached optionally, not automatically caching.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :permanent_page_cache_size

    # Direct action class. Default class is CGIKit::DirectAction.
    attr_accessor :direct_action_class

    # Session store class. Default class is CGIKit::FileSessionStore.
    attr_accessor :session_store_class

    # Context class. Default class is CGIKit::Context.
    attr_accessor :context_class

    # Whether or not caches templates to reduce parsing load.
    attr_accessor :cache_template

    attr_accessor :template_store
    attr_accessor :template_store_class
    attr_accessor :sweep_password

    COMPONENT_REQUEST_HANDLER_KEY     = 'c'
    DIRECT_ACTION_REQUEST_HANDLER_KEY = 'd'
    RESOURCE_REQUEST_HANDLER_KEY      = 'r'
    CGIKIT_LIB                        = 'cgikit'
    COMPONENT_LIB                     = 'components'

    def initialize
      init_attributes
      init_request_handlers
      init_adapter
      init_component_paths
      init_name_spaces
      init
    end

    def init_attributes
      require 'cgikit/components/CKErrorPage/CKErrorPage'
      @main                      = 'MainPage'
      @error_page                = CKErrorPage
      @tmpdir                    = './tmp' || ENV['TMP'] || ENV['TEMP']
      @session_key               = '_session_id'
      @direct_action_session_key = '_sid'
      @manage_session            = false
      @timeout                   = 60 * 60 * 24 * 7
      @session_cookie_expires    = 60 * 60 * 24 * 7
      @store_in_url              = true
      @store_in_cookie           = false
      @auth_by_user_agent        = false
      @auth_by_remote_addr       = false
      @session_class             = Session
      @session_store             = nil
      @session_store_class       = FileSessionStore
      @template_store            = nil
      @template_store_class      = FileTemplateStore
      @encoding                  = nil
      @resources                 = './'
      @validate_api              = true
      @cache_template            = true
      @htmlparser_class          = HTMLParser::HTMLParser
      @page_cache_size           = 30
      @permanent_page_cache_size = 30
      @direct_action_class       = DirectAction
      @context_class             = Context
      @baseurl                   = nil
      @locale                    = nil
      @master_locale             = 'en'
    end

    def init_request_handlers
      @component_request_handler_key     = COMPONENT_REQUEST_HANDLER_KEY
      @direct_action_request_handler_key = DIRECT_ACTION_REQUEST_HANDLER_KEY
      @resource_request_handler_key      = RESOURCE_REQUEST_HANDLER_KEY
      @component_request_handler         = ComponentRequestHandler.new(self)
      @direct_action_request_handler     = DirectActionRequestHandler.new(self)
      @resource_request_handler          = ResourceRequestHandler.new(self)
      @default_request_handler           = @component_request_handler
    end

    def init_adapter
      # decides interface of adapter
      if defined?(MOD_RUBY) then
        @adapter_class = Adapter::ModRuby
        @path = Apache.request.filename
      else
        @adapter_class = Adapter::CGI
        @path = $0
      end
    end

    def init_component_paths
      @component_paths = [Dir.pwd]
      $LOAD_PATH.each do |path|
        @component_paths << File.join(path, CGIKIT_LIB, COMPONENT_LIB)
      end
    end

    def init_name_spaces
      @name_spaces = [CGIKit, Object]
      klass = Object
      self.class.to_s.split('::').each do |class_name|
        klass = klass.const_get(class_name)
        @name_spaces << klass
      end
    end

    # Returns the name of the application without file extension.
    def name
      File.basename( @path, '.*' )
    end

    def take_values_from_hash( hash )
      hash.each do |key, value|
        self[key] = value
      end
    end

    def template_store
      @template_store ||= @template_store_class.new(self)
    end
    
    
    
    def load_all_components( path, subdir = false )
      __each_component_file(path, subdir) do |file|
        require(file)
      end
    end

    def autoload_all_components( path, mod = Object, subdir = false )
      __each_component_file(path, subdir) do |file|
        class_name = File.basename(file, '.rb')
        mod.autoload(class_name, file)
      end
    end

    private

    def __each_component_file(path, subdir)
      if subdir
        pattern = File.join(path, '**', '*.rb')
      else
        pattern = File.join(path, '*.rb')
      end
      
      Dir.glob(pattern) do |file|
        if /\.rb$/ === file and FileTest::readable?(file)
          yield file
        end
      end
    end
    
    public

    def load_configuration( path )
      load(path)
      configure
    end


    #
    # hook
    #

    def init; end

    def configure; end


    #
    # managing sessions
    #

    # Session database object (SessionStore).
    def session_store
      @session_store ||= @session_store_class.new(self)
    end

    # Creates a session.
    def create_session( request )
      begin
        klass = @session_class || Session
        session = klass.new(request.session_id)
        session.awake_from_restoration(self)
        return session
      rescue
        if sid = request.session_id then
          msg = "for #{sid}"
        else
          msg = "because session id is not specified"
        end 
        raise SessionCreationError, "Failed creating session #{msg}."
      end
    end

    # Returns a restored session objects with session ID.
    def restore_session( session_id, context )
      begin
        session = session_store.checkout(session_id, context.request)
      rescue Exception => e
        raise SessionRestorationError, "Failed restoring session for #{session_id}"
      end
      if session then
        context.session = session
        context.session.awake_from_restoration(self)
        context.session.validate
      end
      session
    end

    # Saves the session, and set a cookie if "store_in_cookie" attribute is
    # setted. If "clear" method of the session is called, the session is deleted.
    def save_session( context )
      unless context.has_session? then return end

      if context.session.terminate? then
        sid = context.session.session_id
        @session_store.remove(sid)
      else
        session_store.checkin(context)
        if context.session.store_in_cookie? then
          context.session.set_cookie
        end
      end
    end


    #
    # handling requests
    #

    # Runs the application.
    def run( command_or_request = nil, response = nil )
      if CGIKit.const_defined?(:Command) and \
        CGIKit.const_get(:Command) === command_or_request then
        request = command_or_request.request
      else
        request = command_or_request
      end

      set_adapter_and_handler
      @adapter.run(request, response) do |request|
        set_attributes_from_request(request)
        request.request_handler_key = \
        @default_request_handler.request_handler_key(request)

        begin  
          handler = request_handler(request.request_handler_key)
          request.context_id = handler.context_id(request)
          response = handler.handle_request(request)

        rescue Exception => e
          # create context without session
          request.session_id = nil
          context = @context_class.new(request, self)
          begin
            response = handle_error(e, context)
          rescue Exception => e
            # trap error occured by customized handle_error 
            response = default_error_page(e, context)
          end
        end

        response
      end
      response
    end

    def request_handler( key )
      unless handler = @request_handlers[key] then
        handler = @default_request_handler
      end
      handler
    end

    def set_adapter_and_handler
      if @set_adapter_and_handler then return end
      @adapter = create_adapter
      @set_adapter_and_handler = true
      @request_handlers = {
        @component_request_handler_key     => @component_request_handler,
        @direct_action_request_handler_key => @direct_action_request_handler,
        @resource_request_handler_key      => @resource_request_handler }
    end

    def set_attributes_from_request( request )
      if @set_attributes_from_request then return end
      @baseurl       = request.script_name              unless @baseurl
      @document_root = request.headers['DOCUMENT_ROOT'] unless @document_root
      @web_server_resources ||= @document_root
      @resource_manager       = ResourceManager.new(self)
      @set_attributes_from_request = true
    end

    def take_values_from_request( request, context )
      context.session.take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.session.invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.session.append_to_response(response, context)
    end

    def create_context( request )
      handler = request_handler(request.request_handler_key)
      context = @context_class.new(request, self)
      
      if session = restore_session(request.session_id, context) then
        context.session = session
      else
        session = create_session(request)
        context.session = session
      end
      session.context = context
      
      if component = context.session.restore_page(context.sender_id) then
        root = component.root
        context.component = root
        root.awake_from_restoration(context)
      else
        context.component = page(@main, context)
      end
      context.delete_all
      context
    end

    # Creates an adapter object.
    def create_adapter
      @adapter_class.new
    end


    #
    # handling errors
    #

    # Handles every errors and return an error page component.
    def handle_error( error, context )
      case error
      when SessionCreationError 
        response = handle_session_creation_error(error, context)
      when SessionRestorationError 
        response = handle_session_restoration_error(error, context)
      when PageRestorationError 
        response = handle_page_restoration_error(error, context)
      else
        page = default_error_page(error, context)
        response = page.generate_response
      end
      response
    end

    def handle_session_creation_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_session_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_page_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    # Return a default error page component.
    def default_error_page( error, context )
      error_page       = page('CKErrorPage', context)
      error_page.error = error
      error_page
    end

    def url( key, context, path, query_string, is_secure, port = nil )
      request_handler(key).url(context, path, query_string, is_secure, port)
    end

    def direct_action_url( context, action_class, action_name, query )
      handler = @direct_action_request_handler
      handler.direct_action_url(context, action_class, action_name, query)
    end


    #
    # Creating components
    #

    # Creates a specified page component.
    def page( name, request_or_context, *args )
      context = request_or_context
      if Request === request_or_context then
        context = @context_class.new(request_or_context, self)
      end
      if Class === name then
        klass = name
      else
        klass = class_named(name)
      end
      klass.new(context, *args)
    end

    def class_named( name )
      Utilities.class_named_from(name, @name_spaces)
    end

  end

end


