module CGIKit

# KeyValueCoding provides methods to access object graph.
# The methods try accessing instance variables by using accessor method.
# If it is failure, its try accessing directly.
module KeyValueCoding

  # The method returns "true", the direct access is success. Or failure.
  # Default value is true.
  attr_accessor :access_attributes
  alias access_attributes? access_attributes

  @access_attributes = true

  # Retrieves value for the key.
  def value_for_key( __key )
    # escape sharing local variable scope of instance_eval by __key.
    instance_eval(__key.to_s.untaint)
  end

  def stored_value_for_key( key )
    instance_variable_get("@#{key}")
  end

  # Takes value for the key.
  def take_value_for_key( __key, __value )
    writer = "self.#{__key.to_s}=__value"
    instance_eval(writer.untaint)
  end

  def take_stored_value_for_key( key, value )
    instance_variable_set("@#{key}", value)
  end

  alias value_for_keypath value_for_key
  alias take_value_for_keypath take_value_for_key

  # used only accessing instance variables
  def method_missing( name, *args )
    if access_attributes? then
      if args.empty? then
        return stored_value_for_key(name.to_s)
      elsif (name.to_s =~ /([^=]+)=$/) and (args.size == 1) then
        return take_stored_value_for_key($1, args[0])
      end
    end
    super
  end

end

end
module CGIKit

# Classes under Adapter are interface between CGIKit and web servers.
# 
# The role of adapter is to provide the consistent interface to 
# Application. The difference of the relationship between 
# CGIKit and a web server is aboserbed by the obejct. 
module Adapter
  class CGI
    attr_accessor :headers, :params
    attr_reader :input, :output, :error

    CR  = "\r"
    LF  = "\n"
    EOL = CR + LF

    def initialize
      @headers = ENV
      @input   = $stdin
      @output  = $stdout
      @error   = $stderr
      @params = Utilities.query_from_headers(@headers, @input)
    end

    def create_request
      Request.new(headers, params)
    end

    def create_response
      Response.new(headers)
    end

    def run( request, response, &block )
      request ||= create_request

      if block_given?
        response = block.call(request)
      end

      print response.header
      print response.to_s
    end

  end


  class ModRuby < CGI
    def run( request, response, &block )
      request ||= create_request

      if block_given?
        response = block.call(request)
      end

      # import from cgi.rb
      r = Apache.request
      table = r.headers_out
      response.header.scan(/([^:]+): (.+)#{EOL}/n){ |name, value|
        case name
        when 'Set-Cookie'
          table.add(name, value)
        when /^status$/ni
          r.status_line = value
          r.status = value.to_i
        when /^content-type$/ni
          r.content_type = value
        when /^content-encoding$/ni
          r.content_encoding = value
        when /^location$/ni
          if r.status == 200
            r.status = 302
          end
          r.headers_out[name] = value
        else
          r.headers_out[name] = value
        end
      }
      r.send_http_header

      print response.to_s
    end
  end


  class Template < CGI
    def initialize
      require 'stringio'
      @headers = ENV
      @input   = StringIO.new
      @output  = StringIO.new
      @error   = StringIO.new
      @params  = Hash.new([])
    end

    def run( request, response, &block )
      request ||= create_request

      if block_given?
        response = block.call(request)
      end

      response
    end
  end

end

end



require 'digest/md5'

module CGIKit

# A class for session management. Session objects have a hash of
# arbitary objects and information about browser name, IP address, etc.
# However, you can't set objects that can't be marshal ( IO, Proc, etc. )
# to the session with default database manager FileSessionStore.
class Session

  DEFAULT_SESSION_ID_FIGURES = 16
  DEFAULT_TIMEOUT = 60 * 60 * 24 * 7

  class << self
    def session_id?( session_id )
      session_id and (/\A([0-9A-Za-z]+)\Z/ === session_id)
    end

    def create_session_id
      md5 = Digest::MD5::new
      md5.update Time.now.to_s
      md5.update rand(0).to_s
      md5.update $$.to_s
      md5.hexdigest[0, session_id_figures]
    end

    def session_id_figures
      DEFAULT_SESSION_ID_FIGURES
    end
  end


  # A hash of arbitary objects.
  attr_accessor :values

  # Session ID.
  attr_accessor :session_id

  # Seconds until the session has timed out.
  attr_accessor :timeout

  # Name of browser.
  attr_accessor :user_agent

  # IP address.
  attr_accessor :remote_addr

  # Enables or disables the use of URLs for storing session IDs.
  attr_accessor :store_in_url
  alias store_in_url? store_in_url

  # Enables or disables the use of cookies for storing session IDs.
  attr_accessor :store_in_cookie
  alias store_in_cookie? store_in_cookie

  # Enables or disables session authorization by browsers.
  attr_accessor :auth_by_user_agent
  alias auth_by_user_agent? auth_by_user_agent

  # Enables or disables session authorization by IP addresses.
  attr_accessor :auth_by_remote_addr
  alias auth_by_remote_addr? auth_by_remote_addr

  attr_accessor :context, :application, :session_store, :last_accessed_time, :caches, :context_ids, :frame_components, :permanent_caches, :session_key, :cookie_expires


  def initialize( session_id = nil )
    unless Session.session_id? session_id then
      session_id = Session.create_session_id
    end
    @session_id         = session_id
    @last_accessed_time = Time.new
    @context_ids        = {}
    @caches             = {}
    @permanent_caches   = {}
    @values             = {}
    @frame_components   = {}
    @user_agent         = nil
    @remote_addr        = nil
    @timeout            = DEFAULT_TIMEOUT
    @terminate          = false
    @last_component_id  = -1
    init
  end

  #
  # hook
  
  def init; end


  #
  # accessing
  #

  def []( key )
    @values[key]
  end

  def []=( key, value )
    @values[key] = value
  end

  def remove( key )
    @values.delete(key)
  end

  def terminate
    @terminate = true
  end

  def terminate?
    @terminate or timeout?
  end


  #
  # accessing cached components
  #

  def add_component( component, context = nil, permanently = false )
    unless component_id = component_id(component) then
      component_id = next_component_id()
      if context then
        context_id = context.context_id
      else
        context_id = "#{component_id}.0"
      end
      add_component_for_ids(component, component_id, \
        context_id, permanently)
    end
    component_id
  end

  def next_component_id
    @last_component_id += 1
  end

  def add_component_for_ids( component,
                             component_id,
                             context_id,
                             permanently = false )
    if permanently then
      page_caches = @permanent_caches
    else
      page_caches = @caches
    end
    page_caches[component_id] = component
    @context_ids[context_id] = component_id
  end

  def component_id( component )
    if @caches.value?(component) then
      @caches.index(component)
    elsif @permanent_caches.value?(component) then
      @permanent_caches.index(component)
    else
      nil
    end
  end

  def component_for_component_id( component_id )
    @caches[component_id] || @permanent_caches[component_id]
  end

  def component_for_context_id( context_id )
    page = nil
    unless context_id.nil? then
      ids = context_id.split(Context::SEPARATOR)
      while ids.size > 1
        id = ids.join(Context::SEPARATOR)
        if page = _component(id) then break end
        ids.pop
        ids.push(0)
        id = ids.join(Context::SEPARATOR)
        if page = _component(id) then break end
        ids.pop
      end
    end
    page
  end

  alias component component_for_context_id

  private

  def _component( context_id )
    if id = @context_ids[context_id] then
      @caches[id] || @permanent_caches[id]
    end
  end

  public

  #
  # testing
  #

  # Returns true if the browser is equal to one when the session created.
  def user_agent?( user_agent )
    @auth_by_user_agent and (@user_agent == user_agent)
  end

  # Returns true if the IP address is equal to one when the session created.
  def remote_addr?( remote_addr )
    @auth_by_remote_addr and (@remote_addr == remote_addr)
  end

  # Returns true if the session isn't expired.
  def timeout?
    if @timeout then
      (@timeout != 0) and (timeout_expire <= Time.new)
    else
      false
    end
  end

  def timeout_expire
    @last_accessed_time + @timeout
  end


  #
  # Handling requests
  #


  def take_values_from_request( request, context )
    context.component.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    context.component.invoke_action(request, context)
  end

  def append_to_response( response, context )
    context.component.append_to_response(response, context)
  end


  #
  # Page management
  #

  def save_page( component, permanently = false )
    if permanently then
      caches = @permanent_caches
      size = @application.permanent_page_cache_size
    else
      caches = @caches
      size = @application.page_cache_size
    end
    component.add_all_components(self, permanently)
    if component.context then
      cid = component.context.context_id
      @context_ids[cid] = component_id(component)
    end
    _remove_old_cache(caches, size)
  end

  private

  def _remove_old_cache( caches, size )
    removes = caches.size - size
    if removes > 0 then
      if removed_keys = caches.keys.sort.slice!(0..removes-1) then
        caches.delete_if do |key, value|
          if removed_keys.include?(key) then
            true
          end
        end
      end
    end
  end

  public

  def restore_page( context_id )
    component(context_id)
  end


  #
  # marshaling
  #

  def marshal_dump
    dump                      = {}
    dump[:timeout]            = @timeout
    dump[:values]             = @values
    dump[:session_id]         = @session_id
    dump[:user_agent]         = @user_agent
    dump[:remote_addr]        = @remote_addr
    dump[:caches]             = @caches
    dump[:permanent_caches]   = @permanent_caches
    dump[:context_ids]        = @context_ids
    dump[:frame_components]   = @frame_components
    dump[:last_accessed_time] = @last_accessed_time
    dump[:last_component_id]  = @last_component_id
    dump[:cookie_expires]     = @cookie_expires
    dump[:store_in_url]       = @store_in_url
    dump[:store_in_cookie]    = @store_in_cookie
    dump
  end

  def marshal_load( object )
    @timeout            = object[:timeout]
    @values             = object[:values]
    @session_id         = object[:session_id]
    @user_agent         = object[:user_agent]
    @remote_addr        = object[:remote_addr]
    @context_ids        = object[:context_ids]
    @caches             = object[:caches]
    @permanent_caches   = object[:permanent_caches]
    @frame_components   = object[:frame_components]
    @last_accessed_time = object[:last_accessed_time]
    @last_component_id  = object[:last_component_id]
    @cookie_expires     = object[:cookie_expires]
    @store_in_url       = object[:store_in_url]
    @store_in_cookie    = object[:store_in_cookie]
  end

  def awake_from_restoration( application )
    @application         = application
    @timeout             = application.timeout
    @cookie_expires      = application.session_cookie_expires
    @session_key         = application.session_key
    @session_store       = application.session_store
    @auth_by_remote_addr = application.auth_by_remote_addr
    @auth_by_user_agent  = application.auth_by_user_agent
    @store_in_url        = application.store_in_url
    @store_in_cookie     = application.store_in_cookie
    @last_accessed_time  = Time.new
  end


  #
  # validating
  #

  def validate
    validate_authorization
    validate_timeout
  end

  def validate_authorization
    if (@auth_by_user_agent and \
        !user_agent?(@context.request.user_agent)) or \
       (@auth_by_remote_addr and \
        !remote_addr?(@context.request.remote_addr)) then
      raise Application::SessionAuthorizationError, 'Your session is not authorized.'
    end
  end

  def validate_timeout
    if timeout? then
      flush
      raise Application::SessionTimeoutError, 'Your session has timed out.'
    end
  end

  def flush
    @session_store.remove(@session_id)
    if @store_in_cookie then
      remove_cookie(@context.response)
    end
  end


  #
  # managing cookie
  #

  def remove_cookie
    cookie         = Cookie.new(@session_key)
    cookie.expires = Time.new - 600
    @context.response.add_cookie(cookie)
  end

  def set_cookie
    cookie = Cookie.new(@session_key, @session_id)
    cookie.expires = Time.new + @cookie_expires
    @context.response.add_cookie(cookie)
  end

end

end
module CGIKit

# SessionStore is a class for saving session into storage.
class SessionStore
  def initialize( application )
    @application = application
  end

  def checkin( context )
    save(context)
  end

  def checkout( session_id, request )
    if session_id.nil? then
      return nil
    end
    restore(session_id, request)
  end

  # abstract
  def save( context ); end

  # abstract.
  def restore( session_id, request ); end

  # abstract.
  def remove( session_id ); end

  def sweep_sessions; end

end


# A FileSessionStore object saves a session to a file with marshaling.
class FileSessionStore < SessionStore
  TMPDIR = 'session'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end

  def save( context )
    unless FileTest.directory? @tmpdir
      require 'ftools'
      File.makedirs @tmpdir
    end
    FileLock.exclusive_lock(tmpfile(context.session.session_id)) do |file|
      Marshal.dump(context.session, file)
    end
  end

  def restore( session_id, request )
    session = nil
    if exist?(session_id) then
      FileLock.shared_lock(tmpfile(session_id)) do | file |
        session = Marshal.load file
      end
      session.session_store = self
      session.application = @application
    end
    session
  end

  def tmpfile( session_id )
    File.join(@tmpdir, session_id).untaint
  end

  def exist?( session_id )
    FileTest.exist?(tmpfile(session_id))
  end

  def remove( session_id )
    if FileTest.exist?(tmpfile(session_id))
      File.delete(tmpfile(session_id))
    end
  end

  def sweep_sessions
    successed = 0
    failed = 0
    Dir.foreach(@tmpdir) do |file|
      begin
        unless /\A\./ === file
          path = File.join(@tmpdir, file)
          session = nil
          FileLock.shared_lock(path) do |f|
            session = Marshal.load(f)
          end
          if session.timeout? then
            begin
              File.delete(path)
              successed += 1
            rescue
              failed += 1
            end
          end
        end
      rescue
      end
    end
    [successed, failed]
  end

end


class MemorySessionStore < SessionStore
  def initialize( application )
    super
    @caches = {}
  end

  def save( context )
    @caches[context.session.session_id] = context.session
  end

  def restore( session_id, request )
    if session = @caches[session_id] then
      session.session_store = self
      session.application = @application
    end
    session
  end

  def remove( session_id )
    @caches.delete(session_id)
  end

  def exist?( session_id )
    @caches.key?(session_id)
  end

  def sweep_sessions
    before = @caches.size
    @caches.delete_if do |id, session|
      session.timeout?
    end
    [before - @caches.size, 0]
  end

end

end
module CGIKit

class Template

  attr_accessor :template_store, :component, :template_node, :declaration_store, \
    :template_string, :declaration_hash, :template_mtime, :declaration_mtime

  def initialize
    @template_mtime = Time.now
    @declaration_mtime = Time.now
  end

  def terminate?
    @template_store.terminate?(@component)
  end

  def template_string=( string )
    @template_string = string
    @template_mtime = Time.now
  end

  def declaration_store=( store )
    @declaration_store = store
    @declaration_mtime = Time.now
  end

  def marshal_dump
    not_includes = ['@component', '@template_store']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  def cache_copy
    copy = dup
    copy.template_node = @template_node.cache_copy
    copy
  end
end


class TemplateStore

  def initialize( application )
    @application = application
  end

  def cache?
    @application.cache_template
  end

  def checkin( template )
    save(template)
  end

  def checkout( component )
    template = restore(component)
    if template.nil? or (cache? and terminate?(template, component)) then
      template = create_template(component)
      isnew = true
      checkin(template) if cache?
    end
    template.component = component
    template.template_store = self
    template.template_node.component = component
    template
  end

  def terminate?( template, component )
    terminate_template?(template, component) or \
      terminate_declaration?(template, component)
  end

  def terminate_template?( template, component )
    (component.will_parse_template and \
     (component.will_parse_template != template.template_string)) or \
    (template.template_mtime and \
     (File.mtime(component.template_file) > template.template_mtime))
  end

  def terminate_declaration?( template, component )
    ((component.will_parse_declaration) and \
     (component.will_parse_declaration != template.declaration_hash)) or \
    (template.declaration_mtime and \
     (File.mtime(component.declaration_file) > template.declaration_mtime))
  end

  def create_template( component )
    template = Template.new
    template.component = component
    parser = load_template(component)
    template.template_node = parser.node
    template.template_string = parser.html_string
    template.declaration_store = load_declarations(component, parser.declarations)
    template
  end

  def load_template( component )
    klass = @application.htmlparser_class
    if string = component.will_parse_template then
      parser = klass.new
      parser.parse(string)
    elsif path = component.template_file then
      parser = klass.new(path)
    else
      raise "can't find a template file - #{component.class}"
    end
    parser
  end

  def load_declarations( component, source )
    store = nil
    if hash = component.will_parse_declarations then
      DeclarationStore.merge_source(source, hash)
      store = DeclarationStore.new_with_hash(hash)
    elsif path = component.declaration_file then
      begin
        store = DeclarationStore.new_with_file(path, source)
      rescue Exception => e
        raise "can't load declaration file (#{path}): #{e.message}"
      end
    else
      store = DeclarationStore.new_from_hash(source)
    end
    store
  end

  def save( template )
    raise StandardError, 'Not implemented'
  end
    
  def restore( component )
    raise StandardError, 'Not implemented'
  end
    
  def remove( component )
    raise StandardError, 'Not implemented'
  end
    
end


class FileTemplateStore < TemplateStore

    TMPDIR = 'template_cache'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end
  
  def save( template )
    unless FileTest.directory?(@tmpdir) then
      require 'fileutils'
      FileUtils.makedirs(@tmpdir)
    end
    
    filename = cache_file_name(template.component)
    FileLock.exclusive_lock(filename) do |file|
      Marshal.dump(template, file)
    end
  end

  def restore( component )
    template = nil
    if cache? and exist?(component) then
      FileLock.shared_lock(cache_file_name(component)) do |file|
        template = Marshal.load(file)
      end
    end    
    template
  end
  
  def cache_file_name(component)
    File.join(@tmpdir, component.class.to_s.gsub('::', '__'))
  end

  def exist?( component )
    FileTest.exist?(cache_file_name(component))
  end
  
end


class MemoryTemplateStore < TemplateStore

  def initialize( application )
    super
    @caches = {}
  end

  def save( template )
    @caches[template.component.class] = template.cache_copy
  end

  def restore( component )
    if template = @caches[component.class] then
      template = template.cache_copy
    end
    template
  end

  def exist?( component )
    @caches.key?(component.class)
  end

end

end
module CGIKit

  class RequestHandler
    attr_reader :application

    def initialize( application )
      @application = application
    end


    #
    # parsing session IDs, context IDs and request handler key from URLs
    #

    # Returns a session ID parsed from the request.
    # If you customize URL format including session ID,
    # override the method by subclasses.
    def session_id( request ); end

    def session_id_from_cookie( request )
      sid = nil
      cookie = request.cookie(@application.session_key)
      if cookie then
        sid = cookie.value
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def session_id_from_url( request )
      sid = nil
      if path = request.request_handler_path then
        sid = id_by_separating(path, false)
      end
      sid
    end

    def id_by_separating( path, context = true )
      path = path.reverse.chop.reverse
      separated = path.split('/')
      separated.delete_at(0)
      id = nil
      case separated.size
      when 1
        if context then
          if separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        else
          if !separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        end
      when 2
        if context then
          id = separated.last
        else
          id = separated.first
        end
      end
      if id and !context and !Session.session_id?(id) then
        id = nil
      end
      id
    end

    # Returns a context ID parsed from the request.
    # If you customize URL format including session ID,
    # override the method by subclasses.
    def context_id( request ); end

    def request_handler_key( request )
      key = nil
      if info = request.request_handler_path then
        info = info.reverse.chop.reverse
        separated = info.split('/')
        key = separated.first
      end
      key
    end


    #
    # request-response loop
    #

    # Abstract method. Returns a response object.
    def handle_request( request )
      request.session_id = session_id(request)
      request.context_id = context_id(request)
    end

    def transaction( element, context, &block )
      element.begin_context(context)
      block.call
      element.end_context(context)
    end

    def take_values_from_request( element, request, context )
      transaction(element, context) do
        element.take_values_from_request(request, context)
      end
    end

    def invoke_action( element, request, context )
      result = nil
      transaction(element, context) do
        result = element.invoke_action(request, context)
      end

      if (Component === result) or (Response === result) then
        result
      else
        nil
      end
    end

    def append_to_response( element, response, context, result = nil )
      transaction(element, context) do
        element.append_to_response(response, context)
      end
    end


    #
    # generating URLs
    #

    def url( context, path, query, is_secure, port = nil ); end

    def application_path( request, is_secure = false, port = nil )
      protocol = nil
      if is_secure == true then
        protocol = 'https://'
      else
        protocol = 'http://'
      end

      domain = request.server_name || 'localhost'

      if port.nil? and request.server_port then
        port = request.server_port.to_i
      end
      if port == 80 then
        port = nil
      end

      script = (request.script_name || @application.path).dup
      script.sub!(/\A\//, '')

      if port then
        path = "#{protocol}#{domain}:#{port}/#{script}"
        path.gsub!(/\/\Z/, '')
      else
        path = "#{protocol}#{domain}/#{script}"
      end
      path
    end

    def query_string( hash = {} )
      str = ''
      keys = hash.keys.sort do |a, b|
        a.to_s <=> b.to_s
      end
      keys.each do |key|
        value = hash[key]
        if Array === value then
          value.each do |item|
            str << query_association(key, item)
          end
        else
          str << query_association(key, value)
        end
      end
      str.chop!
      str
    end

    def query_association( key, value )
      unless value.nil? then
        "#{key}=#{Utilities.escape_url(value.to_s)};"
      else
        "#{key};"
      end
    end
  end


  class ComponentRequestHandler < RequestHandler
    def handle_request( request )
      super
      context = @application.create_context(request)
      context.request_handler_key = @application.component_request_handler_key

      transaction(context.component, context) do
        @application.take_values_from_request(request, context)
      end

      result = nil
      context.delete_all
      transaction(context.component, context) do
        result = @application.invoke_action(request, context)
      end

      if Response === result then
        context.response = result
      else
        if (Component === result) and (context.component != result) then
          result.awake_from_restoration(context)
          context.component = result
        else
          result = context.component.root
        end
        context.delete_all
        transaction(result, context) do
          @application.append_to_response(context.response, context)
        end
        context.response.component = result
      end

      @application.save_session(context)
      context.response
    end

    def url( context, path = nil, query = {}, is_secure = false, port = nil )
      str = application_path(context.request, is_secure, port)
      str << "/#{@application.component_request_handler_key}"
      if context.session.store_in_url then
        str << "/#{context.session.session_id}"
      end
      str << "/#{context.context_id}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def session_id( request )
      sid_url = session_id_from_url(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_url then
        sid_url
      else
        nil
      end
    end

    def context_id( request )
      id = nil
      if path = request.request_handler_path then
        id = id_by_separating(path, true)
      end
      id
    end

  end


  class DirectActionRequestHandler < RequestHandler

    class ActionResultError < StandardError #:nodoc:
    end

    def session_id( request )
      sid_query = session_id_from_query(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_query then
        sid_query
      else
        nil
      end
    end

    def session_id_from_query( request )
      if sid = request[@application.direct_action_session_key] then
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def handle_request( request )
      super
      direct_action, action_name = direct_action_and_action_name(request)
      result = direct_action.perform_action(action_name)

      unless result.respond_to?(:generate_response) then
        raise ActionResultError, \
          "Direct action must return an object has generate_response()" +
          " and not nil. - #{direct_action.class}##{action_name}"
      end

      response = result.generate_response
      if Component === result then
        response.component = result
        @application.save_session(result.context)
      end
      response
    end

    def direct_action_and_action_name( request )
      klass, action_name = direct_action_class_and_action_name(request)

      direct_action = klass.new(@application, request)
      if action_name.nil? then
        action_name = direct_action.default_action_name
      end
      if !direct_action.direct_action?(action_name) then
        klass = @application.direct_action_class
        direct_action = klass.new(@application, request)
        action_name = direct_action.default_action_name
      end
      [direct_action, action_name]
    end

    def direct_action_class_and_action_name( request )
      klass = @application.direct_action_class
      action_name = nil
      if path = request.request_handler_path then
        path = path.dup
        path.gsub!(/\A\//, '')
        path.gsub!(/\?(.*)/, '')
        key, class_name, action_name = path.split('/')
        begin
          klass = Object
          class_name.split('::').each do |name|
            klass = klass.const_get(name)
          end
        rescue Exception => e
          klass = @application.direct_action_class
          unless action_name then
            action_name = class_name
          end
        end
      end
      unless klass <= DirectAction then
        klass = @application.direct_action_class
        action_name = nil
      end
      [klass, action_name]
    end

    def url( context, path = nil, query = {}, is_secure = false, port = 80 )
      str = application_path(context.request, is_secure, port)
      str << "/#{@application.direct_action_request_handler_key}"
      str << "/#{path}" if path
      if context.has_session? and context.session.store_in_url then
        query[@application.direct_action_session_key] = context.session.session_id
      end
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def direct_action_url( context, action_class, action_name, query )
      action_class = @application.direct_action_class unless action_class
      key = @application.direct_action_request_handler_key
      path = direct_action_path(action_class, action_name)
      url(context, path, query, false, nil)
    end

    def direct_action_path( action_class, action_name )
      if action_name == DirectAction::DEFAULT_ACTION_NAME then
        action_name = ''
      end
      if @application.direct_action_class.to_s == action_class.to_s then
        action_name
      elsif action_name.empty?
        action_class.to_s
      else
        "#{action_class}/#{action_name}"
      end
    end

  end


  class ResourceRequestHandler < RequestHandler

    RESOURCE_KEY = 'data'

    def handle_request( request )
      super
      response = Response.new
      key = request.form_value(RESOURCE_KEY)
      if data = @application.resource_manager.bytedata(key) then
        response.headers['Content-Type'] = data.content_type
        response.content = data.to_s
        @application.resource_manager.remove_data(key)
      else
        response.status = 404
      end
      response
    end

    def resource_url( name, request )
      str = application_path(request)
      str << "/#{@application.resource_request_handler_key}"
      str << "?#{RESOURCE_KEY}=#{name}"
      str
    end

  end

end
module CGIKit

  VERSION = '2.0.0'

  class CGIKitError < StandardError
  end


  class Application

    include KeyValueCoding

    class SessionCreationError < StandardError #:nodoc:
    end
    class SessionRestorationError < StandardError #:nodoc:
    end
    class SessionAuthorizationError < SessionRestorationError #:nodoc:
    end
    class SessionTimeoutError < SessionRestorationError #:nodoc:
    end
    class PageRestorationError < StandardError #:nodoc:
    end

    # Main component. If session ID or context ID aren't specified,
    # this component is shown. The default value is MainPage.
    attr_accessor :main

    # Locale of an application in a transaction. If the locale is specified,
    # CGIKit change a template for a component. The name of the template
    # includes the component name and the locale name. Also, the template name
    # is seprated by underscore("_"). For example, if the locale is "ja" and
    # the component name is "MainPage", the template name is "MainPage_ja.html".
    attr_accessor :locale

    # Main locale of an application. If the value is equal to
    # locale specified in request, components use templates whose name
    # doesn't include the locale name. If the value is
    # "ja", requested locale is "ja" and the component name is "MainPage",
    # the application uses templated named "MainPage.html".
    attr_accessor :master_locale

    # Document root directory.
    attr_accessor :document_root

    # The application URL based on SCRIPT_NAME.
    attr_accessor :baseurl

    # The file system path of the application.
    attr_accessor :path

    # The file system paths for components. Components are searched under it.
    attr_accessor :component_paths

    # Resource directory.
    # This directory includes files to be used by the application,
    attr_accessor :resources

    # Web server resources directory.
    # This directory includes files to be displayed to browser.
    # The files are used by Image element, etc.
    attr_accessor :web_server_resources

    # ResourceManager object.
    attr_accessor :resource_manager

    # Adapter object.
    attr_accessor :adapter

    # Adapter class. The default value is CGI class.
    attr_accessor :adapter_class

    # Name or class of an error page component to show caught errors.
    attr_accessor :error_page

    # Temporary directory to be used by the framework.
    # The framework uses this to store sessions and template caches.
    attr_accessor :tmpdir

    # Session key. This key is used in cookie.
    attr_accessor :session_key

    # Session key in direct action.
    # This key is used in hidden fields of form and URL when using direct action.
    attr_accessor :direct_action_session_key

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Expiry date of cookie for session. If you set the value to nil,
    # session cookies will be invalid when closing browser.
    attr_accessor :session_cookie_expires

    # Enables or disables the use of URLs for storing session IDs.
    attr_accessor :store_in_url

    # Enables or disables the use of cookies for storing session IDs.
    attr_accessor :store_in_cookie

    # Enables or disables session authorization by browsers.
    # If you set the value to true, the application raises error
    # when an user accesses it with browser that is different from
    # one registered session.
    attr_accessor :auth_by_user_agent

    # Enables or disables session authorization by IP addresses.
    # If you set the value to true, the application raises error
    # when an user accesses it with IP address that is different from
    # one registered session.
    attr_accessor :auth_by_remote_addr

    # Encoding to encode character code of form data.
    # The default implementation uses Kconv to encode Japanese character codes.
    # Then specify constant values of Kconv; Kconv::JIS, Kconv::SJIS, Kconv::EUC, etc.
    # If the value is nil, form data is not encoded. The default value is nil.
    attr_accessor :encoding

    # Request handler key to display components.
    attr_accessor :component_request_handler_key

    # Request handler key to invoke direct actions.
    attr_accessor :direct_action_request_handler_key

    # Request handler key to display resource files.
    attr_accessor :resource_request_handler_key

    # Default request handler key. Default key is component request handler key.
    attr_accessor :default_request_handler

    # Request handler for components.
    attr_accessor :component_request_handler

    # Request handler for direct actions.
    attr_accessor :direct_action_request_handler

    # Request handler for resources.
    attr_accessor :resource_request_handler

    # Session class. Default class is CGIKit::Session.
    attr_accessor :session_class

    # Whether or not validates setting of attributes for each elements.
    # If wrong attribute name or combination are found, raises error.
    attr_accessor :validate_api
    alias validate_api? validate_api

    # HTML parser class. Default class is CGIKit::HTMLParser::HTMLParser.
    attr_accessor :htmlparser_class

    # Size to cache components permanently in session.
    # Newly generated page is cached automatically.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :page_cache_size

    # Size to cache components permanently in session.
    # Permanent page cache is cached optionally, not automatically caching.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :permanent_page_cache_size

    # Direct action class. Default class is CGIKit::DirectAction.
    attr_accessor :direct_action_class

    # Session store class. Default class is CGIKit::FileSessionStore.
    attr_accessor :session_store_class

    # Context class. Default class is CGIKit::Context.
    attr_accessor :context_class

    # Whether or not caches templates to reduce parsing load.
    attr_accessor :cache_template

    attr_accessor :template_store
    attr_accessor :template_store_class
    attr_accessor :sweep_password

    COMPONENT_REQUEST_HANDLER_KEY     = 'c'
    DIRECT_ACTION_REQUEST_HANDLER_KEY = 'd'
    RESOURCE_REQUEST_HANDLER_KEY      = 'r'
    CGIKIT_LIB                        = 'cgikit'
    COMPONENT_LIB                     = 'components'

    def initialize
      init_attributes
      init_request_handlers
      init_adapter
      init_component_paths
      init_name_spaces
      init
    end

    def init_attributes
      require 'cgikit/components/CKErrorPage/CKErrorPage'
      @main                      = 'MainPage'
      @error_page                = CKErrorPage
      @tmpdir                    = './tmp' || ENV['TMP'] || ENV['TEMP']
      @session_key               = '_session_id'
      @direct_action_session_key = '_sid'
      @manage_session            = false
      @timeout                   = 60 * 60 * 24 * 7
      @session_cookie_expires    = 60 * 60 * 24 * 7
      @store_in_url              = true
      @store_in_cookie           = false
      @auth_by_user_agent        = false
      @auth_by_remote_addr       = false
      @session_class             = Session
      @session_store             = nil
      @session_store_class       = FileSessionStore
      @template_store            = nil
      @template_store_class      = FileTemplateStore
      @encoding                  = nil
      @resources                 = './'
      @validate_api              = true
      @cache_template            = true
      @htmlparser_class          = HTMLParser::HTMLParser
      @page_cache_size           = 30
      @permanent_page_cache_size = 30
      @direct_action_class       = DirectAction
      @context_class             = Context
      @baseurl                   = nil
      @locale                    = nil
      @master_locale             = 'en'
    end

    def init_request_handlers
      @component_request_handler_key     = COMPONENT_REQUEST_HANDLER_KEY
      @direct_action_request_handler_key = DIRECT_ACTION_REQUEST_HANDLER_KEY
      @resource_request_handler_key      = RESOURCE_REQUEST_HANDLER_KEY
      @component_request_handler         = ComponentRequestHandler.new(self)
      @direct_action_request_handler     = DirectActionRequestHandler.new(self)
      @resource_request_handler          = ResourceRequestHandler.new(self)
      @default_request_handler           = @component_request_handler
    end

    def init_adapter
      # decides interface of adapter
      if defined?(MOD_RUBY) then
        @adapter_class = Adapter::ModRuby
        @path = Apache.request.filename
      else
        @adapter_class = Adapter::CGI
        @path = $0
      end
    end

    def init_component_paths
      @component_paths = [Dir.pwd]
      $LOAD_PATH.each do |path|
        @component_paths << File.join(path, CGIKIT_LIB, COMPONENT_LIB)
      end
    end

    def init_name_spaces
      @name_spaces = [CGIKit, Object]
      klass = Object
      self.class.to_s.split('::').each do |class_name|
        klass = klass.const_get(class_name)
        @name_spaces << klass
      end
    end

    # Returns the name of the application without file extension.
    def name
      File.basename( @path, '.*' )
    end

    def take_values_from_hash( hash )
      hash.each do |key, value|
        self[key] = value
      end
    end

    def template_store
      @template_store ||= @template_store_class.new(self)
    end
    
    
    
    def load_all_components( path, subdir = false )
      __each_component_file(path, subdir) do |file|
        require(file)
      end
    end

    def autoload_all_components( path, mod = Object, subdir = false )
      __each_component_file(path, subdir) do |file|
        class_name = File.basename(file, '.rb')
        mod.autoload(class_name, file)
      end
    end

    private

    def __each_component_file(path, subdir)
      if subdir
        pattern = File.join(path, '**', '*.rb')
      else
        pattern = File.join(path, '*.rb')
      end
      
      Dir.glob(pattern) do |file|
        if /\.rb$/ === file and FileTest::readable?(file)
          yield file
        end
      end
    end
    
    public

    def load_configuration( path )
      load(path)
      configure
    end


    #
    # hook
    #

    def init; end

    def configure; end


    #
    # managing sessions
    #

    # Session database object (SessionStore).
    def session_store
      @session_store ||= @session_store_class.new(self)
    end

    # Creates a session.
    def create_session( request )
      begin
        klass = @session_class || Session
        session = klass.new(request.session_id)
        session.awake_from_restoration(self)
        return session
      rescue
        if sid = request.session_id then
          msg = "for #{sid}"
        else
          msg = "because session id is not specified"
        end 
        raise SessionCreationError, "Failed creating session #{msg}."
      end
    end

    # Returns a restored session objects with session ID.
    def restore_session( session_id, context )
      begin
        session = session_store.checkout(session_id, context.request)
      rescue Exception => e
        raise SessionRestorationError, "Failed restoring session for #{session_id}"
      end
      if session then
        context.session = session
        context.session.awake_from_restoration(self)
        context.session.validate
      end
      session
    end

    # Saves the session, and set a cookie if "store_in_cookie" attribute is
    # setted. If "clear" method of the session is called, the session is deleted.
    def save_session( context )
      unless context.has_session? then return end

      if context.session.terminate? then
        sid = context.session.session_id
        @session_store.remove(sid)
      else
        session_store.checkin(context)
        if context.session.store_in_cookie? then
          context.session.set_cookie
        end
      end
    end


    #
    # handling requests
    #

    # Runs the application.
    def run( command_or_request = nil, response = nil )
      if CGIKit.const_defined?(:Command) and \
        CGIKit.const_get(:Command) === command_or_request then
        request = command_or_request.request
      else
        request = command_or_request
      end

      set_adapter_and_handler
      @adapter.run(request, response) do |request|
        set_attributes_from_request(request)
        request.request_handler_key = \
        @default_request_handler.request_handler_key(request)

        begin  
          handler = request_handler(request.request_handler_key)
          request.context_id = handler.context_id(request)
          response = handler.handle_request(request)

        rescue Exception => e
          # create context without session
          request.session_id = nil
          context = @context_class.new(request, self)
          begin
            response = handle_error(e, context)
          rescue Exception => e
            # trap error occured by customized handle_error 
            response = default_error_page(e, context)
          end
        end

        response
      end
      response
    end

    def request_handler( key )
      unless handler = @request_handlers[key] then
        handler = @default_request_handler
      end
      handler
    end

    def set_adapter_and_handler
      if @set_adapter_and_handler then return end
      @adapter = create_adapter
      @set_adapter_and_handler = true
      @request_handlers = {
        @component_request_handler_key     => @component_request_handler,
        @direct_action_request_handler_key => @direct_action_request_handler,
        @resource_request_handler_key      => @resource_request_handler }
    end

    def set_attributes_from_request( request )
      if @set_attributes_from_request then return end
      @baseurl       = request.script_name              unless @baseurl
      @document_root = request.headers['DOCUMENT_ROOT'] unless @document_root
      @web_server_resources ||= @document_root
      @resource_manager       = ResourceManager.new(self)
      @set_attributes_from_request = true
    end

    def take_values_from_request( request, context )
      context.session.take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.session.invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.session.append_to_response(response, context)
    end

    def create_context( request )
      handler = request_handler(request.request_handler_key)
      context = @context_class.new(request, self)
      
      if session = restore_session(request.session_id, context) then
        context.session = session
      else
        session = create_session(request)
        context.session = session
      end
      session.context = context
      
      if component = context.session.restore_page(context.sender_id) then
        root = component.root
        context.component = root
        root.awake_from_restoration(context)
      else
        context.component = page(@main, context)
      end
      context.delete_all
      context
    end

    # Creates an adapter object.
    def create_adapter
      @adapter_class.new
    end


    #
    # handling errors
    #

    # Handles every errors and return an error page component.
    def handle_error( error, context )
      case error
      when SessionCreationError 
        response = handle_session_creation_error(error, context)
      when SessionRestorationError 
        response = handle_session_restoration_error(error, context)
      when PageRestorationError 
        response = handle_page_restoration_error(error, context)
      else
        page = default_error_page(error, context)
        response = page.generate_response
      end
      response
    end

    def handle_session_creation_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_session_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_page_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    # Return a default error page component.
    def default_error_page( error, context )
      error_page       = page('CKErrorPage', context)
      error_page.error = error
      error_page
    end

    def url( key, context, path, query_string, is_secure, port = nil )
      request_handler(key).url(context, path, query_string, is_secure, port)
    end

    def direct_action_url( context, action_class, action_name, query )
      handler = @direct_action_request_handler
      handler.direct_action_url(context, action_class, action_name, query)
    end


    #
    # Creating components
    #

    # Creates a specified page component.
    def page( name, request_or_context, *args )
      context = request_or_context
      if Request === request_or_context then
        context = @context_class.new(request_or_context, self)
      end
      if Class === name then
        klass = name
      else
        klass = class_named(name)
      end
      klass.new(context, *args)
    end

    def class_named( name )
      Utilities.class_named_from(name, @name_spaces)
    end

  end

end


module CGIKit

# An API object has information of dynamic element's or component's bindings.
# The API object can validate declaration of the element or component at runtime.
class API

  attr_accessor :element_type, :content, :bindings, :validations

  def initialize( element_type, content = false )
    @element_type = element_type
    @content      = content
    @bindings     = {}
    @validations  = []
  end


  #
  # accessing
  #

  def []( name )
    b = @bindings[name]
    if b
      return b
    end
    
    nil
  end

  def <<( object )
    case object
    when Binding
      @bindings[object.name] = object
    when Validation
      @validations << object
    end
  end


  #
  # testing
  #

  def has_binding?( name )
    if @bindings[name]
      return true
    else
      false
    end
  end


  #
  # validating
  #

  # Returns nil or array contains errors.
  def validate( associations, component_name, element_name )
    errors = []
    bindings = @bindings.values
    target = bindings + @validations
    target.each do |t|
      begin
        t.validate(associations, component_name, element_name, @element_type)
      rescue ValidationError => e
        errors << e
      end
    end

    if errors.empty? then
      nil
    else
      errors
    end
  end


  #
  # generating string of other tags
  #

  # Return a string of values of optional attributes with
  # "other" attribute string excepting the element's own attributes.
  def other( component, associations )
    optional = ''
    associations.each do |key, as|
      if (has_binding?(key) == false) and (key != Declaration::ELEMENT_KEY) and \
        (key != Declaration::OTHER_KEY) then
        value = as.value(component).to_s
        optional << " #{key}=\"#{value}\""
      end
    end

    if as = associations[Declaration::OTHER_KEY] then
      other = ' ' + as.value(component).to_s
    else
      other = ''
    end      

    optional + other
  end

  def enabled( component, associations )
    enabled = true
    associations.each do |key, as|
      if key == Declaration::ENABLED_KEY then
        enabled = as.bool(component)
      end
    end
    if enabled then
      ''
    else
      ' disabled="disabled"'
    end
  end

end


class Binding
  BOOLEAN        = 'Boolean'
  PAGE_NAMES     = 'Page Names'
  MIME_TYPES     = 'MIME Types'
  RESOURCES      = 'Resources'
  DIRECT_ACTIONS = 'Direct Actions'

  attr_accessor :name, :required, :settable

  # Value type to set.
  attr_accessor :value_set

  # Default value.
  attr_writer :default

  def initialize( name )
    @name     = name
    @required = false
    @settable = false
    @default  = nil
  end

  # Returns the default value.
  # If the value is Symbol, it is resolved with key-value coding.
  def default( component = nil )
    if (Symbol === @default) and component then
      component.value_for_keypath(@default)
    else
      @default
    end
  end

  def validate( associations, component_name, element_name, element_type )
    msg = nil
    isbind = false

    associations.each do |key, as|
      if (key == @name) 
        if (@settable == true) and (as.settable? == false) then
          msg = "'#@name' must be bound to a settable value."
        end
        isbind = true
        break
      end
    end

    if (isbind == false) and (@required == true) then
      msg = "'#@name' is a required binding."
    end

    if msg then
      raise ValidationError.new(msg, component_name, element_name, element_type)
    end
  end
end


class Validation
  attr_accessor :message, :condition

  def initialize( message, condition = nil )
    @message = message
    @condition = condition
  end

  def validate( associations, component_name, element_name, element_type )
    if @condition.eval?(associations) then
      raise ValidationError.new(@message, component_name, element_name, element_type)
    end
  end
end


class KeyErrorCondition
  BOUND      = 'bound'
  UNBOUND    = 'unbound'
  SETTABLE   = 'settable'
  UNSETTABLE = 'unsettable'

  attr_reader :key, :error

  def initialize( key, error )
    @key = key
    @error = error
  end

  def eval?( associations )
    as = associations[@key]
    if ((as and (@error == BOUND)) or
        (as.nil? and (@error == UNBOUND)) or
       (as and as.settable? and (@error == SETTABLE)) or
       (as and as.constant? and (@error == UNSETTABLE))) then
      true
    else
      false
    end
  end
end


class AndCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      unless condition.eval? associations then
        return false
      end
    end
    true
  end
end


class OrCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      if condition.eval? associations then
        return true
      end
    end
    false
  end
end


class NotCondition
  attr_reader :condition

  def initialize( condition )
    @condition = condition
  end

  def eval?( associations )
    unless @condition.eval? associations then
      true
    else
      false
    end
  end
end


class ValidationError < StandardError
  attr_accessor :errors, :element_name, :element_type, :component_name

  def initialize( message = '',
                  component_name = nil,
                  element_name = nil,
                  element_type = nil )
    super(message)
    @component_name = component_name
    @element_name = element_name
    @element_type = element_type
    @errors = []
  end

  def <<( error )
    if Array === error then
      @errors.concat(error)
    else
      @errors << error
    end
  end

end


module APIUtilities

  #
  # binding
  #

  def href_binding
    Binding.new(:href)
  end

  def name_binding( required = false )
    binding = Binding.new(:name)
    binding.required = required
    binding
  end

  def page_binding( required = false )
    binding = Binding.new(:page)
    binding.required = required
    binding
  end

  def src_binding( required = false )
    binding = Binding.new(:src)
    binding.required = required
    binding
  end

  def action_binding( required = false )
    binding = Binding.new(:action)
    binding.required = required
    binding
  end

  def list_binding( required = true )
    binding = Binding.new(:list)
    binding.required = required
    binding
  end

  def item_binding()
    binding = Binding.new(:item)
    binding.settable = true
    binding
  end

  def display_binding()
    binding = Binding.new(:display)
    binding
  end

  def value_binding( required = true, settable = false )
    binding = Binding.new(:value)
    binding.required = required
    binding.settable = settable
    binding
  end

  def enabled_binding
    binding = Binding.new(:enabled)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def escape_binding
    binding = Binding.new(:escape)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def checked_binding( default = false )
    binding = Binding.new(:checked)
    binding.default = default
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding
  end

  def selection_binding
    binding = Binding.new(:selection)
    binding.settable = true
    binding
  end

  def selections_binding
    binding = Binding.new(:selections)
    binding.settable = true
    binding
  end

  def selected_value_binding
    binding = Binding.new(:selected_value)
    binding.settable = true
    binding
  end

  def selected_values_binding
    binding = Binding.new(:selected_values)
    binding.settable = true
    binding
  end

  def string_binding
    binding = Binding.new(:string)
    binding
  end

  def validate_binding
    binding = Binding.new(:validate)
    binding
  end

  def pass_binding
    binding = Binding.new(:pass)
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding.default = false
    binding
  end

  def direct_action_binding
    binding = Binding.new(:direct_action)
    binding.value_set = Binding::DIRECT_ACTIONS
    binding
  end

  def action_class_binding
    binding = Binding.new(:action_class)
    binding
  end

  def session_id_binding
    binding = Binding.new(:session_id)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end


  #
  # setting
  #

  def set_validation( api )
    api << validate_binding()
    api << pass_binding()
    api << universal_validation(:validate, :pass)
  end

  def set_direct_action( api )
    api << direct_action_binding()
    api << action_class_binding()
    api
  end


  #
  # condition
  #

  def existential_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::BOUND)
        end
      end
      existentials = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, existentials])
    end
    OrCondition.new(conditions)
  end

  def universal_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::UNBOUND)
        end
      end
      universals = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, universals])
    end
    OrCondition.new(conditions)
  end

  # * or(ex1:bound, ex2:bound) or(un1:bound, un2:bound) or(both1:bound, both2:bound)
  # * on(and(ex, un), and(ex, both), and(un, both))
  def required_condition( existential = [], universal = [], both = [] )
    all_unbound = unbound_condition(existential + universal + both)
    any = any_condition(existential, universal, both)
    OrCondition.new([all_unbound, any])
  end

  def any_condition( existential = [], universal = [], both = [] )
    ex_any = any_bound_condition(existential)
    un_any = any_bound_condition(universal)
    both_any = any_bound_condition(both)

    ex_un = AndCondition.new([ex_any, un_any])
    ex_both = AndCondition.new([ex_any, both_any])
    un_both = AndCondition.new([un_any, both_any])

    ex_cond = existential_condition(*existential)
    un_cond = universal_condition(*universal)

    OrCondition.new([ex_un, ex_both, un_both, ex_cond, un_cond])
  end

  def unbound_condition( names )
    and_condition(names, KeyErrorCondition::UNBOUND)
  end

  def and_condition( names, error )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, error)
    end
    AndCondition.new(conditions)
  end

  def any_bound_condition( names )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
    end
    OrCondition.new(conditions)
  end


  #
  # validation
  #

  def existential_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} can't both be bound."
    Validation.new(msg, existential_condition(*names))
  end

  def universal_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} must be bound when one of the both is bound."
    Validation.new(msg, universal_condition(*names))
  end

  def required_validation( existential = [], universal = [], both = [] )
    composed_validation(true, existential, universal, both)
  end

  def any_validation( existential = [], universal = [], both = [] )
    composed_validation(false, existential, universal, both)
  end

  def composed_validation( is_required, existential, universal, both )
    list_existential = join_names(existential, 'or')
    list_universal = join_names(universal, 'and')
    list_both = join_names(both, 'and')

    if is_required and universal.empty? and both.empty? then
      prefix = 'exactly'
      aux = 'must'
    else
      prefix = 'either'
      aux = 'may'
    end
    msg = "#{prefix} "
    unless existential.empty? then
      msg << "one of #{list_existential} #{aux} be bound"
      if (universal.empty? == false) or (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "both of #{list_universal} #{aux} be bound"
      if (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "either or both of #{list_both} #{aux} be bound"
    end
    msg << '.'

    if is_required then
      Validation.new(msg, required_condition(existential, universal, both))
    else
      Validation.new(msg, any_condition(existential, universal, both))
    end
  end

  def join_names( names, keyword )
    list = ''
    names.each do |name|
      if names.first == name then
        list << "'#{name}'"
      elsif names.last == name then
        list << " #{keyword} '#{name}'"
      else
        list << ", '#{name}'"
      end
    end
    list
  end

  def item_display_value_validation
    msg = "'item' must be bound when 'display' or 'value' is bound."
    item_condition = KeyErrorCondition.new(:item, KeyErrorCondition::UNBOUND)
    display_condition = KeyErrorCondition.new(:display, KeyErrorCondition::BOUND)
    value_condition = KeyErrorCondition.new(:value, KeyErrorCondition::BOUND)
    display_value = OrCondition.new([display_condition, value_condition])
    condition = AndCondition.new([item_condition, display_value])
    Validation.new(msg, condition)
  end

end

end

module CGIKit

class Element
  class UnknownElementError < StandardError; end #:nodoc:
  class AttributeError      < StandardError; end #:nodoc:

  @@apis = {}

  attr_accessor :name, :node, :associations

  class << self
    include APIUtilities

    def api
      unless @@apis[self] then
        @@apis[self] = create_api
      end
      @@apis[self]
    end

    # Implemented by subclasses. Returns a new API object for the element.
    def create_api; end
  end

  def api
    self.class.api
  end

  def begin_context( context )
    context.increment
  end

  def end_context( context ); end

  def take_values_from_request( request, context ); end
  def invoke_action( request, context ); end
  def append_to_response( response, context ); end

  def empty?
    @node.nil? or @node.empty?
  end
end


# The super class of dynamic element classes.
# Dynamic elements convert themselves to HTML.
# These are very important in development with CGIKit.
#
# == Paths for searching elements and components
# Element objects, including components, are usually instantiated
# by Element.instance. This method loads elements/components and creates
# element objects. In this method, some paths are searched to require files.
#
# The searched paths:
# 1. CKApplicaton#component_path
# 2. ($LOAD_PATH)/cgikit/elements
# 3. ($LOAD_PATH)/cgikit/components
#
# The latter two paths are for extention elements or components.
# It is recommended that your own elements or components are installed
# in the 1st path.
class DynamicElement < Element

  attr_accessor :name, :root, :values, :context_id, :application, :parent

  class << self
    def keys_to_delete_from_associations
      []
    end
  end

  def initialize( name, associations, root )
    super()
    @name = name
    @associations = associations
    @root = root
    @values = {}
    @once = {}
    @application = @root.application
    self.class.keys_to_delete_from_associations.each do |key|
      @associations.delete(key)
    end
    init
  end


  #
  # request-response loop
  #

  def take_values_from_request( request, context )
    @node.take_values_from_request(request, context) if @node
  end

  def invoke_action( request, context )
    @node.invoke_action(request, context) if @node
  end

  def append_to_response( response, context ); end

  # Returns value from the request if the context is continued.
  # If not be continued, returns nil.
  def value_from_request( request, context )
    values_from_request(request, context)[0]
  end

  def values_from_request( request, context )
    values = nil
    if (values = request.form_values[context.context_id]) and \
      (values.empty? == false) then
    elsif @values[:name] and context.in_form? and \
      (values = request.form_values[@values[:name].to_s]) and \
      (values.empty? == false) then
    end
    values ||= []

    if !(ByteData === values) and (values.empty? == false) and \
      (encoding = application.encoding) then
      values = encode_strings(values, encoding)
    end
    values
  end

  def encode_strings( strings, encoding )
    encoded = []
    strings.each do |string|
      begin
        encoded << (encode_string(string, encoding) || string)
      rescue Exception => e
        encoded << (failed_encode_string(string, encoding, e) || string)
      end
    end
    encoded
  end


  #
  # take values for binding keys
  #

  def take_value_once( name, does_resolve = true )
    unless @once[name] then
      @values[name] = value(name, does_resolve)
      @once[name] = true
    end
  end

  def take_value( name, does_resolve = true )
    @values[name] = value(name, does_resolve)
  end

  def take_bool( name, does_resolve = true )
    @values[name] = bool(name, does_resolve)
  end

  def value( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.value(root)
      else
        as.value
      end
    else
      if api and api[name] then
        api[name].default(root)
      else
        nil
      end
    end
  end

  def bool( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.bool(root)
      else
        as.bool
      end
    else
      if api and api[name] then
        Association.adapt_to_bool(api[name].default(root))
      else
        nil
      end
    end
  end

  def set_value( name, value )
    if as = @associations[name] then
      as.set_value(value, root)
    end
  end

  def name_value( context )
    escaped_string(@values[:name] || context.context_id)
  end

  def declared?( name )
    @associations.key?(name)
  end

  def direct_action?
    declared?(:direct_action) or declared?(:action_class)
  end


  #
  # escaping
  #

  def escaped_string( string, escape = true )
    if escape then
      string = Utilities.escape_html(string.to_s)
    end
    string
  end


  #
  # creating attributes' string
  #

  def other
    api.other(root, associations)
  end

  def enabled
    api.enabled(root, associations)
  end

  def checked
    " checked=\"checked\""
  end

  def selected
    " selected=\"selected\""
  end


  #
  # validating
  #

  def validate( name )
    take_value(name)
    take_value(Declaration::VALIDATE_KEY, false)
    take_value(Declaration::PASS_KEY, false)
    method = @values[Declaration::VALIDATE_KEY]
    pass = @values[Declaration::PASS_KEY]
    if method and pass then
      result = root.__send__(method, @values[name])
      set_value(Declaration::PASS_KEY, result)
    end
  end


  #
  # hook
  #

  def init; end

  # Convert character code for the form values.
  def encode_string( string, encoding ); end

  # Invoked when encode_string() raised an error.
  # This method should be return string.
  def failed_encode_string( string, encoding, error ); end

end

end


class Array

  def take_values_from_request( request, context )
    each do |item|
      item.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    result = nil
    each do |item|
      if action_result = item.invoke_action(request, context) then
        result = action_result
      end
    end
    result
  end
  
  def append_to_response( response, context )
    each do |item|
      item.append_to_response(response, context)
    end
  end

end

module CGIKit

class Component < Element
  include KeyValueCoding

  CACHE_TEMPLATE_DIR = 'cache'

  # Path for the component.
  attr_reader :path

  # Parent component.
  attr_reader :parent

  attr_accessor :declaration_store, :context

  attr_accessor :context_id, :subcomponents, :application, :declaration_name

  def initialize( context, *args )
    @access_attributes = true
    @subcomponents = []
    @associations = nil
    load_files(context)
    __send__(:init, *args)
  end


  #
  # hook
  #

  # Invoked to initialize component.
  # Form data is setted when the method is called.
  def init; end

  # Invoked when the component parses template.
  # A string returned by the method is used instead of template file.
  def will_parse_template; end

  # Invoked when the component parses declarations.
  # A hash returned by the method is used instead of declaration file.
  def will_parse_declarations; end

  # Invoked when saving the component.
  # If the component has data can't be saved or marshaled with FileSessionStore,
  # you must clear the data in this method.
  def will_save_page; end

  # Invoked after restoring the component.
  def did_restore_page; end


  #
  # accessing
  #

  def root
    component = self
    while (component.root? == false)
      component = component.parent
    end
    component
  end

  def []( key )
    value_for_key(key)
  end

  def []=( key, value )
    take_value_for_key(key, value)
  end

  # Returns base name of the component in name space package.
  def base_name
    self.class.to_s.split('::').last
  end

  # Creates a specified page component.
  def page( name, *args )
    @application.page(name, @context, *args)
  end

  def parent=( parent )
    @parent = parent
    @parent.subcomponents << self
  end

  def cached_template_file
    File.join(@application.tmpdir, CACHE_TEMPLATE_DIR, self.class.to_s).untaint
  end

  def cached_template_dir
    File.join(@application.tmpdir, CACHE_TEMPLATE_DIR).untaint
  end


  #
  # testing
  #

  def root?
    parent.nil?
  end

  def subcomponent?
    @associations and @parent
  end

  # Return true if parent component syncronizes bound attribute to self.
  # Override this method returns false to create a non-synchronizing component.
  def sync?
    true
  end

  def can_set_value?( binding_name )
    if @declaration_store[binding_name] then
      true
    else
      false
    end
  end

  def has_session?
    @context.has_session?
  end

  def loaded?
    @loaded
  end


  #
  # loading
  #

  # loads template, merges declaration hash, loads declaration file
  def load_files( context )
    @context = context
    @application = context.application

    template = @application.template_store.checkout(self)
    @template_node = template.template_node
    @declaration_store = template.declaration_store

    load_associations_from_parent_declaration
    validate_api
    @loaded = true
  end

  def load_associations_from_parent_declaration
    if parent and @declaration_name then
      @associations = parent.declaration_store[@declaration_name]
    end
  end

  def validate_api
    if application.validate_api? then
      @declaration_store.validate_api(self.class)
    end
  end


  #
  # request-response loop
  #

  def begin_context( context )
    increment_and_record_if_subcomponent(context)
    init_context_id(context)
  end

  def increment_and_record_if_subcomponent( context )
    if subcomponent? then
      context.increment
      @parent_context_id = context.context_id
    end
  end

  def init_context_id( context )
    context.delete_all
    if context.has_session? then
      context.init_component_id_with_session(self)
    else
      context.init_component_id_without_session(self)
    end
    context.append_zero
  end

  def component_id( context )
    if context.has_session? then
      unless component_id = session.component_id(self) then
        session.save_page(self)
        component_id = session.component_id(self)
      end
    else
      if context.component_id then
        component_id = context.component_id + 1
      else
        component_id = 0
      end
    end
    component_id
  end

  def end_context( context )
    if subcomponent? then
      context.component = parent
      @parent_context_id = context.context_id
    end
  end

  def sync( context, &block )
    unless loaded? then
      awake_from_restoration(context)
    end
    if subcomponent? and sync? then
      pull_values_from_parent
      result = block.call
      push_values_to_parent
      result
    else
      block.call
    end
  end

  def take_values_from_request( request, context )
    sync(context) do
      @template_node.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    sync(context) do
      @template_node.invoke_action(request, context)
    end
  end

  def append_to_response( response, context )
    sync(context) do
      context.session.save_page(self)
      @template_node.append_to_response(response, context)
    end
  end

  def generate_response
    response = Response.new
    handler = application.component_request_handler
    handler.transaction(self, context) do
      application.append_to_response(response, context)
    end
    response.component = self
    response
  end

  def to_html
    generate_response.content
  end


  #
  # marshaling
  #

  def dup
    component = super
    component.context = @context
    component.declaration_store = @declaration_store
    component.node = @template_node
    component
  end

  def marshal_dump
    will_save_page
    not_includes = ['@context', '@declaration_store', '@template_node',
      '@node', '@application', '@loaded']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  # Invoked after the component is restored to initialize.
  # This method invokes deletage method did_restore_page() finally.
  def awake_from_restoration( context )
    @context = context
    @application = context.application
    load_files(context)
    @subcomponents.each do |sub|
      sub.awake_from_restoration(context)
    end
    did_restore_page
  end

  def add_all_components( session, permanently = false )
    session.add_component(self, @context, permanently)
    @subcomponents.each do |sub|
      sub.add_all_components(session, permanently)
    end
  end


  #
  # aliases
  #

  # Returns a request object of the context.
  def request; @context.request; end

  # Returns a response object of the context.
  def response; @context.response; end

  # Returns the session object.
  # If the session isn't existed, returns a new session.
  def session; @context.session; end

  # Returns a resource manager object.
  def resource_manager; @application.resource_manager; end


  #
  # template paths
  #

  # Searches file path from:
  # 1. component_path/filename
  # 2. component_path/base_name/filename
  def search_path( filename )
    application.component_paths.each do |path|
      paths = [File.join(path, filename), File.join(path, base_name, filename)]
      paths.each do |path|
        if FileTest.exist?(path.untaint) then
          @path = path
          return path
        end
      end
    end
    nil
  end

  # Returns name of the template file.
  def template_file
    request.languages.each do |lang|
      lang.strip!
      lang = lang.split('-')[0]
      if @application.master_locale == lang then
        return search_path("#{base_name}.html")
      elsif file = search_path("#{base_name}_#{lang}.html") then
        return file
      end
    end
    search_path("#{base_name}.html")
  end

  # Returns name of the declaration( binding ) file.
  def declaration_file
    search_path("#{base_name}.ckd").untaint
  end


  #
  # Synchronizing components
  #

  def next_subcomponent( context_id )
    if @context_id == context_id then
      return @subcomponents.first
    end
    @subcomponents.each_with_index do |component, index|
      if context_id == component.context_id then
        return @subcomponents[index]
      end
    end
    nil
  end

  # Allows a subcomponent to invoke an action method of its parent component
  # bound to the child component.
  def perform_parent_action( name )
    parent.value_for_key(name) if sync?
  end

  def pull_values_from_parent
    #settable_in_component,  take_value_for_key
    @associations.each do |key, as|
      if api and (as = api[key]) and as.settable_in_component?(self) then
        take_value_for_key(key, as.value)
      else
        take_value_for_key(key, as.value(parent))
      end
    end
  end

  def push_values_to_parent
    @associations.each do |key, as|
      if (key != Declaration::ELEMENT_KEY) and api.nil? or \
        (api and (as = api[key]) and \
         (as.settable_in_component?(self) == false)) then
        parent.take_value_for_key(key, value_for_key(key))
      end
    end
  end

end

end


if $0 == __FILE__
  require 'cgikit'
  require 'cgikit/lang/ja'  
end

module CGIKit::HTMLParser
  
  def self.attribute_string(attrs)
    str = ''
    attrs.each do |key, value|
      str << " #{key}=\"#{value}\""
    end
    str
  end
  
  # CGIKit::HTMLParser::Node stands for nodes of Template's tree structure.
  # Node is base class. Actually, its subclasses are used in tree structure.
  class Node
    
    attr_accessor :node, :name, :content, :parent, :attributes
    
    def initialize( name = nil )
      @node = []
      @name = name || ''
      @content = ''
    end
    
    # returns the `index`th child node.  
    def []( index )
      @node[index]
    end
    
    # returns the number of children.
    def size
      @node.size
    end
    
    # returns the last child node.
    def last
      @node.last
    end
    
    # returns the root node of Template's tree structure. This must be the CGIKit::HTMLParser::RootNode.
    def root
      cur = self
      while cur.parent
        cur = cur.parent        
      end
      cur
    end
    
    # returns true if `other` is the same. 
    def ==( other )
      (@name == other.name) and (@node == other.node)
    end
    
    # adds node as child. 
    def <<( node )
      node.parent = self
      @node << node
    end
    
    # returns true if `self` has no children.
    def empty?
      @node.nil? or @node.empty?
    end
    
    # sets values of request object to node if node is CGIKit element.
    def take_values_from_request( request, context )
      @node.each do |n|
        n.take_values_from_request( request, context )
      end
    end
    
    # invokes action method of node if node is CGIKit element.
    def invoke_action( request, context )
      result = nil
      @node.each do |node|
        if node_result = node.invoke_action(request, context) then
          result = node_result
        end
      end
      result
    end
    
    # adds HTML to response object
    def append_to_response( response, context )
      @node.each do |node|
        node.append_to_response(response, context)
      end
    end
        
    # converts `self` to HTML 
    def to_s
      string = ''
      @node.each do |node|
        string << node.to_s
      end
      string
    end
   
    def cache_copy
      copy = self.class.new(@name)
      copy.attributes = @attributes
      copy.content = @content
      @node.each do |node|
        copy << node.cache_copy
      end
      copy
    end
    
  end
  
  # This class is root node of Template's tree structure.
  # Of all nodes, this node only has reference to CGIKit::Component object.
  class RootNode < Node
    attr_accessor :component
    
    def marshal_dump
      dump = {}
      (instance_variables - ['@component']).each do |var|
        dump[var] = instance_variable_get(var)
      end
      dump
    end

    def marshal_load( object )
      object.each do |key, value|
        instance_variable_set(key, value)
      end
    end
    
    def parent
      nil
    end
    
  end
  
  # This class has represents text. The role of this class is that
  # when CGIKit expands component's template, TextNode appends its `content` to `response` object.
  class TextNode < Node
    
    def initialize(name = nil)
      @content = ''
    end
    
    def take_values_from_request( request, context )
    end
    
    def invoke_action( request, context )
    end
    
    def append_to_response( response, context )
      response.content << @content
    end
    
    def to_s
      @content
    end
    
    def cache_copy
      copy = self.class.new(@name)
      copy.content = @content
      copy
    end
    
  end
  
  
  class CGIKitNode < Node
    
    attr_accessor :element, :ckid
    
    def initialize( name = nil )
      if ::String === name then
        name = name.intern
      end
      super
    end
    
    def take_values_from_request( request, context )
      element = element_for_context(context)
      handler(context).take_values_from_request(element, request, context)
    end
    
    def invoke_action( request, context )
      element = element_for_context(context)
      if result = handler(context).invoke_action(element, request, context) then
        set_result(result)
      end
      result
    end
    
    def set_result( result )
      @element = result
    end
    
    def append_to_response( response, context )
      element = element_for_context(context)
      unless element then return end
      handler(context).append_to_response(element, response, context)
    end
    
    def element_for_context( context )
      unless @element then
        @element = create_element(context)
        @element.node = @node
      end
      @element
    end
    
    def create_element( context )
      klass = element_class(context)
      unless klass then
        raise "Element class for #{context.parent.class}:#{content} is not specified."
      end
      
      if CGIKit::DynamicElement > klass then
        element = create_dynamic_element(klass, context)
      else
        element = create_subcomponent(klass, context)
      end
      
      init_element_for_form(element)
      element
    end
    
    def create_dynamic_element( klass, context )
      klass.new(@content, associations(context), root.component)
    end
    
    def create_subcomponent( klass, context )
      element = context.component.next_subcomponent(context.context_id)
      unless element then
        element = klass.new(context)
        element.parent = root.component
        element.context_id = context.context_id
      end
      element.declaration_name = content
      element.associations = associations(context)
      element
    end
    
    def init_element_for_form( element )
      if (CGIKitNode === @parent) and (CGIKit::DynamicElement === @parent.element) then
        if CGIKit::Component === element             
          element.parent = root.component    
        else
          element.parent = @parent.element
        end
      end
    end
    
    def element_class( context )
      type = declaration_store.element_type
      if Class === type then
        klass = type
      else
        evaluated = root.component.value_for_keypath(type)
        if Class === evaluated then
          klass = evaluated
        else
          klass = context.application.class_named(evaluated)
        end
      end
      klass
    end
    
    def declaration_store
      unless dec = root.component.declaration_store[@ckid] then
        raise "Element '#@ckid' is not declared in '#{root.component.class}'."
      end
      dec
    end
    
    def associations( context )
      declaration_store.association_hash
    end
    
    def handler( context )
      context.application.request_handler(context.request.request_handler_key)
    end
    
    def to_s
      content = super
      if content.empty? then
        "<#@name#{CGIKit::HTMLParser.attribute_string(@attributes)}/>"
      else
        "<#@name#{CGIKit::HTMLParser.attribute_string(@attributes)}>#{super}</#@name>"
      end
    end
    
    def cache_copy
      copy = super
      copy.ckid = @ckid
      copy
    end
  end
  
  
  
  class HTMLParser
    
    class ParseError < CGIKit::CGIKitError; end #:nodoc:
    
    def self.cgikit_attribute
      :ckid      
    end
    
    def self.cgikit_regexp
      /\Ack:/u
    end
    
    def self.parse_comment_reg
      /\A\s*ck\s/u
    end
    
    def self.element_attribute; CGIKit::Declaration::ELEMENT_KEY; end
    
    attr_accessor :html_string, :declarations, :node, :encoding, \
    :last_token, :next_token, :inner_nodes
    
    def initialize( filename = nil )
      @filename = filename
      
      @ckid = self.class.cgikit_attribute.to_s
      @id_reg = self.class.cgikit_regexp
      @parse_comment_reg = self.class.parse_comment_reg
      
      if filename then
        string = nil
        #CGIKit::FileLock.shared_lock(filename) do |f|
        File.open(filename) do |f|
          string = f.read
        end
        parse(string)
      end
    end
    
    def init_listener
      @buf = []
      @declarations = {}
      
      @tag_level = 1
      @node2tag_level = Hash.new(0)
      @in_doctype = false
      @html_string = ''
      
      @doctype_buf = []
    end
    
    def parse(string)
      unless Object.const_defined?('REXML')
        require 'rexml/document'
        require 'rexml/streamlistener'
        if CGIKit::Application.respond_to?(:precede_iconv_as_rexml_encoding_module) and not(CGIKit::Application.precede_iconv_as_rexml_encoding_module)
          require 'cgikit/lang/encoding-patch.rb'
        end
      end
      
      @html_string = string
      init_listener
      
      @node = CGIKit::HTMLParser::RootNode.new
      @cur = @node
      
      @parser = REXML::Parsers::BaseParser.new(string)
      begin
        __parse
      rescue REXML::ParseException
        raise CGIKit::HTMLParser::HTMLParser::ParseError, "REXML raises Error when parsing #{@filename}.\nREXML error message: #{$!.to_s.gsub(/\r?\n/, ' ')}\n"
      end
      
      if @buf.size > 0
        @cur << buffer2textnode 
      end
      
      self.node
    end
    
    #
    # use REXML::Parsers::BaseParser API
    # copied from REXML::Parsers::TreeParse and StreamParser
    
    # This is dirty but may be a bit faster.
    def __parse
      name = attrs = nil
      # entity string
      while true
        event = @parser.pull
        case event[0]
        when :end_document
          # end of XML
          return
        when :start_element
          # not normalize
          @tag_level += 1
          name = event[1]
          attrs = event[2]
          
          if match_pattern = self.cgikit_element?(name, attrs)
            if tn = buffer2textnode
              @cur << tn
            end
            
            new_node = CGIKit::HTMLParser::CGIKitNode.new(name)
            
            ck_attrs = {}
            attrs.each do |k,v|
              ck_attrs[k.intern] = self.value_for_string(v)
            end
            case match_pattern
            when :id
              ck_attrs[@ckid.intern] = attrs['id'].sub(@id_reg, '')
            when :ns_id
              ck_attrs[@ckid.intern] = attrs['ck:id']
              ck_attrs.delete(:'ck:id')
            end
            
            new_node.attributes = ck_attrs
            add_declaration(new_node.name, new_node.attributes)
            new_node.ckid = new_node.attributes[@ckid.intern].intern
            
            @node2tag_level[@cur] = @tag_level - 1
            @cur << new_node
            @cur = new_node
            @node2tag_level[new_node] = 1
            @tag_level = 1
          else
            @buf << "<#{name}#{CGIKit::HTMLParser.attribute_string(attrs)}>"
          end
        when :end_element
          @tag_level -= 1
          name = event[1]
          if @tag_level == 0
            if node = buffer2textnode
              @cur << node
            end
            
            unless RootNode === @cur
              @cur = @cur.parent
            end
            @tag_level = @node2tag_level[@cur]
          else
            if (@buf.size != 0) and (@buf.last.index("<#{name}") == 0)
              s = @buf.pop
              ss = s.sub(/>\z/um, '/>')
              @buf << ss
            else
              @buf << "</#{name}>"
            end
          end
        when :text
          unless @in_doctype
            # not normalize
            @buf << event[1]
          end
        when :end_doctype
          @in_doctype = false
          end_doctype
        when :start_doctype
          @in_doctype = true                    
          start_doctype( *event[1..-1] )
        when :processing_instruction
          instruction( event[1], event[2] )
        when :externalentity
          externalentity( event[1] )
        when :elementdecl
          elementdecl(event[1])
        when :entitydecl
          entitydecl(event)
        when :comment, :attlistdecl, :cdata, :xmldecl, :notationdecl, :entitydecl
          #__send__( event[0].to_s, *event[1..-1] )
          __send__( event[0], *event[1..-1] )
        else
          raise CGIKit::HTMLParser::HTMLParser::ParseError, "#{@filename}: CGIKit doesn't recognize the event(#{event.inspect})"
        end
      end
    end
  
    #def tag_start(name, attrs)
    #end
    #def tag_end(name)     
    #end
    #def text(text)
    #end
    
    def instruction(name, instruction)
      @buf << %Q|<?#{name}#{instruction}?>|
    end
    
    def comment(comment)
      # comment has already been converted to UTF-8.
      @buf <<  '<!--'
      
      if @parse_comment_reg =~  comment
        @cur << buffer2textnode
        
        s = comment.sub(@parse_comment_reg, '')
        # don't need to set encoding. 
        parser = REXML::Parsers::BaseParser.new(s)
        org = @parser
        @parser = parser 
        __parse
        @parser = org
      else
        @buf << comment
      end
      
      @buf << '-->'
    end
    
    def start_doctype(name, pub_sys, long_name, uri)
      if tn = buffer2textnode
        @cur << tn
      end
      
      s = ''
      s  << "<!DOCTYPE #{name} #{pub_sys}"
      if long_name
        s << ' '
        s << long_name
      end
      if uri
        s << ' '
        s << uri
      end
      
      # for the time being, "[" is used.
      s << '['
      
      @buf << s
    end
    
    def end_doctype
      if REXML::Parsers::BaseParser::DOCTYPE_START =~ @buf.last
        # There is no `markupdecl`
        s = @buf.pop
        ss = s.sub(/\[\z/um, '>')
        @buf << ss
      else
        @buf << ']>' 
      end
    end
    
    def externalentity(content)
      @buf << (content + "\n")
    end    
    
    def elementdecl(content)
      @buf << (content + '>')
    end
    
    def attlistdecl(element_name, attributes, raw_content)
      @buf << raw_content
    end
    
    def entitydecl(event)
      s = ''
      REXML::Entity.new(event).write(s)
      @buf << s
    end
    
    def notationdecl(name, middle, rest)
      @buf << "<!NOTATION #{name} '#{middle} #{rest}'>"
    end
    
    def entity(content)
      @buf << %Q|%#{content};|
    end
    
    def cdata(content)
      @buf << %Q|<![CDATA[#{content}]]>|
    end
    
    def xmldecl(version, encoding, standalone)
      s = ''
      s << %Q|<?xml version="#{version}"|
      
      if encoding
        @encoding = encoding
        s  << %Q| encoding="#{encoding}"|
      end
      
      if standalone
        s << %Q| standalone="#{standalone}"|
      end
      
      s <<  %Q|?>|
      @buf << s
    end
    
    #
    # end of BaseParser API
    #
    
    
    def cgikit_element?(tag, attrs)
      if attrs.size == 0
        false
      else	
        if attrs.key?(@ckid)
          :ckid
        elsif attrs.key?('ck:id')
          :ns_id
        elsif @id_reg =~ attrs['id'] 
          :id
        else
          false
        end
      end
    end
    
    def buffer2textnode
      if @buf.size > 0
        tn = CGIKit::HTMLParser::TextNode.new(nil)
        o = REXML::Output.new(tn.content, @encoding)
        
        # REXML::Output encodes `@buf` and add its result to `tn.content`
        o << @buf.join('')
        
        @buf.clear
        tn
      else
        nil
      end
    end
    
    def add_declaration( tag, ck_attrs )
      dec = {}
      name = ck_attrs[@ckid.intern].intern
      
      #if klass = class_for_element(tag, ck_attrs) 
      #	 dec[self.class.element_attribute] = klass
      #end			
      
      keys = ck_attrs.keys - [@ckid.intern]
      keys.each do |key|
        value = ck_attrs[key]
        if key == self.class.element_attribute then
          dec[key] = class_for_name(value)
        else
          dec[key] = value_for_string(value)
        end
      end
      
      @declarations[name] = dec
    end
    
    def class_named( name )
      CGIKit::Utilities.class_named_from(name)
    end
    alias class_for_name class_named
    
    def class_for_element( tag_type, attributes )
      case tag_type
      when :a        then Link
      when :form     then Form
      when :img      then Image
      when :textarea then Text
      when :select then
        if attributes[:multiple] or attributes.key?(:size) then
          Browser
        else
          Popup
        end
      when :input then
        unless attributes[:type] then
          return TextField
        end
        case attributes[:type].downcase
        when 'text'     then TextField
        when 'password' then TextField
        when 'hidden'   then TextField
        when 'checkbox' then Checkbox
        when 'radio'    then Radio
        when 'submit'   then Submit
        when 'reset'    then Reset
        when 'file'     then Upload
        else
          TextField
        end
      end
    end
    
    def value_for_string( value )
      case value
      when /\A:/ then
        value.sub(/\A:/, '').intern
      when /\A\s*ruby\s*:/ # it should be add `i` option
        value.sub(/\A\s*ruby\s*:/, '').intern
      when 'true' then
        true
      when 'false' then
        false
      when 'nil' then
        nil
      else
        value
      end
    end		
    
  end
  
end


if $0 == __FILE__
  require 'pp'
  
  parser = CGIKit::HTMLParser::HTMLParser.new(ARGV[0])
  #pp parser.node
  pp parser.declarations
end
module CGIKit

class DeclarationStore
  class DeclarationError < StandardError #:nodoc:
  end

  class << self
    def new_with_file( filename, source = nil )
      hash = nil
      str = nil
      open(filename) { |f| str = f.read }
      Thread.start(str) do
        $SAFE = 4
        hash = eval(str)
      end.join
      hash ||= {}
      merge_source(source, hash) if source
      new_from_hash(hash)
    end

    def merge_source( source, destination )
      source.each do |decname, dechash|
        if destination.key?(decname) then
          dechash.each do |attrname, attrvalue|
            unless destination[decname].key?(attrname) then
              destination[decname][attrname] = attrvalue
            end
          end
        else
          destination[decname] = dechash
        end
      end
    end

    def new_from_hash( hash )
      decs = new
      hash.each do |name, values|
        decs[name] = Declaration.new_from_hash(name, values)
      end
      decs.original_hash = hash
      decs
    end
  end

  attr_accessor :declarations, :original_hash

  def initialize
    @declarations = {}
  end

  def []( key )
    @declarations[key]
  end

  def []=( key, declaration )
    @declarations[key] = declaration
  end

  def each
    @declarations.each do |key, declaration|
      yield key, declaration
    end
  end

  def keys
    @declarations.keys
  end

  def validate_api( component_class = nil )
    msg = "Found validation errors in \"#{component_class}\"."
    error = ValidationError.new(msg)
    keys = @declarations.keys
    keys.map! {|key| key.to_s}
    keys.sort.each do |key|
      dec = @declarations[key.intern]
      if api = dec.element_type.api then
        errors = api.validate(dec.association_hash, component_class, key)
        if errors then
          error << errors
        end
      end
    end

    unless error.errors.empty? then
      raise error
    end
  end

end


# A Declaration object has a declaration set.
#
#  ex)
#  :BindingName => {
#    :element => String,        # element type (class)
#    :value => :'method.chain', # symbol value is dealt as method chain
#    :empty => false            # other value is dealt as constant value
#  }
#    
class Declaration
  attr_accessor :element_name, :element_type, :association_hash

  ELEMENT_KEY  = :element
  OTHER_KEY    = :other
  VALIDATE_KEY = :validate
  PASS_KEY     = :pass
  ENABLED_KEY  = :enabled

  class << self
    def new_from_hash( name, hash )
      unless hash.key?(ELEMENT_KEY) then
        raise "'#{name}' don't define #{ELEMENT_KEY.inspect}."
      end
      dec = Declaration.new(name, hash[ELEMENT_KEY])
      ass = nil
      hash.each do |key, value|
        if Symbol === value then
          ass = Association.new_with_keypath value
        else
          ass = Association.new_with_value value
        end
        dec[key] = ass
      end
      dec
    end
  end

  def initialize( name, type )
    @element_name = name
    @element_type = type
    @association_hash = {}
  end

  def []( name )
    @association_hash[name]
  end

  def []=( name, value )
    @association_hash[name] = value
  end

  def each
    @association_hash.each do |key, as|
      yield key, as
    end
  end
end

end
module CGIKit

# ByteData objects manage bytes.
class ByteData

  # Path of a file saving the bytes.
  attr_accessor :path

  # Content type of the bytes.
  attr_accessor :content_type

  class << self

    # Create an instance from specified IO object.
    # If you give this a File object, sets "path" attribute of the instance.
    def new_with_io( io, offset = nil, count = nil )
      io.pos = offset if offset
      bytes  = new io.read(count)
      if io.respond_to? 'path' then
        bytes.path = io.path
      end
      bytes
    end

    # Create an instance from specified file.
    def new_with_file( filename )
      bytes = nil
      open(filename) do |f|
        bytes      = new(f.read)
        bytes.path = f.path
        if ext = File.extname(filename) then
          ext.tr!('.', '')
          bytes.content_type = ResourceManager.mime(ext)
        end
      end
      bytes
    end

  end

  def initialize( string = nil )
    @bytes = string || ''
  end

  def tempfile?
    false
  end

  # Returns bytes with spcified length or whole length if you omit it.
  def bytes( length = nil )
    if length then
      @bytes.slice(0, length)
    else
      @bytes.to_s
    end
  end

  # Executes the block for every byte.
  def each
    @bytes.each_byte do |byte|
      yield byte
    end
  end

  # Returns true if the bytes of each objects are equal.
  def ==( bytes )
    @bytes == bytes.bytes
  end

  # Returns length of the bytes.
  def length
    @bytes.size
  end
  alias size length

  # Appends bytes to the bytes.
  def <<( bytes )
    if bytes.is_a?(ByteData) then
      @bytes << bytes.bytes
    else
      @bytes << bytes
    end
    self
  end

  # Writes the bytes to a specified file.
  def write_to_file( filename, lock = true )
    if lock then
      FileLock.exclusive_lock(filename, 'w+b') do |file|
        file.write to_s
      end
    else
      File.open(filename, 'w+b') do |file|
        file.write to_s
      end
    end      
  end

  # Returns the object as a string.
  def to_s
    @bytes.to_s
  end

  def open; end
  def close; end

end


class TempfileByteData < ByteData

  attr_accessor :tempfile

  def initialize( tempfile )
    @tempfile = tempfile
    close
  end

  def tempfile?
    true
  end

  # Returns bytes with spcified length or whole length if you omit it.
  def bytes( length = nil )
    open do
      @tempfile.read(length)
    end
  end

  # Executes the block for every byte.
  def each
    open do
      @tempfile.each_byte do |byte|
        yield byte
      end
    end
  end

  def ==( bytes )
    @tempfile == bytes.tempfile
  end

  def length
    open do
      @tempfile.size
    end
  end

  def <<( bytes )
    open do
      @tempfile.seek(0, IO::SEEK_END)
      if ByteData === bytes then
        @tempfile << bytes.bytes
      else
        @tempfile << bytes
      end
      self
    end
  end

  def to_s
    bytes
  end

  def open( &block )
    @tempfile.open if @tempfile.closed?
    if block_given? then
      @tempfile.rewind
      value = block.call
      close
      return value
    end
  end

  def close
    @tempfile.close unless @tempfile.closed?
  end

  def _dump( limit )
    Marshal.dump(bytes(), limit)
  end

  def _load( object )
    ByteData.new(Marshal.load(object))
  end

end


# ResourceManager class manages resources of an application.
class ResourceManager
  MIME = {
    'ez'      => 'application/andrew-inset',
    'hqx'     => 'application/mac-binhex40',
    'cpt'     => 'application/mac-compactpro',
    'doc'     => 'application/msword',
    'bin'     => 'application/octet-stream',
    'dms'     => 'application/octet-stream',
    'lha'     => 'application/octet-stream',
    'lzh'     => 'application/octet-stream',
    'exe'     => 'application/octet-stream',
    'class'   => 'application/octet-stream',
    'so'      => 'application/octet-stream',
    'dll'     => 'application/octet-stream',
    'oda'     => 'application/oda',
    'pdf'     => 'application/pdf',
    'ai'      => 'application/postscript',
    'eps'     => 'application/postscript',
    'ps'      => 'application/postscript',
    'smi'     => 'application/smil',
    'smil'    => 'application/smil',
    'mif'     => 'application/vnd.mif',
    'xls'     => 'application/vnd.ms-excel',
    'ppt'     => 'application/vnd.ms-powerpoint',
    'wbxml'   => 'application/vnd.wap.wbxml',
    'wmlc'    => 'application/vnd.wap.wmlc',
    'wmlsc'   => 'application/vnd.wap.wmlscriptc',
    'bcpio'   => 'application/x-bcpio',
    'vcd'     => 'application/x-cdlink',
    'pgn'     => 'application/x-chess-pgn',
    'cpio'    => 'application/x-cpio',
    'csh'     => 'application/x-csh',
    'dcr'     => 'application/x-director',
    'dir'     => 'application/x-director',
    'dxr'     => 'application/x-director',
    'dvi'     => 'application/x-dvi',
    'spl'     => 'application/x-futuresplash',
    'gtar'    => 'application/x-gtar',
    'hdf'     => 'application/x-hdf',
    'js'      => 'application/x-javascript',
    'skp'     => 'application/x-koan',
    'skd'     => 'application/x-koan',
    'skt'     => 'application/x-koan',
    'skm'     => 'application/x-koan',
    'latex'   => 'application/x-latex',
    'nc'      => 'application/x-netcdf',
    'cdf'     => 'application/x-netcdf',
    'sh'      => 'application/x-sh',
    'shar'    => 'application/x-shar',
    'swf'     => 'application/x-shockwave-flash',
    'sit'     => 'application/x-stuffit',
    'sv4cpio' => 'application/x-sv4cpio',
    'sv4crc'  => 'application/x-sv4crc',
    'tar'     => 'application/x-tar',
    'tcl'     => 'application/x-tcl',
    'tex'     => 'application/x-tex',
    'texinfo' => 'application/x-texinfo',
    'texi'    => 'application/x-texinfo',
    't'       => 'application/x-troff',
    'tr'      => 'application/x-troff',
    'roff'    => 'application/x-troff',
    'man'     => 'application/x-troff-man',
    'me'      => 'application/x-troff-me',
    'ms'      => 'application/x-troff-ms',
    'ustar'   => 'application/x-ustar',
    'src'     => 'application/x-wais-source',
    'xhtml'   => 'application/xhtml+xml',
    'xht'     => 'application/xhtml+xml',
    'zip'     => 'application/zip',
    'au'      => 'audio/basic',
    'snd'     => 'audio/basic',
    'mid'     => 'audio/midi',
    'midi'    => 'audio/midi',
    'kar'     => 'audio/midi',
    'mpga'    => 'audio/mpeg',
    'mp2'     => 'audio/mpeg',
    'mp3'     => 'audio/mpeg',
    'aif'     => 'audio/x-aiff',
    'aiff'    => 'audio/x-aiff',
    'aifc'    => 'audio/x-aiff',
    'm3u'     => 'audio/x-mpegurl',
    'ram'     => 'audio/x-pn-realaudio',
    'rm'      => 'audio/x-pn-realaudio',
    'rpm'     => 'audio/x-pn-realaudio-plugin',
    'ra'      => 'audio/x-realaudio',
    'wav'     => 'audio/x-wav',
    'pdb'     => 'chemical/x-pdb',
    'xyz'     => 'chemical/x-xyz',
    'bmp'     => 'image/bmp',
    'gif'     => 'image/gif',
    'ief'     => 'image/ief',
    'jpeg'    => 'image/jpeg',
    'jpg'     => 'image/jpeg',
    'jpe'     => 'image/jpeg',
    'png'     => 'image/png',
    'tiff'    => 'image/tiff',
    'tif'     => 'image/tiff',
    'djvu'    => 'image/vnd.djvu',
    'djv'     => 'image/vnd.djvu',
    'wbmp'    => 'image/vnd.wap.wbmp',
    'ras'     => 'image/x-cmu-raster',
    'pnm'     => 'image/x-portable-anymap',
    'pbm'     => 'image/x-portable-bitmap',
    'pgm'     => 'image/x-portable-graymap',
    'ppm'     => 'image/x-portable-pixmap',
    'rgb'     => 'image/x-rgb',
    'xbm'     => 'image/x-xbitmap',
    'xpm'     => 'image/x-xpixmap',
    'xwd'     => 'image/x-xwindowdump',
    'igs'     => 'model/iges',
    'iges'    => 'model/iges',
    'msh'     => 'model/mesh',
    'mesh'    => 'model/mesh',
    'silo'    => 'model/mesh',
    'wrl'     => 'model/vrml',
    'vrml'    => 'model/vrml',
    'css'     => 'text/css',
    'html'    => 'text/html',
    'htm'     => 'text/html',
    'asc'     => 'text/plain',
    'txt'     => 'text/plain',
    'rtx'     => 'text/richtext',
    'rtf'     => 'text/rtf',
    'sgml'    => 'text/sgml',
    'sgm'     => 'text/sgml',
    'tsv'     => 'text/tab-separated-values',
    'wml'     => 'text/vnd.wap.wml',
    'wmls'    => 'text/vnd.wap.wmlscript',
    'etx'     => 'text/x-setext',
    'xml'     => 'text/xml',
    'xsl'     => 'text/xml',
    'mpeg'    => 'video/mpeg',
    'mpg'     => 'video/mpeg',
    'mpe'     => 'video/mpeg',
    'qt'      => 'video/quicktime',
    'mov'     => 'video/quicktime',
    'mxu'     => 'video/vnd.mpegurl',
    'avi'     => 'video/x-msvideo',
    'movie'   => 'video/x-sgi-movie',
    'ice'     => 'x-conference/x-cooltalk'
  }

  TMPDIR = 'resource'
  DEFAULT_TMP_DATA_KEY_FIGURES = 16

  class << self
    def create_tmp_data_key
      md5 = Digest::MD5::new
      md5.update Time.now.to_s
      md5.update rand(0).to_s
      md5.update $$.to_s
      md5.hexdigest[0, tmp_data_key_figures]
    end

    def tmp_data_key_figures
      DEFAULT_TMP_DATA_KEY_FIGURES
    end

    def mime( extension )
      MIME[extension]
    end
  end

  def initialize( application )
    @application          = application
    @resources            = @application.resources
    @web_server_resources = @application.web_server_resources
    @document_root        = @application.document_root
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end

  # Returns the public URL for the specified resource
  # when it is under the web server resources directory.
  # Otherwise returns nil.
  def url( name, request = nil )
    url = nil
    if filepath = path(name) then
      if @web_server_resources and \
        /\A#{File.expand_path(@web_server_resources)}/ === filepath then
        url = filepath.sub(File.expand_path(@document_root), '') if @document_root
        return url
      elsif @tmpdir and /\A#{File.expand_path(@tmpdir)}/ === filepath then
        url = @application.resource_request_handler.resource_url(name, request)
      end
    end
    url
  end

  # Returns the file path of the specified resource.
  def path( name )
    resourcedir = [@tmpdir, @resources, @web_server_resources]
    resourcedir.each do |dir|
      filepath = File.join(dir, name).untaint
      absolute = File.expand_path(filepath).untaint
      if FileTest.exist?(absolute) then
        return absolute
      end
    end
    nil
  end

  # Returns a ByteData object for the specified resource.
  def bytedata( name )
    data = nil
    if filepath = path(name) then
      if filepath =~ /#{File.expand_path(@tmpdir)}/ then
        cache = nil
        FileLock.shared_lock(filepath) do |file|
          cache = Marshal.load(file)
        end
        data = ByteData.new(cache[:data])
        data.path = filepath
        data.content_type = cache[:mime]
      else
        data = ByteData.new_with_file(filepath)
        data.content_type = content_type(filepath)
      end
    end
    data
  end

  # Returns a String object for the specified resource.
  def string( name )
    bytedata(name).to_s
  end

  def write( path, content, lock = true )
    if lock then
      FileLock.exclusive_lock(path, 'wb') do |file|
        file.write(content.to_s)
      end
    else
      File.open(path, 'wb') do |file|
        file.write(content.to_s)
      end
    end      
  end

  def write_resource( name, content, lock = true )
    path = File.join(@resources, name)
    write(path, content, lock)
  end

  def write_web_server_resource( name, content, lock = true )
    path = File.join(@web_server_resources, name)
    write(path, content, lock)
  end

  # Finds the content type for extension of the specified path.
  # If the path don't have extension, returns nil.
  def content_type( path )
    # for ruby 1.6
    base = File.basename path
    type = nil
    if base =~ /\.([^\.]+)$/ then
      type = MIME[$1]
    end
    type
  end

  def set_data( data, key, mime )
    unless FileTest.directory? @tmpdir
      require 'ftools'
      File.makedirs @tmpdir
    end
    cache = {}
    cache[:data] = data
    cache[:key]  = key
    cache[:mime] = mime
    FileLock.exclusive_lock(tmpfile(key)) do |file|
      Marshal.dump(cache, file)
    end
  end

  def tmpfile( filename )
    File.join(@tmpdir, filename).untaint
  end

  def exist?( filename )
    FileTest.exist?(tmpfile(filename))
  end

  def remove_data( key )
    path = tmpfile(key)
    if FileTest.exist?(path)
      File.delete(path)
    end
  end

end

end


module CGIKit

# The super class of HTTP Request-Response classes.
class Message
  # HTML content.
  attr_accessor :content

  # Hash of HTTP headers.
  attr_accessor :headers

  # HTTP version. The default value is "1.1".
  attr_accessor :http_version

  # The encoding used for the content.
  attr_accessor :encoding

  # Array of Cookie objects.
  attr_reader   :cookies

  EOL = "\r\n"

  def initialize( headers = nil )
    @headers      = headers || {}
    @http_version = "1.1"
    @cookies      = []
    @content      = ''
  end

  # Adds a cookie object.
  def add_cookie( cookie )
    @cookies << cookie
  end

  # Removes the specified cookie in the cookies.
  def remove_cookie( cookie )
    @cookies.delete cookie
  end

  # Returns HTTP version.
  def http_version_line
    "HTTP/#@http_version"
  end
end

end
module CGIKit

class Request < Message

  attr_accessor :form_values, :session_id, :context_id, \
  :request_handler_key, :request_handler_path

  class << self
    # Parse query string and return a hash of parameters.
    def parse_query_string( query )
      params = Hash.new([])
      query.split(/[&;]/n).each do |pairs|
        key, value = pairs.split('=',2).collect{|v| Utilities.unescape_url(v) }
        if params.has_key?(key)
          params[key].push(value)
        else
          params[key] = [value]
        end
      end
      params
    end
  end

  def initialize( headers = nil, form_values = nil )
    super headers
    @cookies     = Cookie.parse_raw_cookie(@headers['HTTP_COOKIE'])
    @form_values = Hash.new([])
    @form_values.update(form_values) if form_values
    @request_handler_path = @headers['PATH_INFO']
  end

  def form_value( key )
    if @form_values.key?(key)
      @form_values[key].first
    else
      ''
    end
  end
  alias [] form_value

  def cookie( key )
    @cookies.each { | cookie |
      if cookie.name == key
        return cookie
      end
    }
    nil
  end

  def cookie_value( key )
    @cookies.each { | cookie |
      if cookie.name == key
        return cookie.value
      end
    }
    nil
  end

  def cookie_values( key = nil )
    if key then
      _cookie_values_for_key( @cookies, key )
    else
      _cookie_values @cookies
    end
  end

  def languages
    langs   = []
    if accept_language then
      langs = accept_language.split(',').collect { |entry|
        lang, quality = entry.split(';')
        if /^q=(.+)/ =~ quality
          quality = $1.to_f
        else
          quality = 1.0
        end
        [lang, quality]
      }.sort { |a, b| b[1] <=> a[1] }.collect { |i| i[0] }
    end
    langs
  end


  #
  # HTTP request headers
  #

  def accept;            @headers['HTTP_ACCEPT']          end
  def accept_charset;    @headers['HTTP_ACCEPT_CHARSET']  end
  def accept_language;   @headers['HTTP_ACCEPT_LANGUAGE'] end
  def auth_type;         @headers['AUTH_TYPE']            end
  def content_length;    @headers['CONTENT_LENGTH']       end
  def content_type;      @headers['CONTENT_TYPE']         end
  def from;              @headers['HTTP_FROM']            end
  def gateway_interface; @headers['GATEWAY_INTERFACE']    end
  def path_info;         @headers['PATH_INFO']            end
  def path_translated;   @headers['PATH_TRANSLATED']      end
  def query_string;      @headers['QUERY_STRING']         end
  def raw_cookie;        @headers['HTTP_COOKIE']          end
  def referer;           @headers['HTTP_REFERER']         end
  def remote_addr;       @headers['REMOTE_ADDR']          end
  def remote_host;       @headers['HTTP_HOST']            end
  def remote_ident;      @headers['REMOTE_IDENT']         end
  def remote_user;       @headers['REMOTE_USER']          end
  def request_method;    @headers['REQUEST_METHOD']       end
  def script_name;       @headers['SCRIPT_NAME']          end
  def server_name;       @headers['SERVER_NAME']          end
  def server_port;       @headers['SERVER_PORT']          end
  def server_protocol;   @headers['SERVER_PROTOCOL']      end
  def server_software;   @headers['SERVER_SOFTWARE']      end
  def uri;               @headers['REQUEST_URI']          end
  def user_agent;        @headers['HTTP_USER_AGENT']      end

  def accept=(value);            @headers['HTTP_ACCEPT']=value          end
  def accept_charset=(value);    @headers['HTTP_ACCEPT_CHARSET']=value  end
  def accept_language=(value);   @headers['HTTP_ACCEPT_LANGUAGE']=value end
  def auth_type=(value);         @headers['AUTH_TYPE']=value            end
  def content_length=(value);    @headers['CONTENT_LENGTH']=value       end
  def content_type=(value);      @headers['CONTENT_TYPE']=value         end
  def from=(value);              @headers['HTTP_FROM']=value            end
  def gateway_interface=(value); @headers['GATEWAY_INTERFACE']=value    end
  def path_info=(value);         @headers['PATH_INFO']=value            end
  def path_translated=(value);   @headers['PATH_TRANSLATED']=value      end
  def query_string=(value);      @headers['QUERY_STRING']=value         end
  def raw_cookie=(value);        @headers['HTTP_COOKIE']=value          end
  def referer=(value);           @headers['HTTP_REFERER']=value         end
  def remote_addr=(value);       @headers['REMOTE_ADDR']=value          end
  def remote_host=(value);       @headers['HTTP_HOST']=value            end
  def remote_ident=(value);      @headers['REMOTE_IDENT']=value         end
  def remote_user=(value);       @headers['REMOTE_USER']=value          end
  def request_method=(value);    @headers['REQUEST_METHOD']=value       end
  def script_name=(value);       @headers['SCRIPT_NAME']=value          end
  def server_name=(value);       @headers['SERVER_NAME']=value          end
  def server_port=(value);       @headers['SERVER_PORT']=value          end
  def server_protocol=(value);   @headers['SERVER_PROTOCOL']=value      end
  def server_software=(value);   @headers['SERVER_SOFTWARE']=value      end
  def uri=(value);               @headers['REQUEST_URI']=value          end
  def user_agent=(value);        @headers['HTTP_USER_AGENT']=value      end

  alias url uri
  alias url= uri=

  private

  def _cookie_values( cookies )
    values = {}
    cookies.each do |cookie|
      values[cookie.name] = cookie.value
    end
    values
  end

  def _cookie_values_for_key( cookies, key )
    values = []
    cookies.each do |cookie|
      if cookie.name == key then
        values << cookie.value
      end
    end
    values
  end
end

end
module CGIKit

# A response object that is sent to a browser by a Adapter object.
#
# == Getting a response object
# You can get a response object by Application#response or Component#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class Response < Message
  # Status code in HTTP. Default status is 200 ( OK ).
  attr_accessor :status

  attr_accessor :component

  STATUS = { 
    100 => 'Continue',
    101 => 'Switching Protocols',
    200 => 'OK',
    201 => 'Created',
    202 => 'Accepted',
    203 => 'Non-Authoritative Information',
    204 => 'No Content',
    205 => 'Reset Content',
    206 => 'Partial Content',
    300 => 'Multiple Choices',
    301 => 'Moved Permanently',
    302 => 'Found',
    303 => 'See Other',
    304 => 'Not Modified',
    305 => 'Use Proxy',
    307 => 'Temporary Redirect',
    400 => 'Bad Request',
    401 => 'Unauthorized',
    402 => 'Payment Required',
    403 => 'Forbidden',
    404 => 'Not Found',
    405 => 'Method Not Allowed',
    406 => 'Not Acceptable',
    407 => 'Proxy Authentication Required',
    408 => 'Request Timeout',
    409 => 'Conflict',
    410 => 'Gone',
    411 => 'Length Required',
    412 => 'Precondition Failed',
    413 => 'Request Entity Too Large',
    414 => 'Request-URI Too Long',
    415 => 'Unsupported Media Type',
    416 => 'Requested Range Not Satisfiable',
    417 => 'Expectation Failed',
    500 => 'Internal Server Error',
    501 => 'Not Implemented',
    502 => 'Bad Gateway',
    503 => 'Service Unavailable',
    504 => 'Gateway Timeout',
    505 => 'HTTP Version Not Supported'
  }

  def initialize( headers = nil )
    super
    @status = 200
    @headers['Content-Type'] = 'text/html'
  end

  def header
    response = ''
    response << ( __cookie || '' )
    response << ( _general_header  || '' )
    response << ( _response_header || '' )
    response << _entity_header
    response << EOL
    response
  end

  def to_s
    response = ''
    response << (self.content || '') unless redirect?
    response
  end
  private

  def _general_header
    header = ''
    header << ( __header('Cache-Control') || '' )
    header << ( __header('Connection') || '' )
    header << ( __header('Date') || '' )
    header << ( __header('Pragma') || '' )
    header << ( __header('Trailer') || '' )
    header << ( __header('Transfet-Encoding') || '' )
    header << ( __header('Upgrade') || '' )
    header << ( __header('Via') || '' )
    header << ( __header('Warning') || '' )
  end

  def _response_header
    header = ''
    header << ( __header('Accept-Ranges') || '' )
    header << ( __header('Age') || '' )
    header << ( __header('ETag') || '' )
    header << ( __header('Location') || '' )
    header << ( __header('Proxy-Authenticate') || '' )
    header << ( __header('Retry-After') || '' )
    header << ( __header('Server') || '' )
    header << ( __header('Vary') || '' )
    header << ( __header('WWW-Authenticate') || '' )
  end

  def _entity_header
    header = ''
    header << ( __header('Allow') || '' )
    header << ( __header('Content-Encoding') || '' )
    header << ( __header('Content-Language') || '' )
    header << ( __header('Content-Length') || '' )
    header << ( __header('Content-Location') || '' )
    header << ( __header('Content-MD5') || '' )
    header << ( __header('Content-Range') || '' )
    header << __content_type
    header << ( __header('Content-Disposition') || '' )
    header << ( __header('Expires') || '' )
    header << ( __header('Last-Modified') || '' )
  end

  def __header( header )
    "#{header}: #{self.headers[header]}#{EOL}" if self.headers[header]
  end

  def __content_type
    header = "Content-Type: #{self.headers['Content-Type']}"
    header << "; charset=#{self.encoding}" if self.encoding
    header << EOL
  end

  def __cookie
    return if @cookies.empty?

    header = ''
    @cookies.each { | cookie |
      header << "Set-Cookie: #{cookie.to_s}"
      header << EOL
    }
    header
  end

  public

  def status_line
    "#{http_version_line} #@status #{STATUS[@status]}#{EOL}"
  end

  # Sends a temporary redirect response to the client using the specified URL.
  def set_redirect( url )
    @status = 302
    @headers['Location'] = url
  end

  # Returns true if the response is setted redirect.
  def redirect?
    (@status == 302) or (@status == 307)
  end

  # Returns self. This method is invoked to display results in direct action.
  def generate_response
    self
  end
end

end
module CGIKit

# Cookie is a class for cookie.
# To send cookies to a browser needs to create cookie objects 
# and set them to a response object. Instead of creating cookie objects,
# you can also get cookie objects from a request object.
#
# Cookie objects have a pair of a cookie name and value.
# If you make the objects have multiple values for one name, 
# you must write code by yourself. 
# 
# == Controlling cookie objects
#
# === Creating cookies
# Give arguments of initialize() a name or a pair of name/value.
# The value of cookie is omittable.
#
#  cookie = Cookie.new( name, value )
#
# === Getting cookies from a request object
# Request has some methods for getting cookies.
# The methods are cookie(key), cookies, cookie_value(key), cookie_values(key).
# See also Request.
#
# === Setting cookies to a response object
# Response has methods for setting cookies. These methods are 
# defined in Message, the superclass of Response.
# Use add_cookie(cookie) and remove_cookie(cookie).
class Cookie
  # Name of the cookie.
  attr_accessor :name

  # Value of the cookie.
  attr_accessor :value

  # Restricts the cookie in the site.
  attr_accessor :path

  # Domain that can receive the cookie.
  attr_accessor :domain

  # Expiry date. You set Time object to the cookie object.
  # The value is formatted when the cookie is returned.
  attr_accessor :expires

  # Decides whether the cookie is encrypted or not.
  attr_accessor :secure

  class << self
    # Parse raw cookie string and return an array of cookies.
    def parse_raw_cookie( raw_cookie )
      cookies = []
      return cookies unless raw_cookie

      raw_cookie.split('; ').each do |pairs|
        name, value = pairs.split('=',2)
        name  = Utilities.unescape_url name
        value = Utilities.unescape_url value

        cookies << Cookie.new( name, value )
      end

      cookies
    end
  end

  def initialize( name, value = nil, domain = nil, path = nil, secure = false )
    @name   = name
    @value  = value
    @domain = domain
    @path   = path
    @secure = secure
  end

  def to_s
    buf = "#@name="
    if @value   then buf << Utilities.escape_url(@value.to_s) end
    if @domain  then buf << "; domain=#@domain" end
    if @path    then buf << "; path=#@path" end
    if @expires then buf << "; expires=#{Utilities.date(@expires)}" end
    if @secure == true then buf << '; secure' end
    buf
  end
end

end
module CGIKit

# Utilitis is a module wihch collects utility methods based on cgi.rb.
module Utilities
  CR            = "\015"
  LF            = "\012"
  EOL           = CR + LF
  RFC822_DAYS   = %w[ Sun Mon Tue Wed Thu Fri Sat ]
  RFC822_MONTHS = %w[ Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ]
  MEM_CONTENT_LENGTH = 10240

  def query_from_headers( headers, input )
    if ("POST" == headers['REQUEST_METHOD']) and
        %r|\Amultipart/form-data.*boundary=\"?([^\";,]+)\"?|n.match(headers['CONTENT_TYPE'])
      boundary = $1.dup
      query_from_multipart(headers, input,
                           boundary, Integer(headers['CONTENT_LENGTH']))
    else
      case headers['REQUEST_METHOD']
      when "HEAD" then query = query_string_from_head(headers)
      when "GET"  then query = query_string_from_get(headers)
      when "POST" then query = query_string_from_post(headers, input)
      else             query = query_string_from_shell end
      Request.parse_query_string(query)
    end
  end

  def query_string_from_get( headers )
    headers['QUERY_STRING'] or ""
  end

  alias :query_string_from_head :query_string_from_get

  def query_string_from_post( headers, input )
    input.binmode
    input.read(Integer(headers['CONTENT_LENGTH'])) or ''
  end

  def query_string_from_shell
    require "shellwords"
    msg = %|(offline mode: enter name=value pairs on standard input)\n|
    string = unless ARGV.empty?
               ARGV.join(' ')
             else
               if STDIN.tty?
                 STDERR.print(msg)
               end
               readlines.join(' ').gsub(/\n/n, '')
             end.gsub(/\\=/n, '%3D').gsub(/\\&/n, '%26')
    words = Shellwords.shellwords(string)

    if words.find{|x| /=/n.match(x) }
      words.join('&')
    else
      words.join('+')
    end
  end

  def query_from_multipart( headers, input, boundary, content_length )
    params   = Hash.new([])
    boundary = "--" + boundary
    buf      = ""
    bufsize  = 10 * 1024

    # start multipart/form-data
    input.binmode
    boundary_size   = boundary.size + EOL.size
    content_length -= boundary_size
    status          = input.read boundary_size
    if status == nil then
      raise EOFError, "no content body"
    elsif boundary + EOL != status
      raise EOFError, "bad content body"
    end

    until content_length == -1
      head = nil
      tmp = nil
      if content_length > MEM_CONTENT_LENGTH
        require "tempfile"
        tmp = Tempfile.new("CGIKit")
        tmp.binmode
        data = TempfileByteData.new(tmp)
      else
        data = ByteData.new
      end

      until head and /#{boundary}(?:#{EOL}|--)/n.match(buf)
        if (not head) and /#{EOL}#{EOL}/n.match(buf)
          buf = buf.sub(/\A((?:.|\n)*?#{EOL})#{EOL}/n) do
            head = $1.dup
            ""
          end
          next
        end

        if head and ( (EOL + boundary + EOL).size < buf.size )
          data << buf[0 ... (buf.size - (EOL + boundary + EOL).size)]
          buf[0 ... (buf.size - (EOL + boundary + EOL).size)] = ""
        end

        if bufsize < content_length then
          c = input.read(bufsize) or ''
        else
          c = input.read(content_length) or ''
        end
        buf += c
        content_length -= c.size
      end

      buf = buf.sub(/\A((?:.|\n)*?)(?:#{EOL})?#{boundary}(#{EOL}|--)/n) do
        data << $1
        if "--" == $2
          content_length = -1
        end
        ""
      end

      /Content-Disposition:.* filename="?([^\";]*)"?/ni.match(head)
      filename = ($1 || "").dup
      if /Mac/ni.match(headers['HTTP_USER_AGENT']) and
          /Mozilla/ni.match(headers['HTTP_USER_AGENT']) and
          (not /MSIE/ni.match(headers['HTTP_USER_AGENT']))
        filename = Utilities.unescape_url filename
      end
      data.path = filename

      /Content-Type: ([^\r\n]*)/ni.match(head)
      if $1 then
        data.content_type = $1.dup
      else
        data = data.to_s
      end

      /Content-Disposition:.* name="?([^\";]*)"?/ni.match(head)
      name = $1.dup

      if params.has_key? name then
        params[name].push data
      else
        params[name] = [data]
      end
    end

    params
  end

  # Returns an encoded string for URL.
  def escape_url( string )
    string.gsub( /([^ a-zA-Z0-9_.-]+)/n ) do
      '%' + $1.unpack( 'H2' * $1.size ).join( '%' ).upcase
    end.tr( ' ', '+' )
  end

  # Returns a string decoded from URL.
  def unescape_url( string )
    string.tr( '+', ' ' ).gsub( /((?:%[0-9a-fA-F]{2})+)/n ) do
      [ $1.delete( '%' ) ].pack( 'H*' )
    end
  end

  # Escapes HTML control characters.
  def escape_html( string )
    string.gsub(/&/n, '&amp;').
      gsub(/\"/n, '&quot;').
      gsub(/>/n, '&gt;').
      gsub(/</n, '&lt;').
      gsub(/'/n, '&#39;')
  end

  # Unescapes HTML control characters.
  def unescape_html( string )
    string.gsub(/&(.*?);/n) do
      match = $1.dup
      case match
      when /\Aamp\z/ni           then '&'
      when /\Aquot\z/ni          then '"'
      when /\Agt\z/ni            then '>'
      when /\Alt\z/ni            then '<'
      when /\A#0*(\d+)\z/n       then
        if Integer($1) < 256
          Integer($1).chr
        else
          if Integer($1) < 65536 and \
            ($KCODE[0] == ?u or $KCODE[0] == ?U)
            [Integer($1)].pack("U")
          else
            "&##{$1};"
          end
        end
      when /\A#x([0-9a-f]+)\z/ni then
        if $1.hex < 256
          $1.hex.chr
        else
          if $1.hex < 65536 and \
            ($KCODE[0] == ?u or $KCODE[0] == ?U)
            [$1.hex].pack("U")
          else
            "&#x#{$1};"
          end
        end
      else
        "&#{match};"
      end
    end
  end

  # Formats Time object in RFC1123.
  # For example, "Sat, 1 Jan 2000 00:00:00 GMT".
  def date( time )
    t = time.clone.gmtime
    return format("%s, %.2d %s %.4d %.2d:%.2d:%.2d GMT",
      RFC822_DAYS[t.wday], t.day, RFC822_MONTHS[t.month-1], t.year,
      t.hour, t.min, t.sec)
  end

  def class_named_from( name, spaces = [CGIKit, Object] )
    spaces.each do |klass|
      base = klass
      name.split("::").each do |i|
        if i.empty? then
          i = :Object
        end
        if base.const_defined?(i) then
          base = base.const_get(i)
        else
          base = nil
          break
        end
      end
      return base if base
    end
    raise "No such '#{name}' class from #{spaces.inspect}."
  end

  module_function :query_from_headers
  module_function :query_string_from_get
  module_function :query_string_from_post
  module_function :query_string_from_head
  module_function :query_string_from_shell
  module_function :query_from_multipart
  module_function :escape_url
  module_function :unescape_url
  module_function :escape_html
  module_function :unescape_html
  module_function :date
  module_function :class_named_from
end


# FileLock is for locking files.
class FileLock
  # Creates a shared file lock on a file.
  def self.shared_lock( filename, mode = 'r' )
    File.open( filename, mode ) do | io |
      io.flock File::LOCK_SH
      yield io
      io.flock File::LOCK_UN
    end
  end

  # Creates a exclusive file lock on a file.
  def self.exclusive_lock( filename, mode = 'w' )
    File.open( filename, mode ) do | io |
      io.flock File::LOCK_EX
      yield io
      io.flock File::LOCK_UN
    end
  end
end

end
module CGIKit

# == Context ID Format
# * When generating a component, 0 appends to context ID.
# * When generating a dynamic element, last element ID is incremented.
#   If in form, "name" attribute will be setted value or the last element ID.
# * In form, 0 appends to context ID.
#   The 0 last element ID is deleted at end of the form.
class Context

  SEPARATOR = '.'

  attr_accessor :request, :response, :application, :sender_id, :in_form
  attr_accessor :component, :request_handler_key

  attr_writer :session

  alias in_form? in_form

  def initialize( request = nil, application = nil )
    @request = request
    @response = Response.new
    @application = application
    @contexts = []
    @counts = []
    @session = nil
    @sender_id = request.context_id if request
  end

  def context_id( digit = false )
    if @contexts.empty? then
      ''
    elsif digit == true
      copied = @contexts.dup
      copied.pop
      copied.push(@counts.last)
      copied.join(SEPARATOR)
    else
      @contexts.join(SEPARATOR)
    end
  end

  def context_id=( id )
    delete_all
    id.to_s.split(SEPARATOR).each do |item|
      if item =~ /\A\d+\Z/ then
        @contexts << item.to_i
        @counts << item.to_i
      else
        @contexts << item
      end
    end
  end

  def element_id
    @contexts.last.to_s
  end

  def component_id
    @contexts.first
  end

  def component_id=( cid )
    @contexts[0] = cid
  end 

  def has_session?
    !@session.nil?
  end

  # Returns the session. If the receiver doesn't have a session,
  # it creates and returns a new session.
  def session
    unless has_session? then
      session = @application.restore_session(@request.session_id, self)
      unless session then
        session = @application.create_session(@request)
      end
      @session = session
      init_component_id_with_session
    end
    @session
  end

  def init_component_id_with_session( component = @component )
    unless component_id = session.component_id(component) then
      @session.save_page(component)
      component_id = session.component_id(component)
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_component_id_without_session( component )
    if @component_id then
      component_id = @component_id + 1
    else
      component_id = 0
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_context_id_and_component_id( component, component_id )
    @component = component
    self.component_id = component_id
  end

  def action?( request, context_id = nil )
    context_id ||= self.context_id
    context_id == request.context_id
  end

  def in_form=( flag )
    @in_form = flag
    if @in_form == true then
      @form_id = context_id
    else
      @form_id = nil
    end
  end

  def current_form?( request )
    @form_id and (@form_id == request.context_id)
  end


  #
  # manipulating element ID
  #

  def append( string )
    @contexts << string
    _append_zero_to_count
  end

  alias << append

  def append_zero
    @contexts << 0
    _append_zero_to_count
  end

  def delete_all
    @contexts.clear
    _delete_all_of_count
  end

  def delete
    @contexts.pop
    _delete_last_of_count
  end

  def increment( string = nil )
    if string then
      @contexts.pop
      @contexts << string
      _increment_count
    else
      _increment_count
      @contexts.pop
      @contexts << @counts.last
    end
  end

  private

  def _append_zero_to_count
    @counts << 0
  end

  def _delete_all_of_count
    @counts.clear
  end

  def _delete_last_of_count
    @counts.pop
  end

  def _increment_count
    index = @contexts.size - 1
    @counts[index] ||= 0
    @counts[index] += 1
  end

  public

  #
  # generating URLs
  #

  def direct_action_url( action_class,
                         action_name,
                         query = {},
                         contains_session_id = true )
    action_class ||= @application.direct_action_class || CGIKit::DirectAction
    app = @session || @application
    store = app.store_in_url
    app.store_in_url = contains_session_id
    url = @application.direct_action_url(self, action_class, action_name, query)
    app.store_in_url = store
    url
  end

  def component_action_url( query = {}, is_secure = false )
    complete_url(@application.component_request_handler_key, nil, query, is_secure)
  end

  def complete_url( key, path, query, is_secure, port = nil )
    @application.url(key, self, path, query, is_secure, port)
  end

  def url( key, path = nil, query = {} )
    complete_url(key, path, query, false)
  end

end

end
module CGIKit

# new_with_keypath, new_with_value
# constant?(component=nil), settable?(component=nil)
# set_value(value, component), value(component)

class Association
  class InternalInconsistencyError < StandardError #:nodoc:
  end

  attr_accessor :keypath
  attr_accessor :_value, :_constant #:nodoc:

  class << self
    def new_with_keypath( keypath )
      as = Association.new
      as.keypath = keypath
      as._constant = false
      as
    end

    def new_with_value( value )
      as = Association.new
      as._value = value
      as._constant = true
      as
    end

    def adapt_to_bool(value)
      if (value == false) or value.nil? or \
        (value == 0) or (value == 'false') then
        false
      else
        true
      end
    end
  end

  def constant?
    _constant == true
  end

  def constant_in_component?( component )
    if _constant == true then
      return false
    else
      return (component.can_set_value?(@keypath) == false)
    end
  end

  def settable?
    _constant != true
  end

  def settable_in_component?( component )
    if _constant == true then
      return false
    else
      return component.can_set_value?(@keypath)
    end
  end

  def set_value( value, component )
    if settable? then
      component.take_value_for_keypath(keypath, value)
    else
      raise InternalInconsistencyError, "The receiver's value is not settable."
    end
  end

  def value( component = nil )
    if constant? then
      _value
    elsif component.nil? then
      keypath
    else
      component.value_for_keypath(keypath)
    end
  end

  # Returns false if - null, false, 0, "false"
  def bool( component = nil )
    value = value(component)
    Association.adapt_to_bool(value)
  end

end

end
module CGIKit

# == URLs to invoke methods
# ../App.cgi/d/::            default_action on class "DirectAction"
# ../App.cgi/d/search::      search_action on class "DirectAction" or
#                            default_action on class "search"
# ../App.cgi/d/Data/search:: search_action on class "Data"
class DirectAction
  include KeyValueCoding

  DEFAULT_ACTION_NAME = 'default'
  ACTION_TEXT = '_action'
  SWEEP_ACTION_NAME = '__sweep_sessions'
  SWEEP_PASS_KEY = 'pass'
  SWEEP_PROOF = '<!-- CGIKit::DirectAction#__sweep_sessions -->'
  SWEEP_DELETED_START = '<!-- SWEEP_DELETED_START -->'
  SWEEP_DELETED_END   = '<!-- SWEEP_DELETED_END -->'
  SWEEP_FAILED_START = '<!-- SWEEP_FAILED_START -->'
  SWEEP_FAILED_END   = '<!-- SWEEP_FAILED_END -->'

  attr_reader :context, :request, :application

  def initialize( application, request )
    @application = application
    @request = request
    @context = @application.context_class.new(@request, @application)
    @handler = @application.direct_action_request_handler
  end


  #
  # accessing
  #

  def session
    unless @session then
      unless @session = existing_session then
        @session = @application.session_class.new
      end
    end
    @session
  end

  def existing_session
    unless session_id = session_id(@request) then
      return nil
    end
    session = @application.restore_session(session_id, @context)
    if session.nil? or session.terminate? then
      nil
    else
      session
    end
  end

  def session_id( request )
    @handler.session_id(request)
  end

  def page( name )
    @application.page(name, @context)
  end

  def key_for_action( action )
    action + ACTION_TEXT
  end

  def default_action_name
    DEFAULT_ACTION_NAME
  end


  #
  # performing
  #

  def perform_action( action )
    value_for_key(key_for_action(action))
  end

  def default_action
    page(@application.main)
  end


  #
  # testing
  #

  def direct_action?( action )
    respond_to?(key_for_action(action))
  end

  def self.sweep_page?( content )
    /#{SWEEP_PROOF}/ === content
  end


  #
  # taking form values
  #

  def take_form_values( keys )
    keys.each do |key|
      value = @request.form_value(key)
      take_value_for_key(key, value)
    end
  end

  def take_form_value_arrays( keys )
    keys.each do |key|
      value = @request.form_values[key]
      take_value_for_key(key, value)
    end
  end


  #
  # sweeping
  #

  def __sweep_sessions_action
    response = Response.new
    if @application.sweep_password then
      response.content = <<EOF
<html><head><title>Sweep Sessions</title></head><body>
#{SWEEP_PROOF}<form method="POST">
<input type="password" name="#{SWEEP_PASS_KEY}"/><input type="submit" value="Sweep"/>
</form>
EOF
      pass = @request.form_value('pass')
      if pass == @application.sweep_password then
        successed, failed = @application.session_store.sweep_sessions
        response.content << <<EOF
<p><strong>Deleted: #{SWEEP_DELETED_START}#{successed}#{SWEEP_DELETED_END}, 
Failed: #{SWEEP_FAILED_START}#{failed}#{SWEEP_FAILED_END}</p>
EOF
      end
      response.content << "</body></html>"
    end
    response
  end

  def self.sweep_info( content )
    /#{SWEEP_DELETED_START}([0-9]+)#{SWEEP_DELETED_END}/ === content
    deleted = $1
    /#{SWEEP_FAILED_START}([0-9]+)#{SWEEP_FAILED_END}/ === content
    failed = $1
    [deleted, failed]
  end

end

end
module CGIKit

class Browser < DynamicElement

  class << self
    def create_api
      api = API.new(:Browser)
      multiple = Binding.new(:multiple)
      multiple.value_set = Binding::BOOLEAN
      multiple.default = true
      api << multiple
      api << name_binding()
      api << display_binding()
      api << list_binding()
      api << item_binding()
      api << escape_binding()
      api << value_binding(false, false)
      api << selections_binding()
      api << selected_values_binding()
      api << enabled_binding()
      api << item_display_value_validation()
      api << existential_validation(:selected_values, :selections)
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    take_value(:list)
    take_value(:selections)
    take_value(:selected_values)
    names = [:selections, :selected_values]
    names.each do |name|
      if @values[name].nil? then
        @values[name] = []
        set_value(name, @values[name])
      end
    end

    options = values_from_request(request, context)
    if !options.empty? and context.current_form?(request) then
      if declared?(:selections) then
        @values[:selections].clear
        if declared?(:value) then
          @values[:list].each do |item|
            set_value(:item, item)
            take_value(:value)
            if options.include?(@values[:value].to_s) then
              @values[:selections] << item
            end
          end
        else
          options.each do |option|
            @values[:selections] << @values[:list][option.to_i]
          end
        end
      elsif declared?(:selected_values) then
        @values[:selected_values].replace(options)
      end
    end
  end

  def append_to_response( response, context )
    take_value(:list)
    take_value(:selections)
    take_value(:selected_values)
    take_bool(:escape)
    take_bool(:multiple)
    options = values_from_request(context.request, context)

    html = "<select name=\"#{name_value(context)}\""
    if @values[:multiple] then html << ' multiple="multiple"' end
    html << other()
    html << enabled()
    html << ">\n"

    if @values[:list] then
      @values[:list].each_with_index do |item, index|
        set_value(:item, item)
        take_value(:value)
        take_value(:display)

        html << "<option"
        if (@values[:selections] and @values[:selections].include?(item)) or \
          (options and 
             (options.include?(@values[:value].to_s) or \
              options.include?(index.to_s))) then
          html << " selected=\"selected\""
        end
        if @values[:value] then
          value = escaped_string(@values[:value], @values[:escape])
          html << " value=\"#{value}\""
        else
          html << " value=\"#{index}\""
        end
        html << ">"

        unless display = @values[:display] then
          display = item.to_s
        end
        display = escaped_string(display, @values[:escape])
        html << "#{display}</option>\n"
      end
    end
    html << '</select>'
    response.content << html
  end
end

end
module CGIKit

class Checkbox < DynamicElement

  class << self
    def create_api
      api = API.new(:Checkbox)
      api << name_binding()
      api << checked_binding()
      api << selection_binding()
      api << value_binding(false, false)
      api << enabled_binding()
      api << required_validation([:checked, :value])
      api << universal_validation(:value, :selection)
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  DEFAULT_VALUE = '1'

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    take_value(:value)
    take_value(:selection)
    take_bool(:checked)

    if value = value_from_request(request, context) then
      if declared?(:checked) then
        set_value(:checked, true)
      else
        set_value(:selection, @values[:value])
      end
    elsif context.current_form?(request) then
      if declared?(:checked) then
        if @values[:checked] == true then
          set_value(:checked, false)
        end
      else
        if @values[:value] == @values[:selection] then
          set_value(:selection, nil)
        end
      end
    end
  end

  def append_to_response( response, context )
    take_value(:value)
    take_value(:selection)
    take_bool(:checked)

    html =  "<input type=\"checkbox\" name=\"#{name_value(context)}\""
    html << %Q' value="#{escaped_string(@values[:value]) || DEFAULT_VALUE}"'
    html << other()
    if declared?(:checked) then
      if @values[:checked] == true then
        html << checked()
      end
    elsif @values[:selection] and (@values[:value] == @values[:selection]) then
      html << checked()
    end
    html << enabled()
    html << '/>'
    response.content << html
  end
end

end
module CGIKit

class Conditional < DynamicElement

  class << self
    def create_api
      api = API.new(:Conditional)
      condition = Binding.new(:condition)
      condition.required = true
      condition.value_set = Binding::BOOLEAN
      negate = Binding.new(:negate)
      negate.value_set = Binding::BOOLEAN
      negate.default = false
      api << condition
      api << negate
      api
    end
  end

  def begin_context( context )
    context.increment
    context.append_zero
  end

  def end_context( context )
    context.delete
  end

  def append_to_response( response, context )
    take_value(:condition)
    take_value(:negate)

    if (((not @values[:negate]) and @values[:condition]) or \
        ((not @values[:condition]) and @values[:negate])) then
      @node.append_to_response(response, context)
    end
  end

end

end
module CGIKit

class Content < DynamicElement

  def take_values_from_request( request, context )
    preserve_context_id(context) do
      @root.node.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    result = nil
    preserve_context_id(context) do
      result = @root.node.invoke_action(request, context)
    end
    result
  end

  def append_to_response( response, context )
    preserve_context_id(context) do
      @root.node.append_to_response(response, context)
    end
  end

  def preserve_context_id( context, &block )
    before = context.context_id
    context.context_id = @root.parent_context_id
    block.call
    context.context_id = before
  end

end

end
module CGIKit

class Form < DynamicElement

  MULTIPART_FORM_DATA = "multipart/form-data"

  class << self
    def create_api
      api = API.new(:Form)
      method = Binding.new(:method)
      method.default = 'POST'
      enctype = Binding.new(:enctype)
      query = Binding.new(:query)
      query.default = :'{}'
      upload = Binding.new(:upload)
      upload.value_set = Binding::BOOLEAN
      upload.default = false
      action = Binding.new(:action)

      api << method
      api << enctype
      api << href_binding()
      api << query
      api << upload
      api << action
      api << name_binding()
      api << session_id_binding()
      api << existential_validation(:href, :action)
      api << existential_validation(:enctype, :upload)
      set_direct_action(api)
      api
    end
  end

  attr_accessor :has_action_in_container

  def init
    @has_action_in_container = false
  end

  def notify_existing_action_in_container
    @has_action_in_container = true
  end

  def begin_context( context )
    take_value(:name)
    unless direct_action? then
      context.session.save_page(@root)
      context.component_id = context.session.component_id(@root)
    end
    context.increment
    context.append_zero
    context.in_form = true
  end

  def end_context( context )
    context.delete
    context.in_form = false
  end

  def take_values_from_request( request, context )
    @node.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    take_value(:action, false)
    before = context.context_id
    result = @node.invoke_action(request, context)
    if !@has_action_in_container and \
      context.action?(request, before) and @values[:action] then
      result = @root[@values[:action]]
    end
    result
  end

  def append_to_response( response, context )
    take_value(:method)
    take_value(:enctype)
    take_value(:href)
    take_value(:query)
    take_bool(:upload)
    take_value(:session_id)
    take_value(:direct_action, false)
    take_value(:action_class)

    unless (@values[:method].downcase == 'post') or \
           (@values[:method].downcase == 'get') then
      @values[:method] = 'post'
    end
    if @values[:upload] == true then
      @values[:enctype] = MULTIPART_FORM_DATA
    end

    html =  "<form name=\"#{name_value(context)}\" method=\"#{@values[:method]}\""
    if @values[:href] then
      html << " action=\"#{@values[:href]}\""
    elsif direct_action? then
      url = context.direct_action_url(@values[:action_class],
                                      @values[:direct_action],
                                      {},
                                      @values[:session_id])
      html << " action=\"#{url}\""
    else
      url = context.component_action_url()
      html << " action=\"#{url}\""
    end
    if @values[:enctype] then
      html << " enctype=\"#{@values[:enctype]}\""
    end

    html << other()
    html << ">\n"
    if @values[:query] then
      html << hidden_fields(@values[:query])
    end
    response.content << html

    @node.append_to_response(response, context)
    response.content << "\n</form>\n"
  end

  def hidden_fields( query )
    fields = ''
    query.each do | key, value |
      value = Utilities.escape_html(value.to_s)
      fields << \
        "<input type=\"hidden\" name=\"#{key}\" value=\"#{value}\"/>\n"
    end
    fields
  end
end

end
module CGIKit

class Frame < DynamicElement

  class << self
    def create_api
      api = API.new(:Frame)
      api << page_binding()
      api << src_binding()
      api << value_binding(false)
      api << required_validation([:page, :src, :value])
      api
    end
  end

  def begin_context( context )
    context.increment
  end

  def invoke_action( request, context )
    result = nil
    if context.action?(request) then
      result = context.session.component(context.context_id)
    end
    result
  end

  def append_to_response( response, context )
    take_value(:src)

    src = nil
    if @values[:src] then
      src = values[:src]
    else
      src = context.url(context.request.request_handler_key)
    end
    html =  "<frame src=\"#{src}\""
    html << other()
    html << "/>"
    response.content << html

    # the reason is if a browser accesses each frames concurrently,
    # context IDs may be repeated.
    generate_and_register_components(context)
  end

  def generate_and_register_components( context )
    if declared?(:page) or declared?(:value) then
      if declared?(:page) then
        take_value(:page)
        page = application.page(@values[:page], context)
      elsif declared?(:value) then
        page = value(:value)
      end
      session = context.session
      session.save_page(page)
      component_id = session.component_id(page)
      session.frame_components[context.context_id] = component_id
    end
  end

end

end
module CGIKit

class GenericElement < DynamicElement

  NO_CLOSE_TAGS = [ 'area', 'base', 'basefont', 'br', 'col', 'frame', 'hr',
                    'img', 'input', 'link', 'map', 'meta', 'param' ]
  NAME_TAGS = ['frame', 'form', 'input', 'select', 'textarea', 'button',
               'a', 'img', 'map', 'applet', 'iframe', 'meta', 'object', 'param']
  VALUE_TAGS = ['param', 'li', 'input', 'option', 'button']
  HREF_TAGS = ['a', 'area']

  class << self
    def create_api
      api = API.new(:GenericElement)
      tag = Binding.new(:tag)
      tag.required = true
      displayed = Binding.new(:displayed)
      displayed.value_set = Binding::BOOLEAN
      displayed.default = true
      form_value = Binding.new(:form_value)
      form_values = Binding.new(:form_values)
      invoke_action = Binding.new(:invoke_action)
      api << tag
      api << displayed
      api << form_value
      api << form_values
      api << invoke_action
      api << name_binding()
      api
    end
  end


  #
  # testing
  #

  def form?
    @values[:tag].downcase == 'form'
  end

  def link?
    @values[:tag].downcase == 'a'
  end

  def no_close_tag?( tag )
    NO_CLOSE_TAGS.include?(tag)
  end

  def has_name?( tag )
    NAME_TAGS.include?(tag)
  end

  def has_value?( tag )
    VALUE_TAGS.include?(tag)
  end

  def has_href?( tag )
    HREF_TAGS.include?(tag)
  end


  #
  # request-response loop
  #

  def begin_context( context )
    take_value(:name)
    take_value(:tag)
    context.increment(@values[:name])
    if form? then
      context.append_zero
      context.in_form = true
    elsif link? then
      context.append_zero
    end
  end

  def end_context( context )
    if form? then
      context.delete
      context.in_form = false
    elsif link? then
      context.delete
    end
  end

  def take_values_from_request( request, context )
    if form_values = values_from_request(request, context) then
      if declared?(:form_value) then
         set_value(:form_value, form_values[0])
      elsif declared?(:form_values) then
         set_value(:form_values, form_values)
      end
    end
    @node.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    take_value(:invoke_action, false)
    result = nil
    if declared?(:invoke_action) then
      if (link? and context.action?(request)) or \
        value_from_request(request, context) then
        result = @root[@values[:invoke_action]]
      end
    else
      result = @node.invoke_action(request, context)
    end
    result
  end

  def append_to_response( response, context )
    take_bool(:displayed)
    take_value(:form_value)
    take_value(:form_values)
    unless @values[:displayed] then return end

    if @values[:form_value] then
      value = @values[:form_value]
    elsif @values[:form_values] then
      value = @values[:form_values]
    end

    html =  "<#{@values[:tag]}"
    if has_name?(@values[:tag]) then
      html << " name=\"#{name_value(context)}\""
    end
    if has_value?(@values[:tag]) and \
      (declared?(:form_value) or declared?(:form_values)) then
      html << " value=\"#{value}\""
    end
    if has_href?(@values[:tag]) and declared?(:invoke_action) then
      url = context.url(context.request.request_handler_key)
      html << " href=\"#{url}\""
    end
    html << other()
    if no_close_tag?(@values[:tag]) then
      html << '/>'
    else
      html << '>'
    end
    response.content << html
    unless empty? then
      @node.append_to_response(response, context)
    end

    unless no_close_tag?(@values[:tag]) then
      response.content << "</#{@values[:tag]}>"
    end
  end

end

end
module CGIKit

class Image < DynamicElement

  class << self
    def create_api
      api = API.new(:Image)
      data = Binding.new(:data)
      file = Binding.new(:file)
      file.value_set = Binding::RESOURCES
      mime = Binding.new(:mime)
      mime.value_set = Binding::MIME_TYPES

      api << data
      api << file
      api << mime
      api << src_binding()
      api << required_validation([:file, :src, :data])

      api
    end
  end

  def append_to_response( response, context )
    take_value(:file)
    take_value(:src)
    take_value(:data)

    if @values[:file] then
      @values[:src] = @application.resource_manager.url(@values[:file])
    elsif @values[:data] then
      take_value(:mime)
      key = ResourceManager.create_tmp_data_key
      @application.resource_manager.set_data(@values[:data], key, @values[:mime])
      @values[:src] = @application.resource_manager.url(key, context.request)
    end

    html = "<img src=\"#{@values[:src]}\""
    html << other()
    html << '/>'
    response.content << html
  end
end

end
module CGIKit

class Link < DynamicElement

  class << self
    def create_api
      api = API.new(:Link)
      page = Binding.new(:page)
      page.value_set = Binding::PAGE_NAMES
      string = Binding.new(:string)
      secure = Binding.new(:secure)
      secure.value_set = Binding::BOOLEAN
      secure.default = false
      query = Binding.new(:query)
      query.default = :'{}'

      api << action_binding()
      api << enabled_binding()
      api << href_binding()
      api << escape_binding()
      api << page
      api << string
      api << secure
      api << query
      api << session_id_binding()
      set_direct_action(api)
      api << required_validation([:action, :href, :page], [], \
        [:direct_action, :action_class])
      api
    end
  end

  def begin_context( context )
    context.increment
    context.append_zero
  end

  def end_context( context )
    context.delete
  end

  def invoke_action( request, context )
    take_value(:page)
    take_value(:action, false)
    result = nil
    if context.action?(request) then
      if @values[:page] then
        result = @application.page(@values[:page], context)
      else
        result = @root[@values[:action]]
      end
    end
    result
  end

  def append_to_response( response, context )
    take_value(:name)
    take_bool(:enabled)
    take_bool(:escape)
    take_value(:href)
    take_value(:page)
    take_value(:action, false)
    take_value(:string)
    take_bool(:secure)
    take_value(:query)
    take_value(:session_id)
    take_value(:direct_action, false)
    take_value(:action_class)

    if @values[:string] then
      @values[:string] = escaped_string(@values[:string], @values[:escape])
    end

    unless @values[:enabled] then
      if @values[:string] then
        response.content << @values[:string]
      elsif empty? == false then
        @node.append_to_response(response, context)
      end
      return
    end

    take_value_for_query_from_attributes
    if @values[:page] or @values[:action] then
      # create session if it doesn't have session
      context.session
      url = context.component_action_url(@values[:query], @values[:secure])
    elsif direct_action? then
      url = context.direct_action_url(@values[:action_class],
                                      @values[:direct_action],
                                      @values[:query],
                                      @values[:session_id])
    end

    html = '<a'
    if @values[:href] then
      html << " href=\"#{@values[:href]}\""
    else
      html << " href=\"#{url}\""
    end
    html << other()
    html << '>'

    response.content << html
    if empty? == false then
      @node.append_to_response(response, context)
    elsif @values[:string] then
      response.content << @values[:string]
    end

    response.content << '</a>'
  end

  def take_value_for_query_from_attributes
    associations.reject! do |key, as|
      if (api.has_binding?(key) == false) and (key.to_s =~ /\A\?/) then
        query_value = value(key)
        query_key = key.to_s.sub(/\A\?/, '')
        @values[:query][query_key] = query_value
        true
      else
        false
      end
    end
  end

end

end
module CGIKit

class Popup < DynamicElement

  class << self
    def create_api
      api = API.new(:Popup)
      default = Binding.new(:default)
      selected_value = Binding.new(:selected_value)
      api << default
      api << selected_value
      api << name_binding()
      api << display_binding()
      api << list_binding()
      api << item_binding()
      api << escape_binding()
      api << value_binding(false, false)
      api << selection_binding()
      api << enabled_binding()
      api << item_display_value_validation()
      api << existential_validation(:selected_value, :selection)
      api
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    take_value(:list)
    if (option = value_from_request(request, context)) and \
      context.current_form?(request) then
      if declared?(:selection) then
        if option.empty? then
          selection = nil
        elsif declared?(:value)
          @values[:list].each do |item|
            set_value(:item, item)
            take_value(:value)
            if option == @values[:value].to_s then
              selection = item
              break
            end
          end
        else
          selection = @values[:list][option.to_i]
        end
        set_value(:selection, selection)
      elsif declared?(:selected_value) then
        set_value(:selected_value, option)
      end
    end
  end

  def append_to_response( response, context )
    take_value(:list)
    take_bool(:escape)
    take_value(:default)
    take_value(:selection)
    take_value(:selected_value)
    option = value_from_request(context.request, context)

    html = "<select name=\"#{name_value(context)}\""
    html << other()
    html << enabled()
    html << ">\n"

    if @values[:default] then
      html << '<option value=""'
      if @values[:selection].nil? then
        html << selected()
      end
      html << ">#{@values[:default]}</option>\n"
    end

    if @values[:list] then
      @values[:list].each_with_index do |item, index|
        set_value(:item, item)
        take_value(:value)
        take_value(:display)

        html << "<option"
        if (@values[:selection] == item) or \
          (option and !option.empty? and
            ((option == @values[:value].to_s) or \
            (option == index.to_s))) then
          html << selected()
        end
        if @values[:value] then
          value = escaped_string(@values[:value], @values[:escape])
          html << " value=\"#{value}\""
        else
          html << " value=\"#{index}\""
        end
        html << ">"

        unless display = @values[:display] then
          display = item.to_s
        end
        display = escaped_string(display, @values[:escape])
        html << "#{display}</option>\n"
      end
    end
    html << '</select>'
    response.content << html
  end
end

end
module CGIKit

class Radio < DynamicElement
  class << self
    def create_api
      api = API.new(:Radio)
      api << name_binding()
      api << checked_binding()
      api << selection_binding()
      api << value_binding(false, false)
      api << enabled_binding()
      api << required_validation([:checked, :value])
      api << universal_validation(:value, :selection)
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  DEFAULT_VALUE = '1'

  def begin_context( context )
    take_value(:name)
    context.increment
    context.append_zero
    context.increment(@values[:name])
  end

  def end_context( context )
    context.delete
  end

  def take_values_from_request( request, context )
    take_bool(:checked)
    take_value(:value)
    take_value(:selection)

    if value = value_from_request(request, context) then
      if declared?(:checked) then
        set_value(:checked, true)
      elsif value == @values[:value] then
        set_value(:selection, @values[:value])
      end
    elsif context.current_form?(request) then
      if declared?(:checked) then
        if @values[:checked] == true then
          set_value(:checked, false)
        end
      else
        if @values[:value] == @values[:selection] then
          set_value(:selection, nil)
        end
      end
    end
  end

  def append_to_response( response, context )
    take_bool(:checked)
    take_value(:value)
    take_value(:selection)

    html = "<input type=\"radio\" name=\"#{name_value(context)}\""
    if @values[:value] then
      @values[:value] = escaped_string(@values[:value])
    end
    html << %Q' value="#{@values[:value] || DEFAULT_VALUE}"'
    if declared?(:checked) then
      if @values[:checked] == true then
        html << checked()
      end
    elsif @values[:selection] and (@values[:value] == @values[:selection]) then
      html << checked()
    end
    html << other()
    html << enabled()
    html << '/>'
    response.content << html
  end
end

end
module CGIKit

class Repetition < DynamicElement

  class << self
    def create_api
      api = API.new(:Repetition)
      index = Binding.new(:index)
      count = Binding.new(:count)
      key = Binding.new(:key)
      key.settable = true
      api << list_binding(false)
      api << item_binding()
      api << index
      api << count
      api << key
      api << required_validation([:count, :list])
      api
    end
  end

  def begin_context( context )
    context.increment
    context.append_zero
  end

  def end_context( context )
    context.delete
  end

  def take_values_from_request( request, context )
    take_value(:index, false)
    take_value(:count)
    take_value(:list)

    each do |node|
      node.take_values_from_request(request, context)
      increment_context_for_next_loop(context)
    end
  end

  def increment_context_for_next_loop( context )
    context.delete
    context.increment
    context.append_zero
  end

  def invoke_action( request, context )
    result = nil
    each do |node|
      if node_result = node.invoke_action(request, context) then
        result = node_result
      end
      increment_context_for_next_loop(context)
    end
    result
  end

  def append_to_response( response, context )
    take_value(:list)
    take_value(:index, false)
    take_value(:count)

    each do |node|
      node.append_to_response(response, context)
      increment_context_for_next_loop(context)
    end
  end

  def each
    if list = @values[:list] then
      if Hash === list then
        list.each_with_index do |item, index|
          set_value(:key, item[0])
          set_value(:item, item[1])
          set_value(:index, index) if @values[:index]
          yield @node
        end
      else
        index = 0
        list.each do |item|
          set_value(:item, item)
          set_value(:index, index) if @values[:index]
          yield @node
          index += 1
        end
      end
    elsif @values[:count] then
      @values[:count].times do |index|
        set_value(:index, index) if @values[:index]
        yield @node
      end
    end
  end

end

end
module CGIKit

class Reset < DynamicElement

  class << self
    def create_api
      api = API.new(:Reset)
      value = Binding.new(:value)
      value.default = 'Reset'
      api << value
      api << enabled_binding()
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def append_to_response( response, context )
    take_value(:value)
    html  = "<input type=\"reset\" name=\"#{name_value(context)}\" value=\"#{escaped_string(@values[:value])}\""
    html << other()
    html << enabled()
    html << "/>"
    response.content << html
  end

end

end
module CGIKit

class String < DynamicElement

  class << self
    def create_api
      api = API.new(:String)
      empty = Binding.new(:empty)
      br = Binding.new(:br)
      br.value_set = Binding::BOOLEAN
      api << value_binding()
      api << escape_binding()
      api << empty
      api << br
      api
    end
  end

  def append_to_response( response, context )
    take_value(:value)
    take_value(:empty)
    take_bool(:escape)
    take_bool(:br)

    str = ''
    if @values[:value] then
      str = @values[:value].to_s
    elsif @values[:empty] then
      str = @values[:empty].to_s
    end
    str = escaped_string(str, @values[:escape])
    if @values[:br] then
      str.gsub!(/(\r\n|\r|\n)/, "<br />")
    end
    response.content << str
  end
end

end
module CGIKit

class Submit < DynamicElement
  class << self
    def create_api
      api = API.new(:Sumbit)
      action = Binding.new(:action)
      value = Binding.new(:value)
      value.default = 'Submit'
      api << action
      api << value
      api << enabled_binding()
      api << name_binding()
      set_direct_action(api)
      api << any_validation([:action], [], [:direct_action, :action_class])
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def invoke_action( request, context )
    take_value(:action, false)
    take_value(:direct_action, false)
    take_value(:action_class)

    result = nil
    if value_from_request(request, context) then
      if @values[:action] then
        result = @root[@values[:action]]
        notify_existing_action_to_form
      elsif @values[:direct_action] or @values[:action_class] then
        result = perform_direct_action(request, @values[:action_class],
          @values[:direct_action])
        notify_existing_action_to_form
      end
    end
    result
  end

  def notify_existing_action_to_form
    ancestor = @parent
    while ancestor do
      if Form === ancestor then
        ancestor.notify_existing_action_in_container
        break
      end
      ancestor = ancestor.parent
    end
  end

  def perform_direct_action( request, klass, action = DirectAction::DEFAULT_ACTION )
    unless klass then
      klass = @application.direct_action_class
    end
    direct_action = klass.new(@application, request)
    direct_action.perform_action(action)
  end

  def append_to_response( response, context )
    take_value(:value)

    if @values[:value] then
      @values[:value] = escaped_string(@values[:value])
    end
    html = "<input type=\"submit\" name=\"#{name_value(context)}\" value=\"#{@values[:value]}\""
    html << enabled()
    html << other()
    html << "/>"
    response.content << html
  end

end

end
module CGIKit

class Switcher < DynamicElement
  class << self
    def create_api
      api = API.new(:Switcher)
      component = Binding.new(:component)
      component.value_set = Binding::PAGE_NAMES
      component.required = true
      api << component
      api
    end
  end

  def begin_context( context )
    take_value_once(:component)
    context.increment
    unless @switch then
      page = context.session.restore_page(context.context_id)
      if page == root then
        @switch = application.page(@values[:component], context)
        @switch.parent = root
      else
        @switch = page
      end
      @switch.node = @node
      @switch.declaration_name = @name
      @switch.awake_from_restroration(context)
    end
    @switch.begin_context(context)
  end

  def end_context( context )
    @switch.end_context(context)
  end

  def take_values_from_request( request, context )
    @switch.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    @switch.invoke_action(request, context)
  end

  def append_to_response( response, context )
    @switch.append_to_response(response, context)
  end
end

end
module CGIKit

class Text < DynamicElement

  class << self
    def create_api
      api = API.new(:Text)
      api << name_binding()
      api << value_binding(true, true)
      api << enabled_binding()
      set_validation(api)
      api
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    if value = value_from_request(request, context) then
      set_value(:value, value)
      validate(:value)
    end
  end

  def append_to_response( response, context )
    take_value(:value)
    html = "<textarea name=\"#{name_value(context)}\""
    html << other()
    html << enabled()
    html << '>'
    if @values[:value] then html << escaped_string(@values[:value]) end
    html << '</textarea>'
    response.content << html
  end
end

end
module CGIKit

class TextField < DynamicElement
  class << self
    def create_api
      api = API.new(:TextField)
      type = Binding.new(:type)
      type.default = 'text'
      size = Binding.new(:size)
      maxlength = Binding.new(:maxlength)
      api << type
      api << size
      api << maxlength
      api << name_binding()
      api << value_binding(true, true)
      api << enabled_binding()
      set_validation(api)
      api
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    if value = value_from_request(request, context) then
      set_value(:value, value)
      validate(:value)
    end
  end

  def append_to_response( response, context )
    take_value(:value)
    take_value(:type)
    take_value(:size)
    take_value(:maxlength)

    html = "<input name=\"#{name_value(context)}\" type=\"#{@values[:type]}\""
    if @values[:value] and (@values[:type] != 'password') then
      html << " value=\"#{escaped_string(@values[:value])}\""
    end
    if @values[:size] then
      html << " size=\"#{@values[:size]}\""
    end
    if @values[:maxlength] then
      html << " maxlength=\"#{@values[:maxlength]}\""
    end
    html << other()
    html << enabled()
    html << '/>'
    response.content << html
  end
end

end
module CGIKit

class Upload < DynamicElement

  class << self
    def create_api
      api = API.new(:Upload)
      api << value_binding()
      api << enabled_binding()
      set_validation(api)
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    if (value = value_from_request(request, context)) and (ByteData === value) then
      set_value(:value, value)
      validate(:value)
    end
  end

  def append_to_response( response, context )
    html  = "<input type=\"file\" name=\"#{name_value(context)}\""
    html << other()
    html << enabled()
    html << "/>"
    response.content << html
  end
end

end
