require 'strscan'

module CKWiki

class HTMLRenderer
  include CGIKit::Utilities

  def initialize( page, context )
    @page = page
    @context = context
  end

  def render
    content    = content_for_rendering
    html       = ''
    ul_nested  = 0
    ol_nested  = 0
    dl         = false
    pre        = false
    blockquote = false
    table      = false
    scanner    = StringScanner.new(content)
    until scanner.eos? do
      scanner.scan(/\A[\r\n\t]+/)
      render_heading(scanner, html)
      ul_nested  = render_ul_list(scanner, html, ul_nested)
      ol_nested  = render_ol_list(scanner, html, ol_nested)
      dl         = render_dl_list(scanner, html, dl)
      pre        = render_pre_format(scanner, html, pre)
      blockquote = render_blockquote(scanner, html, blockquote)
      table      = render_table(scanner, html, table)
      render_paragraph(scanner, html)
    end
    html
  end

  def content_for_rendering
    content = @page.content.dup
    content.gsub!(/\r\n/, "\n")
    content.gsub!(/\r/, "\n")
    content << "\n\n"
    content
  end

  def render_heading( scanner, html )
    for x in 1..6
      marks = '!' * x
      reg = /^#{marks}([^!\n]*)\n/
      if str = scanner.scan(reg) then
        str.gsub!(/^#{marks}/, '')
        str.chop!
        str = render_inline str
        html << "<h#{x+1}>#{str}</h#{x+1}>\n"
      end
    end
  end

  def render_ul_list( scanner, html, nested )
    render_list(scanner, html, nested, 'ul')
  end

  def render_ol_list( scanner, html, nested )
    render_list(scanner, html, nested, 'ol')
  end

  def render_list( scanner, html, nested, type )
    mark, tag = list_type(type)
    nested = render_list_for_level(scanner, html, nested, mark, tag)
    if scanner.match?(/\A\s*[^#{mark}]/) then
      if nested then
        html << ("</#{tag}>\n" * nested)
        nested = 0
      end
    end
    nested
  end

  def render_list_for_level( scanner, html, nested, mark, tag )
    if str = scanner.scan(/\A(#{mark}+)[^#{mark}]([^\n]*)\n/) then
      level = scanner[1].size
      if nested == 0 then
        nested += level
        html << ("<#{tag}>\n" * level)
      else
        case level
        when 1
          if nested >= 2 then
            nested = 1
            html << ("</#{tag}>\n" * nested)
          end
        when 2
          if nested == 1 then
            nested += 1
            html << "<#{tag}>\n"
          elsif nested >= 3 then
            nested = 2
            html << "</#{tag}>\n"
          end
        when 3
          if nested <= 2 then
            nested += 3 - nested
            html << ("<#{tag}>\n" * (4 - nested))
          end
        end
      end
      str.gsub!(/^#{mark*level}/, '')
      str.chop!
      str = render_inline(str)
      html << "<li>#{str}\n"
    end
    nested
  end

  def list_type( type )
    if type == 'ul' then
      mark = "\\*"
      tag  = "ul"
    else
      mark = "#"
      tag  = "ol"
    end
    [mark, tag]
  end

  def render_dl_list( scanner, html, dl )
    if str = scanner.scan(/\A:[^\n]*\n/) then
      unless dl then
        dl = true
        html << "<dl>\n"
      end

      str =~ /\A:([^:]*):(.*)/
      dt = $1
      dd = $2
      if dt and dd then
        dt = render_inline(dt)
        dd = render_inline(dd)
        html << "<dt>#{dt}</dt><dd>#{dd}</dd>\n"
      else
        html << "#{str}\n"
      end
    end

    if scanner.match?(/\A[^:]/) and dl then
      dl = false
      html << "</dl>\n"
    end

    dl
  end

  def render_pre_format( scanner, html, pre )
    if str = scanner.scan(/\A[ ]+([^\n]*)\n/) then
      str.gsub!(/\A /, '')
      unless pre then
        pre = true
        html << '<pre>'
      end
      str = escape_html(str)
      html << str
    elsif str = scanner.scan(/\A\{\{\{(.*)\}\}\}/m) then
      str.gsub!(/\A\{\{\{(.*)\}\}\}/m) do $1 end
      str = escape_html(str)
      html << "<pre>#{str}</pre>\n"
    end

    if scanner.match?(/\A[^ ]/) then
      if pre then
        html << "</pre>\n"
        pre = false
      end
    end

    pre
  end

  def render_blockquote(scanner, html, blockquote)
    if str = scanner.scan(/\A\"\"([^\n]*)\n/) then
      unless blockquote then
        blockquote = true
        html << "<blockquote>\n"
      end

      str.gsub!(/^\"\"/, '')
      str.chop!
      str = render_inline str
      html << str
    end

    if scanner.match?(/\A[^\"]/m) then
      if blockquote == true then
        blockquote = false
        html << "\n</blockquote>\n"
      end
    end

    blockquote
  end

  def render_table( scanner, html, table )
    if str = scanner.scan(/\A\|\|([^\n]*)\n/) then
      unless table then
        table = true
        html << "<table>\n"
      end

      html << '<tr>'
      str.chop!
      str.gsub!(/\A\|\|(.*)\Z/) do $1 end
      columns = str.split('||')
      columns.each do |column|
        column = render_inline(column)
        html << "<td>#{column}</td>"
      end
      html << "</tr>\n"

      if !scanner.match?(/\A[\|]/) and table then
        table = false
        html << "</table>\n"
      end
    end
    table
  end

  def render_paragraph( scanner, html )
    if str = scanner.scan(/\A[^:!\*#\s\"\|](.*?)\n{2,}/m) then
    #if str = scanner.scan(/\A(.*)\n{2,}/m) then
      str.gsub!(/\s+$/, '')
      str.tr!("\r\n\t", '')
      str = render_inline(str)
      html << "<p>#{str}</p>\n" if str =~ /[^\s]/
    end
  end

  def render_inline( string )
    html = ''
    scanner = StringScanner.new(string)
    until scanner.eos?
      if render_inline_strong(scanner, html) then next end
      if render_inline_wikiname(scanner, html) then next end
      if render_inline_url(scanner, html) then next end
      if render_inline_specified_wikiname(scanner, html) then next end
      if render_inline_resource_url(scanner, html) then next end
      if render_inline_resource(scanner, html) then next end
      if render_inline_code(scanner, html) then next end
      unless scanner.eos?
        str = scanner.getch
        escape_html(str)
        html << str
      end
    end
    html
  end

  def render_inline_strong( scanner, html )
    if str = scanner.scan(/\A(''(?!'{1,}))(.*?)''/) then
      str.gsub!(/''(.*)''/) do $1 end
      escape_html(str)
      html << "<strong>#{str}</strong>"
      return true
    end
  end

  def render_inline_wikiname( scanner, html )
    if scanner.match?(/\A[A-Za-z0-9]+(#(.*))?/) then
      if str = scanner.match?(/\Ahttp:\/\//) then
      elsif str = scanner.scan(/\A([A-Z][a-z0-9]+){2,}/) then
        html << @context.wiki_url(str)
        return true
      elsif str = scanner.scan(/\A([a-z][A-Za-z0-9]+)/) then
        html << str
        return true
      end
    end
  end

  def render_inline_url( scanner, html )
    url_reg = 'http:\/\/[\/A-Za-z0-9_\.,\?:&=;\-\+\$@~\*!\']+'
    if str = scanner.scan(/\A#{url_reg}/) then
      html << "<a href=\"#{str}\">#{str}</a>"
      return true
    end
  end

  def render_inline_specified_wikiname( scanner, html )
    if str = scanner.scan(/\A\[\[([^\[\]]+)\]\]/) then
      str.gsub!(/\[\[(.*)\]\]/) do $1 end
      if /\|/ === str then
        body, name = str.split(/\|/, 2)
        escape_html(body)
        html << @context.wiki_url(name, body)
      else
        html << @context.wiki_url(str)
      end
      return true
    end
  end

  def render_inline_resource_url( scanner, html )
    if name = scanner.scan(/\A<<<(.*?)>>>/) then
      name.gsub!(/<<<(.*)>>>/) do $1 end
      if /\Ahhttp:\/\// === name then
        url = name
      else
        url = @context.application.resource_manager.url(name)
      end
      html << "<a href=\"#{url}\">#{name}</a>"
      return true
    end
  end

  def render_inline_resource( scanner, html )
    if name = scanner.scan(/\A<<(.*?)>>/) then
      name.gsub!(/<<(.*)>>/) do $1 end
      if /\Ahhttp:\/\// === name then
        url = name
      else
        url = @context.application.resource_manager.url(name)
      end

      if /\.(jpg|png|gif)\Z/=== url
        html << "<img src=\"#{url}\">"
      else
        content = @context.application.resource_manager.bytedata(name).to_s
        html << "<pre>#{content}</pre>"
      end
      return true
    end
  end

  def render_inline_code( scanner, html )
    if str = scanner.scan(/\A=(.*?)=/) then
      str.gsub!(/=(.*)=/) do $1 end
      escape_html(str)
      html << "<code>#{str}</code>"
      return true
    end
  end

end


end
