require 'yaml'

module CKWiki

  class Index
    # for repetition
    include Enumerable

    attr_accessor :items, :limit, :application

    def self.new_with_application( app )
      new(app.resource_manager.string(resource_name), app)
    end

    def self.resource_name; end

    def initialize( string, app = nil )
      @application = app
      @limit = 0
      @items = parse(string) || []
      if @items.empty? then
        empty_index
      end
    end

    def each
      @items.each do |item|
        yield item
      end
    end

    def empty_index; end

    def parse( string )
      YAML::load(string)
    end

    def add( item )
      values = values(item)
      unless @items.include?(values) then
        add_values(values)
      end
      sort
    end

    def add_values( values )
      @items << values
    end

    def sort; end

    def remove( item )
      @items.delete(values(item))
    end

    def values( item ); end

    def write
      @application.resource_manager.
        write_resource(self.class.resource_name, @items.to_yaml)
    end

  end


  class PageIndex < Index

    Struct.new("PageIndex", :name)

    def self.resource_name
      PAGE_INDEX
    end

    def empty_index
      Dir.foreach(@application.resources) do |file|
        path = File.join(@application.resources, file)
        if File.directory?(path) or (/\A\./ === file) or (/~\Z/ === file) \
          or Application.index_names.include?(file) then
          next
        end
        @items << Struct::PageIndex.new(file)
      end
    end

    def values( page )
      Struct::PageIndex.new(page.name)
    end

    def sort
      @items.sort! do |a, b|
        a.name <=> b.name
      end
    end

    def has_name?( name )
      @items.each do |item|
        if item.name == name then
          return true
        end
      end
      false
    end
  end


  class UploadIndex < Index

    Struct.new("UploadIndex", :name, :length)

    def self.resource_name
      UPLOAD_INDEX
    end

    def values( bytedata )
      Struct::UploadIndex.new(bytedata.path, bytedata.length)
    end

    def add_values( values )
      @items.unshift(values)
    end
  end


  class RecentIndex < Index

    Struct.new("RecentIndex", :name, :date)

    def self.resource_name
      RECENT_INDEX
    end

    def values( page )
      Struct::RecentIndex.new(page.name, Time.new)
    end

    def add_values( values )
      @items.delete_if do |item|
        item.name == values.name
      end
      @items.unshift(values)
    end

  end


  class InversePageIndex < Index

    Struct.new("InversePageIndex", :name, :links)

    def self.resource_name
      INVERSE_PAGE_INDEX
    end

    def initialize( string, app = nil )
      @limit = 0
      @items = parse(string) || {}
      @application = app
    end

    def each
      @items.keys.sort.each do |key|
        yield @items[key]
      end
    end

    def inverse_page_names( name )
      @items[name].links if @items[name]
    end

    def update
      @items.clear
      @application.page_index.each do |index|
        name = index.name
        @application.page_index.each do |inverse|
          next if name == inverse.name
          page = WikiPage.new_with_resource_manager(inverse.name,
                                                    @application.resource_manager)
          if page.content.include?(index.name) then
            @items[name] ||= Struct::InversePageIndex.new(name, [])
            @items[name].links << inverse.name
          end
        end
      end
      write()
    end

  end

end
