/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006, 2008 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "editorwin.h"
#include "user_commands.h"

static void editor_win_class_init(EditorWinClass *klass);
static void editor_win_init(EditorWin *self);
static void add_button_clicked_callback(GtkWidget *w, gpointer user_data);
static void remove_button_clicked_callback(GtkWidget *w, gpointer user_data);
static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data);
static GtkTreeModel *create_model(void);
static void keyword_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data);
static void command_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data);
static void add_columns(EditorWin *self);

GType editor_win_get_type()
{
  static GType type = G_TYPE_INVALID;

  if (G_UNLIKELY(type == G_TYPE_INVALID)) {
    static const GTypeInfo info = {
      sizeof(EditorWinClass),
      NULL,
      NULL,
      (GClassInitFunc) editor_win_class_init,
      NULL,
      NULL,
      sizeof(EditorWin),
      0,
      (GInstanceInitFunc) editor_win_init,
      NULL,
    };
    type = g_type_register_static(GTK_TYPE_DIALOG, "EditorWin", &info, 0);
  }
  return type;
}

static GtkDialogClass *parent_class = NULL;

static void editor_win_class_init(EditorWinClass *klass)
{
  parent_class = g_type_class_peek_parent(klass);
}

#define DEFAULT_WIDTH 400
#define DEFAULT_HEIGHT 360

enum {
  COLUMN_KEYWORD, COLUMN_COMMAND, COLUMN_EDITABLE, NUM_COLUMNS
};

static void editor_win_init(EditorWin *self)
{
  GtkWidget *hbox;
  GtkWidget *scrolled;
  GtkWidget *add_button;
  GtkWidget *remove_button;
  GtkTreeModel *model;
  GtkAccelGroup *accel;

  gtk_window_set_title(GTK_WINDOW(self), _("Edit commands"));
  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			 NULL);
  gtk_container_set_border_width(GTK_CONTAINER(self), 4);
  gtk_window_set_default_size(GTK_WINDOW(self),
			      DEFAULT_WIDTH, DEFAULT_HEIGHT);

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  model = create_model();
  self->treeview = gtk_tree_view_new_with_model(model);
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(self->treeview), TRUE);
  gtk_tree_selection_set_mode
    (gtk_tree_view_get_selection(GTK_TREE_VIEW(self->treeview)),
     GTK_SELECTION_SINGLE);
  g_object_unref(model);
  add_columns(self);
  gtk_container_add(GTK_CONTAINER(scrolled), self->treeview);

  hbox = gtk_hbox_new(FALSE, 0);
  add_button = gtk_button_new_with_mnemonic(_("_Add new"));
  remove_button = gtk_button_new_with_mnemonic(_("_Remove"));
  gtk_box_pack_end(GTK_BOX(hbox), remove_button, FALSE, FALSE, 4);
  gtk_box_pack_end(GTK_BOX(hbox), add_button, FALSE, FALSE, 4);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(self)->vbox),
		     scrolled, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(self)->vbox),
		     hbox, FALSE, FALSE, 4);

  g_signal_connect(add_button, "clicked",
		   G_CALLBACK(add_button_clicked_callback), self);
  g_signal_connect(remove_button, "clicked",
		   G_CALLBACK(remove_button_clicked_callback), self);

  accel = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(self), accel);
  gtk_widget_add_accelerator(GTK_WIDGET(self), "close", accel,
			     GDK_Escape, 0, 0);

  gtk_widget_show_all(GTK_WIDGET(self));

  self->is_edited = FALSE;
}

GtkWidget *editor_win_new(void)
{
  EditorWin *self = g_object_new(TYPE_EDITOR_WIN, NULL);

  return GTK_WIDGET(self);
}

static void add_button_clicked_callback(GtkWidget *w, gpointer user_data)
{
  EditorWin *self = (EditorWin*)user_data;

  GtkTreeIter iter;
  GtkTreeModel *model =
    gtk_tree_view_get_model(GTK_TREE_VIEW(self->treeview));
  GtkTreePath *path;

  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     COLUMN_KEYWORD, _("New item"),
		     COLUMN_COMMAND, _("Edit this line before save!"),
		     COLUMN_EDITABLE, TRUE, -1);

  path = gtk_tree_model_get_path(model, &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(self->treeview), path, NULL, FALSE);

  gtk_tree_path_free(path);
}

static void remove_button_clicked_callback(GtkWidget *w, gpointer user_data)
{
  EditorWin *self = (EditorWin*)user_data;

  GtkTreeIter iter;
  GtkTreeModel *model =
    gtk_tree_view_get_model(GTK_TREE_VIEW(self->treeview));
  GtkTreeSelection *selection =
    gtk_tree_view_get_selection(GTK_TREE_VIEW(self->treeview));

  if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
    gint i;
    GtkTreePath *path;

    path = gtk_tree_model_get_path(model, &iter);
    i = gtk_tree_path_get_indices(path)[0];
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_tree_path_free(path);
    self->is_edited = TRUE;
  }
}

void editor_win_save_changes(EditorWin *self)
{
  if (self->is_edited) {
    GHashTable *new_user_commands =
      g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    GtkTreeIter iter;
    GtkTreeModel *model =
      gtk_tree_view_get_model(GTK_TREE_VIEW(self->treeview));
    gboolean has_element =
      gtk_tree_model_get_iter_first(model, &iter);

    while (has_element) {
      gchar *keyword, *command;

      gtk_tree_model_get(model, &iter, COLUMN_KEYWORD, &keyword,
			 COLUMN_COMMAND, &command, -1);

      g_hash_table_insert(new_user_commands, keyword, command);

      has_element = gtk_tree_model_iter_next(model, &iter);
    }

    free_user_commands();

    user_commands = new_user_commands;
    user_commands_changed = TRUE;
  }
}

static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data)
{
  GtkListStore *store = (GtkListStore*)user_data;
  GtkTreeIter iter;

  gtk_list_store_append(store, &iter);
  gtk_list_store_set(store, &iter, COLUMN_KEYWORD, keyword,
		     COLUMN_COMMAND, command, COLUMN_EDITABLE, TRUE, -1);
}

static GtkTreeModel *create_model(void)
{
  GtkListStore *store = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING,
					   G_TYPE_BOOLEAN);

  user_commands_table_foreach(list_store_append_func, store);
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store),
				       COLUMN_KEYWORD, GTK_SORT_ASCENDING);

  return GTK_TREE_MODEL(store);
}

static void keyword_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data)
{
  EditorWin *self = (EditorWin*)user_data;
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(self->treeview));
  GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
  GtkTreeIter iter;

  gtk_tree_model_get_iter(model, &iter, path);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_KEYWORD,
		     new_text, -1);
  self->is_edited = TRUE;
}

static void command_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data)
{
  EditorWin *self = (EditorWin*)user_data;
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(self->treeview));
  GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
  GtkTreeIter iter;

  gtk_tree_model_get_iter(model, &iter, path);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_COMMAND,
		     new_text, -1);
  self->is_edited = TRUE;
}

static void add_columns(EditorWin *self)
{
  GtkTreeView *treeview = GTK_TREE_VIEW(self->treeview);
  GtkCellRenderer *keyword_renderer = gtk_cell_renderer_text_new();
  GtkCellRenderer *command_renderer = gtk_cell_renderer_text_new();
  GtkTreeViewColumn *keyword_column;
  GtkTreeViewColumn *command_column;

  keyword_column = gtk_tree_view_column_new_with_attributes
    (_("Keyword"), keyword_renderer, "text", COLUMN_KEYWORD,
     "editable", COLUMN_EDITABLE, NULL);
  command_column = gtk_tree_view_column_new_with_attributes
    (_("Command"), command_renderer, "text", COLUMN_COMMAND,
     "editable", COLUMN_EDITABLE, NULL);

  gtk_tree_view_column_set_sizing(keyword_column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing(command_column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);

  gtk_tree_view_append_column(treeview, keyword_column);
  gtk_tree_view_append_column(treeview, command_column);

  g_signal_connect(keyword_renderer, "edited",
		   G_CALLBACK(keyword_renderer_edited_callback), self);
  g_signal_connect(command_renderer, "edited",
		   G_CALLBACK(command_renderer_edited_callback), self);
}

