package bodybuilder.test.dicon;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import bodybuilder.builder.Builder;
import bodybuilder.test.common.Execute;
import bodybuilder.util.ObjectUtils;
import bodybuilder.util.jdom.JDOMUtils;

/**
 * インジェクションオブジェクト
 */
public class InjectedObjects {

    /**
     * コンポーネント名のリスト
     */
    private List names = new ArrayList();

    /**
     * コンポーネントのマップ
     */
    private Map objects = new LinkedHashMap();

    /**
     * コンポーネントの型のリスト
     */
    private List classes = new ArrayList();

    /**
     * 成功時の処理を行うメソッドのマップ
     */
    private Map successes = new LinkedHashMap();

    /**
     * 失敗時の処理を行うメソッドのマップ
     */
    private Map failures = new LinkedHashMap();

    /**
     * 後始末を行うメソッドのマップ
     */
    private Map windups = new LinkedHashMap();

    /**
     * コンストラクタ。
     * 
     * @param component コンポーネント要素
     * @param type インジェクションタイプ
     */
    public InjectedObjects(Element component, int type) {
        // <inject>を取得。
        List children = component.getChildren("inject");

        // <inject>のコンポーネントをフィールドに格納。
        for (int i = 0; i < children.size(); i++) {
            Element child = (Element) children.get(i);
            String name = JDOMUtils.getRequiredAttrValue(child, "name");
            String argType = JDOMUtils.getRequiredAttrValue(child, "type");
            Object object = Builder.getValue(child);

            // 成功処理メソッドを取得。
            Element successElement = child.getChild("success");
            Execute[] success = null;

            if (successElement != null) {
                success = getExecutes(successElement);
            } else {
                success = new Execute[0];
            }

            // 失敗処理メソッドを取得。
            Element failureElement = child.getChild("failure");
            Execute[] failure = null;

            if (failureElement != null) {
                failure = getExecutes(failureElement);
            } else {
                failure = new Execute[0];
            }

            // 後始末メソッドを取得。
            Element windupElement = child.getChild("windup");
            Execute[] windup = null;

            if (windupElement != null) {
                windup = getExecutes(windupElement);
            } else {
                windup = new Execute[0];
            }

            add(name, object, argType, success, failure, windup);
        }
    }

    /**
     * メソッドを取得する。
     * 
     * @param elem 要素
     * @return 後始末を行うメソッド
     */
    private Execute[] getExecutes(Element elem) {
        List executes = new ArrayList();

        // <exec>を取得。
        List children = elem.getChildren("exec");

        // 実行するメソッドを生成。
        for (int i = 0; i < children.size(); i++) {
            Element child = (Element) children.get(i);
            executes.add(new Execute(child));
        }

        return (Execute[]) executes.toArray(new Execute[executes.size()]);
    }

    /**
     * フィールドにコンポーネントを追加する。
     * 
     * @param name コンポーネント名
     * @param object コンポーネント
     * @param type コンポーネントの型
     * @param success 成功時の処理を行うメソッド
     * @param failure 成功時の処理を行うメソッド
     * @param windup 後始末を行うメソッド
     */
    private void add(String name, Object object, String type,
            Execute[] success, Execute[] failure, Execute[] windup) {
        Class clazz = ObjectUtils.getClass(type);
        names.add(name);
        objects.put(name, object);
        classes.add(clazz);
        successes.put(name, success);
        failures.put(name, failure);
        windups.put(name, windup);
    }

    /**
     * コンポーネント名のリストを取得する。
     * 
     * @return コンポーネント名のリスト
     */
    public String[] getNames() {
        return (String[]) names.toArray(new String[names.size()]);
    }

    /**
     * コンポーネントを取得する。
     * 
     * @param name コンポーネント名
     * @return コンポーネント
     */
    public Object getObject(String name) {
        return objects.get(name);
    }

    /**
     * コンポーネントのマップを取得する。
     * 
     * @return コンポーネントのマップ
     */
    public Map getObjectMap() {
        return objects;
    }

    /**
     * コンポーネントのリストを取得する。
     * 
     * @return コンポーネントのリスト
     */
    public Object[] getObjects() {
        return objects.values().toArray();
    }

    /**
     * コンポーネントの型のリストを取得する。
     * 
     * @return コンポーネントの型のリスト
     */
    public Class[] getClasses() {
        return (Class[]) classes.toArray(new Class[classes.size()]);
    }

    /**
     * 成功時の処理を行うメソッドを取得する。
     * 
     * @param name コンポーネント名
     * @return 成功処理メソッド
     */
    public Execute[] getSuccess(String name) {
        return (Execute[]) successes.get(name);
    }

    /**
     * 成功時の処理を行うメソッドのマップを取得する。
     * 
     * @return 成功処理メソッドのマップ
     */
    public Map getSuccesses() {
        return successes;
    }

    /**
     * 失敗時の処理を行うメソッドを取得する。
     * 
     * @param name コンポーネント名
     * @return 失敗処理メソッド
     */
    public Execute[] getFailure(String name) {
        return (Execute[]) failures.get(name);
    }

    /**
     * 失敗時の処理を行うメソッドのマップを取得する。
     * 
     * @return 失敗処理メソッドのマップ
     */
    public Map getFailures() {
        return failures;
    }

    /**
     * 後始末をするメソッドを取得する。
     * 
     * @param name コンポーネント名
     * @return 後始末をするメソッド
     */
    public Execute[] getWindUp(String name) {
        return (Execute[]) windups.get(name);
    }

    /**
     * 後始末をするメソッドのマップを取得する。
     * 
     * @return 後始末をするメソッドのマップ
     */
    public Map getWindUps() {
        return windups;
    }

}