package bodybuilder.inspector;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import bodybuilder.util.Config;
import bodybuilder.util.ObjectUtils;

import junit.framework.AssertionFailedError;

/**
 * インスペクター
 */
public abstract class Inspector {

    /////////////////////////////////////////////////////////////////
    // target property

    /**
     * ターゲットクラスのリスト
     */
    private List targets = new ArrayList();

    /**
     * ターゲットクラスのリストを取得する。
     * 
     * @return ターゲットクラスのリスト
     */
    public List getTargets() {
        return targets;
    }

    /**
     * ターゲットクラスを追加する。
     * 
     * @param target ターゲットクラス
     */
    public void addTarget(String target) {
        targets.add(target);
    }

    /////////////////////////////////////////////////////////////////
    // abstract method

    /**
     * 二つのオブジェクトが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 現実の値
     * @param trace バックトレース
     */
    public abstract void assertEquals(Object expected, Object actual,
            ObjectBackTrace trace);

    /////////////////////////////////////////////////////////////////
    // public method

    /**
     * 二つのオブジェクトが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 現実の値
     */
    public static void assertObjectEquals(Object expected, Object actual) {
        // バックトレースを生成してオブジェクトを検査。
        ObjectBackTrace trace = new ObjectBackTrace();
        assertObjectEquals(expected, actual, trace);
        trace.clear();
    }

    /**
     * 二つのオブジェクトが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 現実の値
     * @param trace バックトレース
     */
    public static void assertObjectEquals(Object expected, Object actual,
            ObjectBackTrace trace) {
        trace.append(actual);
        // いづれかがnullではないか検査。
        assertNullEquals(expected, actual, trace);

        // ともにnullの場合は処理を抜ける。
        if (expected == null && actual == null) {
            return;
        }

        // 期待するオブジェクトにマップされたインスペクターを取得。
        Inspector inspector = InspectorMapping.getInspector(expected);

        if (inspector != null) {
            // インスペクターがある場合
            // 実際のオブジェクトの実装クラスリストを取得。
            List actualClassNames = ObjectUtils.getClassNames(actual);
            // インスペクターのターゲットクラスリストを取得。
            List targets = inspector.getTargets();
            boolean isImplement = false;

            // インスペクターのターゲットクラスが実際のオブジェクトに実装されているかチェック。
            for (int i = 0; i < actualClassNames.size(); i++) {
                String type = (String) actualClassNames.get(i);
                String pkg = ObjectUtils.getPackage(type, true);

                if (targets.contains(type) || targets.contains(pkg)) {
                    isImplement = true;
                    break;
                }
            }

            // インスペクターのターゲットクラスが実装されていない場合はエラー。
            if (!isImplement) {
                rethrow("unimplement class", inspector.getTargets(), actual
                        .getClass().getName(), trace);
            }

            trace.indent();
            // オブジェクトを検査。
            inspector.assertEquals(expected, actual, trace);
            trace.unindent();
        } else {
            // インスペクターがない場合
            // オブジェクトのクラスを検査。
            assertClassEquals(expected, actual, trace);

            if (isPossibleRegexComparing(expected, actual)) {
                // 可能な場合は正規表現で検査。
                String regex = ((String) expected).trim();
                regex = regex.substring(1, regex.length() - 1);
                assertRegexEquals(regex, (String) actual, trace);
            } else {
                // 正規表現書式のエスケープを解除。
                if (expected instanceof String) {
                    expected = unescapeRegex((String) expected);
                }

                // オブジェクトが異なる場合は例外を投げる。
                if (!expected.equals(actual)) {
                    rethrow(expected, actual, trace);
                }
            }
        }
    }

    /////////////////////////////////////////////////////////////////
    // utility method

    // null differ //////////////////////////////////////////////////

    /**
     * オブジェクトが等しい(いづれかがnullではない)ことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     */
    public static void assertNullEquals(Object expected, Object actual) {
        ObjectBackTrace trace = new ObjectBackTrace();
        assertNullEquals(expected, actual, trace);
        trace.clear();
    }

    /**
     * オブジェクトが等しい(いづれかがnullではない)ことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertNullEquals(Object expected, Object actual,
            ObjectBackTrace trace) {
        // いずれかがnullの場合は例外を投げる。
        if ((expected != null && actual == null)
                || (expected == null && actual != null)) {
            rethrow(expected, actual, trace);
        }
    }

    // class differ /////////////////////////////////////////////////

    /**
     * クラスが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     */
    public static void assertClassEquals(Object expected, Object actual) {
        ObjectBackTrace trace = new ObjectBackTrace();
        assertClassEquals(expected, actual, trace, true);
        trace.clear();
    }

    /**
     * クラスが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertClassEquals(Object expected, Object actual,
            ObjectBackTrace trace) {
        assertClassEquals(expected, actual, trace, Config.isInspectionClass());
    }

    /**
     * クラスが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     * @param exec 検査する場合はtrue
     */
    protected static void assertClassEquals(Object expected, Object actual,
            ObjectBackTrace trace, boolean exec) {
        // クラスを検査しない場合は処理を抜ける。
        if (!exec) {
            return;
        }

        // クラスを取得。
        Class expectedClass = expected.getClass();
        Class actualClass = actual.getClass();

        // クラスが異なる場合は例外を投げる。
        if (!expectedClass.getName().equals(actualClass.getName())) {
            rethrow("classes differ", expectedClass, actualClass, trace);
        }
    }

    // size differ //////////////////////////////////////////////////

    /**
     * サイズが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     */
    public static void assertSizeEquals(int expected, int actual) {
        ObjectBackTrace trace = new ObjectBackTrace();
        assertSizeEquals("size differ", expected, actual, trace, true);
        trace.clear();
    }

    /**
     * サイズが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertSizeEquals(int expected, int actual,
            ObjectBackTrace trace) {
        assertSizeEquals("size differ", expected, actual, trace);
    }

    /**
     * サイズが等しいことを表明する。
     * 
     * @param message メッセージ
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertSizeEquals(String message, int expected,
            int actual, ObjectBackTrace trace) {
        assertSizeEquals(message, expected, actual, trace, Config
                .isInspectionSize());
    }

    /**
     * サイズが等しいことを表明する。
     * 
     * @param message メッセージ
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     * @param exec 検査する場合はtrue
     */
    protected static void assertSizeEquals(String message, int expected,
            int actual, ObjectBackTrace trace, boolean exec) {
        // サイズを検査しない場合は処理を抜ける。
        if (!exec) {
            return;
        }

        // サイズが異なる場合は例外を投げる。
        if (expected != actual) {
            rethrow(message, String.valueOf(expected), String.valueOf(actual),
                    trace);
        }
    }

    // keyset differ ////////////////////////////////////////////////

    /**
     * キーセットが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     */
    public static void assertKeySetEquals(Set expected, Set actual) {
        assertKeySetEquals(expected, actual, true);
    }

    /**
     * キーセットが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param strictly 厳密に精査する場合はtrue
     */
    public static void assertKeySetEquals(Set expected, Set actual,
            boolean strictly) {
        ObjectBackTrace trace = new ObjectBackTrace();
        assertKeySetEquals("key set differ", expected, actual, trace, strictly);
        trace.clear();
    }

    /**
     * キーセットが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertKeySetEquals(Set expected, Set actual,
            ObjectBackTrace trace) {
        assertKeySetEquals("key set differ", expected, actual, trace);
    }

    /**
     * キーセットが等しいことを表明する。
     * 
     * @param message メッセージ
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertKeySetEquals(String message, Set expected,
            Set actual, ObjectBackTrace trace) {
        assertKeySetEquals(message, expected, actual, trace, Config
                .isInspectionKeySetStrictly());
    }

    /**
     * キーセットが等しいことを表明する。
     * 
     * @param message メッセージ
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     * @param strictly 厳密に精査する場合はtrue
     */
    protected static void assertKeySetEquals(String message, Set expected,
            Set actual, ObjectBackTrace trace, boolean strictly) {
        // イテレータを取得。
        Iterator expectedIte = expected.iterator();
        Iterator actualIte = actual.iterator();

        // 期待するキーセットが実際のキーセットに含まれているか検査。
        while (expectedIte.hasNext()) {
            Object expectedElement = expectedIte.next();

            if (!actual.contains(expectedElement)) {
                rethrow(message, expectedElement, null, trace);
            }
        }

        // 厳密に検査する場合、キーセットが完全に一致するか検査。
        if (strictly) {
            while (actualIte.hasNext()) {
                Object actualElement = actualIte.next();

                if (!expected.contains(actualElement)) {
                    rethrow(message, null, actualElement, trace);
                }
            }
        }
    }

    // regex differ /////////////////////////////////////////////////

    /**
     * 正規表現書式のエスケープを解除する。
     * 
     * @param str 文字列
     * @return アンエスケープした文字列
     */
    protected static String unescapeRegex(String str) {
        // エスケープした正規表現書式ではない場合はそのまま返す。
        if (!str.matches("^ *\\\\+/.+/ *$")) {
            return str;
        }

        // 文字列をバッファに移す。
        StringBuffer buf = new StringBuffer(str);

        // 「\」を削除。
        int idx = buf.indexOf("\\");
        buf.deleteCharAt(idx);

        // アンエスケープした文字列を返す。
        return buf.toString();
    }

    /**
     * オブジェクトが正規表現で比較可能かどうかを返す。
     * 
     * @param regex 期待する正規表現
     * @param actual 実際の値
     * @return 比較可能な場合はtrue
     */
    protected static boolean isPossibleRegexComparing(Object regex,
            Object actual) {
        // 正規表現の比較を行わない場合はfalseを返す。
        if (!Config.isInspectionRegex()) {
            return false;
        }

        // 文字列ではない場合はfalseを返す。
        if (!(regex instanceof String && actual instanceof String)) {
            return false;
        }

        String regexStr = ((String) regex).trim();

        if (regexStr == null || regexStr.length() < 3) {
            // 文字列がnull、または2文字以下の場合はfalse。
            return false;
        } else if (!regexStr.startsWith("/") || !regexStr.endsWith("/")) {
            // 正規表現書式ではない場合はfalseを返す。
            return false;
        }

        // trueを返す。
        return true;
    }

    /**
     * 文字列が正規表現にマッチすることを表明する。
     * 
     * @param regex 期待する正規表現
     * @param actual 実際の値
     */
    public static void assertRegexEquals(String regex, String actual) {
        ObjectBackTrace trace = new ObjectBackTrace();
        assertRegexEquals(regex, actual, trace);
        trace.clear();
    }

    /**
     * 文字列が正規表現にマッチすることを表明する。
     * 
     * @param regex 期待する正規表現
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertRegexEquals(String regex, String actual,
            ObjectBackTrace trace) {
        assertRegexEquals("not macth regular expression", regex, actual, trace);
    }

    /**
     * 文字列が正規表現にマッチすることを表明する。
     * 
     * @param message メッセージ
     * @param regex 期待する正規表現
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void assertRegexEquals(String message, String regex,
            String actual, ObjectBackTrace trace) {
        // いずれもnullの場合は処理を抜ける。
        if (regex == null && actual == null) {
            return;
        }

        // いずれかがnullの場合は例外を投げる。
        if (actual == null || regex == null) {
            rethrow(message, regex, actual, trace);
        }

        // 文字列が正規表現にマッチしない場合は例外を投げる。
        if (!actual.matches(regex)) {
            rethrow(message, regex, actual, trace);
        }
    }

    /////////////////////////////////////////////////////////////////

    /**
     * 例外を投げる。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void rethrow(Object expected, Object actual,
            ObjectBackTrace trace) {
        rethrow(null, expected, actual, trace);
    }

    /**
     * 例外を
     * 
     * @param message メッセージ
     * @param expected 期待する値
     * @param actual 実際の値
     * @param trace バックトレース
     */
    protected static void rethrow(String message, Object expected,
            Object actual, ObjectBackTrace trace) {
        StringBuffer buffer = new StringBuffer();

        // メッセージをバッファに追加。
        if (message == null) {
            buffer.append("objects differ");
        } else {
            buffer.append(message);
        }

        // ダンプメッセージを組み立てる。
        buffer.append(" ");
        buffer.append("expected:<");
        buffer.append(expected);
        buffer.append("> but was:<");
        buffer.append(actual);
        buffer.append(">");
        buffer.append(System.getProperty("line.separator"));
        buffer.append(trace.dump());

        // 例外を投げる。
        throw new AssertionFailedError(buffer.toString());
    }

}