package bodybuilder.util.dbunit;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Map;

import org.dbunit.Assertion;
import org.dbunit.DatabaseUnitException;
import org.dbunit.database.DatabaseConnection;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.SortedDataSet;
import org.dbunit.dataset.SortedTable;
import org.dbunit.dataset.excel.XlsDataSet;
import org.dbunit.dataset.filter.DefaultColumnFilter;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.dbunit.dataset.xml.XmlDataSet;
import org.dbunit.operation.DatabaseOperation;

import bodybuilder.exception.BodyBuilderException;

/**
 * DbUnitユーティリティ
 */
public class DbUnitUtils {

    /**
     * ドライバ
     */
    private static Class driver = null;

    /**
     * URL
     */
    private String url = null;

    /**
     * ユーザ
     */
    private String user = null;

    /**
     * パスワード
     */
    private String password = null;

    /**
     * ドライバを登録する。
     * 
     * @param driverName ドライバ名
     */
    public static void registerDriver(String driverName) {
        try {
            if (driver == null) {
                driver = Class.forName(driverName);
            }
        } catch (ClassNotFoundException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタ。
     * 
     * @param url URL
     * @param user ユーザ
     * @param password パスワード
     */
    public DbUnitUtils(String url, String user, String password) {
        this.url = url;
        this.user = user;
        this.password = password;
    }

    /////////////////////////////////////////////////////////////////
    // connection method

    /**
     * コネクションを取得する。
     * 
     * @return コネクション
     */
    private IDatabaseConnection getConnection() {
        try {
            // コネクションを取得して、DbUnitのコネクションに変換。
            Connection conn = DriverManager.getConnection(url, user, password);
            return new DatabaseConnection(conn);
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コネクションを閉じる。
     * 
     * @param iconn コネクション
     */
    private void close(IDatabaseConnection iconn) {
        // コネクションがnullなら何もしない。
        if (iconn == null) {
            return;
        }

        // コネクションを閉じる。
        try {
            iconn.close();
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // setup method

    /**
     * データベースをセットアップする。
     * 
     * @param filename データセットのファイル名
     * @param type データセットの型
     */
    public void setUpDatabase(String filename, String type) {
        IDatabaseConnection conn = null;

        try {
            // コネクションを取得。
            conn = getConnection();
            // データセットを取得。
            IDataSet data = getDataSet(filename, type);
            // データベースをクリーンアップしてからデータを投入。
            DatabaseOperation.CLEAN_INSERT.execute(conn, data);
        } catch (DatabaseUnitException e) {
            throw new BodyBuilderException(e);
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        } finally {
            close(conn);
        }
    }

    /////////////////////////////////////////////////////////////////
    // assertion method

    /**
     * データベースとデータセットが等しいことを表明する。
     * 
     * @param filename データセットのファイル名
     * @param type データセットの型
     * @param ignore 検査しないカラムのマップ
     */
    public void assertDataSetEquals(String filename, String type, Map ignore) {
        IDatabaseConnection conn = null;

        try {
            // コネクションを取得。
            conn = getConnection();
            // データセットを取得。
            IDataSet expectedDataSet = getDataSet(filename, type);
            IDataSet actualDataSet = conn.createDataSet();
            // 検査するテーブル名を取得。
            String[] names = expectedDataSet.getTableNames();

            // 期待するデータセットにあるテーブルだけ検査。
            for (int i = 0; i < names.length; i++) {
                String name = names[i];
                ITable expectedTable = expectedDataSet.getTable(name);
                ITable actualTable = actualDataSet.getTable(name);
                // TODO テーブル名を常に大文字に変換しても問題ないか？
                String[] columns = (String[]) ((ignore != null) ? ignore
                        .get(name.toUpperCase()) : null);
                assertTableEquals(expectedTable, actualTable, columns);
            }
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } finally {
            close(conn);
        }
    }

    /**
     * 二つのテーブルが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 実際の値
     * @param ignore 検査しないカラム
     */
    public void assertTableEquals(ITable expected, ITable actual,
            String[] ignore) {
        IDatabaseConnection conn = null;

        try {
            // コネクションを取得。
            conn = getConnection();

            // 検査しないカラムがあればフィルタードテーブルを取得。
            if (ignore != null && ignore.length > 0) {
                expected = getFilteredTable(expected, ignore);
                actual = getFilteredTable(actual, ignore);
            }

            // レコード件数がいずれも0件の場合は処理を抜ける。
            if (expected.getRowCount() == 0 && actual.getRowCount() == 0) {
                return;
            }

            // テーブルをソートして検査。
            expected = new SortedTable(expected);
            actual = new SortedTable(actual);
            Assertion.assertEquals(expected, actual);
        } catch (DatabaseUnitException e) {
            throw new BodyBuilderException(e);
        } finally {
            close(conn);
        }
    }

    /////////////////////////////////////////////////////////////////
    // dataset method

    /**
     * フィルタードテーブルを取得する。
     * 
     * @param table テーブル
     * @param ignore 検査しないカラム
     * @return フィルタードテーブル
     */
    private ITable getFilteredTable(ITable table, String[] ignore) {
        try {
            // TODO カラム名の大文字/小文字は気にしなくてよいか？
            return DefaultColumnFilter.excludedColumnsTable(table, ignore);
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * データセットを取得する。
     * 
     * @param filename ファイル名
     * @param type データセットの型
     * @return データセット
     */
    private IDataSet getDataSet(String filename, String type) {
        // 指定された型のデータセットを返す。
        if (XmlDataSet.class.getName().equals(type)) {
            return getXmlDataSet(filename);
        } else if (FlatXmlDataSet.class.getName().equals(type)) {
            return getFlatXmlDataSet(filename);
        } else if (XlsDataSet.class.getName().equals(type)) {
            return getXlsDataSet(filename);
        } else {
            // XML、Excelワークシート以外には対応していない。
            throw new BodyBuilderException("unimplemented dataset '" + type
                    + "'.");
        }
    }

    /**
     * XMLデータセットを取得する。
     * 
     * @param filename ファイル名
     * @return XMLデータセット
     */
    private XmlDataSet getXmlDataSet(String filename) {
        try {
            return new XmlDataSet(new FileInputStream(filename));
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * フラットXMLデータセットを取得する。
     * 
     * @param filename ファイル名
     * @return フラットXMLデータセット
     */
    private FlatXmlDataSet getFlatXmlDataSet(String filename) {
        try {
            return new FlatXmlDataSet(new File(filename));
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * XLSデータセットを取得する。
     * 
     * @param filename ファイル名
     * @return XLSデータセット
     */
    private JxlDataSet getXlsDataSet(String filename) {
        try {
            return new JxlDataSet(new File(filename));
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // viewing method

    /**
     * データベースの内容をXMLテキストとして取得する。
     * 
     * @return データベースの内容
     */
    public String getDatabaseContent() {
        IDatabaseConnection conn = null;

        try {
            conn = getConnection();
            IDataSet dataSet = conn.createDataSet();
            StringWriter writer = new StringWriter();
            // 出力はソートする。
            FlatXmlDataSet.write(new SortedDataSet(dataSet), writer);
            return writer.toString();
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        } finally {
            close(conn);
        }
    }

    /**
     * テーブルの内容をXMLテキストとして取得する。
     * 
     * @param name テーブル名
     * @return テーブルの内容
     */
    public String getTableContent(String name) {
        IDatabaseConnection conn = null;

        try {
            conn = getConnection();
            IDataSet dataSet = conn.createDataSet(new String[] { name });
            StringWriter writer = new StringWriter();
            // 出力はソートする。
            FlatXmlDataSet.write(new SortedDataSet(dataSet), writer);
            return writer.toString();
        } catch (SQLException e) {
            throw new BodyBuilderException(e);
        } catch (DataSetException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        } finally {
            close(conn);
        }
    }

}