package bodybuilder.ant.xmltest;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.optional.junit.BatchTest;
import org.apache.tools.ant.taskdefs.optional.junit.FormatterElement;
import org.apache.tools.ant.taskdefs.optional.junit.JUnitTask;
import org.apache.tools.ant.taskdefs.optional.junit.JUnitTest;
import org.apache.tools.ant.types.CommandlineJava;
import org.apache.tools.ant.types.Environment;

import bodybuilder.test.XMLTestRunner;
import bodybuilder.util.FileUtils;
import bodybuilder.util.Utils;

/**
 * XMLJUnitタスク
 */
public class XMLJUnitTask extends JUnitTask {

    /**
     * テストケースディレクトリのプロパティキー
     */
    static final String TEST_PROP_KEY_ROOT_DIRS = "bodybuilder.test.root.dirs";

    /**
     * 深さのシステムプロパティのキー
     */
    static final String TEST_PROP_KEY_DEPTH = "bodybuilder.xmltest.dir.depth";

    /**
     * コンストラクタ。
     */
    public XMLJUnitTask() throws Exception {
        super();
        // フォークモードのデフォルト値をセット。
        super.setForkMode(new ForkMode(ForkMode.PER_TEST));
    }

    /////////////////////////////////////////////////////////////////
    // fork attribute

    /**
     * フォークフラグ
     */
    private boolean forked = true;

    /**
     * フォークフラグをセットする。
     * 
     * @param value フォークフラグ
     */
    public void setFork(boolean value) {
        forked = value;
    }

    /////////////////////////////////////////////////////////////////
    // fork attribute

    /**
     * フォークモードをセットする。
     * 
     * @param mode フォークモード
     */
    public void setForkMode(ForkMode mode) {
        // フォークモードをセットできないようにする。
        throw new BuildException(
                "The <xmltest> type doesn't support the \"forkmode\" attribute.");
    }

    /////////////////////////////////////////////////////////////////
    // todir attribute

    /**
     * 出力先ディレクトリ
     */
    private File todir = null;

    /**
     * 出力先ディレクトリをセットする。
     * 
     * @param value 出力先ディレクトリ
     */
    public void setTodir(File value) {
        todir = value;
    }

    /////////////////////////////////////////////////////////////////
    // depth attribute

    /**
     * ディレクトリの深さ
     */
    private int depth = 4;

    /**
     * ディレクトリの深さをセットする。
     * 
     * @param value ディレクトリの深さ
     */
    public void setDepth(int value) {
        depth = value;
    }

    /////////////////////////////////////////////////////////////////
    // exformatter element

    /**
     * 拡張フォーマッタ要素を追加する。
     * 
     * @param fe 拡張フォーマッタ要素
     */
    public void addExFormatter(ExtendedFormatterElement fe) {
        super.addFormatter(fe);
    }

    /**
     * フォーマッタ要素を追加する。
     * 
     * @param fe フォーマッタ要素
     */
    public void addFormatter(FormatterElement fe) {
        // フォーマッタ要素を追加できないようにする。
        throw new BuildException(
                "The <xmljunit> type doesn't support the nested \"formatter\" element.");
    }

    /////////////////////////////////////////////////////////////////
    // xmltest element

    /**
     * テストリスト
     */
    private List tests = new ArrayList();

    /**
     * XMLテスト要素を追加する。
     * 
     * @param test XMLテスト要素
     */
    public void addXmlTest(XMLJUnitTest test) {
        tests.add(test);
    }

    /**
     * テスト要素を追加する。
     * 
     * @param test テスト要素
     */
    public void addTest(JUnitTest test) {
        // テスト要素を追加できないようにする。
        throw new BuildException(
                "The <xmljunit> type doesn't support the nested \"test\" element.");
    }

    /**
     * バッチテストを生成する。
     * 
     * @return バッチテスト
     */
    public BatchTest createBatchTest() {
        // バッチテストを生成できないようにする。
        throw new BuildException(
                "The <xmljunit> type doesn't support the nested \"batchtest\" element.");
    }

    /////////////////////////////////////////////////////////////////
    // commandline

    /**
     * コマンドライン
     */
    private ExtendedCommandlineJava commandline = null;

    /**
     * コマンドラインを取得する。
     * 
     * @return コマンドライン
     */
    protected CommandlineJava getCommandline() {
        if (commandline == null) {
            commandline = new ExtendedCommandlineJava();
        }

        return commandline;
    }

    /////////////////////////////////////////////////////////////////
    // execute task

    /**
     * タスクを実行する。
     */
    public void execute() throws BuildException {
        // パスリストを生成。
        List pathList = new ArrayList();

        for (int i = 0; i < tests.size(); i++) {
            // XMLテスト要素を取得。
            XMLJUnitTest test = (XMLJUnitTest) tests.get(i);

            // テスト名がセットされていた場合はエラー。
            if (test.isNamed()) {
                throw new BuildException(
                        "The <test> type doesn't support the \"name\" attribute.");
            }

            // パスリストにパスを追加。
            String path = test.getPath();
            pathList.add(path);
            // フォーク属性をセット。
            test.setFork2(forked);

            // 出力先ディレクトリをセット。
            if (test.getTodir() == null) {
                test.setTodir(todir);
            }

            if (!forked) {
                if (i > 0) {
                    // フォークしない場合は２つ以上のテストを追加しない。
                    continue;
                }

                // フォークしない場合はXMLテストランナーのクラス名を出力ファイル名にセット。
                test.setOutfile("TEST-" + XMLTestRunner.class.getName());
            } else {
                // フォークする出力ファイル名をセット。
                String outfile = "TEST-"
                        + FileUtils.toDotSeparatedValue(FileUtils.trim(path,
                                depth));
                test.setOutfile(outfile);
            }

            // テストを追加。
            super.addTest(test);
        }

        if (forked) {
            // フォークする場合はコマンドラインに属性をセット。
            ExtendedCommandlineJava excl = (ExtendedCommandlineJava) getCommandline();
            excl.setPathList(pathList);
            excl.setDepth(depth);
        } else {
            // フォークしない場合はシステムプロパティをセット。
            addConfiguredSysproperty(TEST_PROP_KEY_ROOT_DIRS, Utils.join(
                    pathList, ","));
            // 念のため、深さもプロパティにセット。
            addConfiguredSysproperty(TEST_PROP_KEY_DEPTH, String.valueOf(depth));
        }

        // タスクを実行。
        super.execute();
    }

    /**
     * システムプロパティを追加する。
     * 
     * @param key キー
     * @param value 値
     */
    public void addConfiguredSysproperty(String key, String value) {
        Environment.Variable var = new Environment.Variable();
        var.setKey(key);
        var.setValue(value);
        addConfiguredSysproperty(var);
    }

}