from inform_or_utility.create_packing_instructions import create_packing_instructions
from inform_or_utility.draw_packing_instructions import draw_packing_instructions
from inform_or_utility.draw_3D_packing_instructions import plot_solution
from inform_or_experiments.split_model_pfsp import SplitModelPFSP
from multiprocessing import Pool, cpu_count
import argparse
import pprint
import time
import os


def run_experiment(input_file: str, boxes_file: str, output: str, first: int or None, last: int or None, vis_2d: bool,
                   vis_3d: bool):
    """ Executes the experiment and writes the solution to output_file.

    Parameters
    ----------
    input_file: str
        The file defining the orders of the experiment.
        Has to be a CSV file generated by the INFORM Generator.py.

    boxes_file: str
        A csv file specifying the dimensions for all boxes.

    output: str
        This is the file used to store the optimal solution.

    first: int or None
        The first order of the given file to be packed.
        If None the packing will start with the first order of the file.

    last: int or None
        The last order of the given file to be packed.
        If None the model will pack up to the last order of the file.

    vis_2d: bool
        Specifies if the packing should be saved as a 2 dimensional top down view.

    vis_3d: bool
        Specifies if the packing should be animated in a 3 dimensional view after solving the problem.

    Returns
    -------
    """
    experiment = SplitModelPFSP(input_file, boxes_file)

    order_from = first
    order_to = last
    if isinstance(order_to, int):
        order_to += 1
    order_names = list(experiment.orders.keys())[order_from:order_to]

    start = time.perf_counter()

    with Pool(cpu_count() - 1 or 1) as pool:
        optimal_solution = dict(zip(order_names, pool.map(experiment.process, order_names)))

    end = time.perf_counter()
    with open(os.path.join(output, 'complete_split_pfsp_dict.txt'), 'w') as solution_file:
        pprint.pprint(optimal_solution, solution_file)

    if vis_2d:
        for order, packing in optimal_solution.items():
            picture_folder = os.path.join(output, order)
            os.makedirs(picture_folder, exist_ok=True)
            draw_packing_instructions({order: packing}, experiment.length_boxes, experiment.width_boxes, picture_folder)

    if vis_3d:
        for order, packing in optimal_solution.items():
            print(f'Animating 3D packing of order {order}')
            plot_solution({order: packing}, experiment.length_boxes, experiment.width_boxes, experiment.height_boxes)
    create_packing_instructions(optimal_solution, output, 'split_pfsp', False)

    num_orders = len(optimal_solution)
    num_boxes = 0
    num_items = 0

    total_ratio = 0
    for order, boxes in optimal_solution.items():
        num_boxes += len(boxes)
        for box, items in boxes.items():
            box_volume = experiment.length_boxes[box[0]] * experiment.height_boxes[box[0]] * experiment.width_boxes[box[0]]
            vol_items = 0
            num_items += len(items)
            for item in items:
                vol_items += experiment.length_items[item[0]] * experiment.height_items[item[0]] * experiment.width_items[item[0]]
            total_ratio += vol_items / box_volume

    print(f'Average volume ratio for the orders is {total_ratio/num_boxes}.')

    print(f'\nProcessed {num_orders} orders.\n')
    print(f'\nTime elapsed:\t{round(end - start)} seconds\n')
    print(f'Packed {num_items} items into {num_boxes} boxes.')


def __parse_arguments() -> argparse.Namespace:
    """ Function to process the commandline arguments.

    Returns
    -------
    argparse.Namespace
        An object containing the values for each commandline argument.
    """

    parser = argparse.ArgumentParser()

    # Mandatory Arguments
    parser.add_argument('input', help='path to the input file to be read', type=str)
    parser.add_argument('boxes', help='path to the csv file containing the dimensions of all boxes', type=str)
    parser.add_argument('output', help="path to the folder to store all output files. Expected to end on '\\' or '/' depending on the operating system", type=str)

    # Optional Arguments
    parser.add_argument('-f', '--first',
                        help='order ID of the first order in the input file to be packed. If none is given the packing will start with the first order in the input file',
                        type=int, default=None)
    parser.add_argument('-l', '--last',
                        help="order ID of the last order in the input file to be packed. If none is given the packing will end with the last order in the input file",
                        type=int, default=None)
    parser.add_argument('-vis3', '--visualize3', help='Display packing as a 3D animation.', action='store_true')
    parser.add_argument('-vis2', '--visualize2', help='Save packing as 2D plots.', action='store_true')

    # Print Version
    parser.add_argument('--version', action='version', version='%(prog)s - 1.0.0')

    arguments = parser.parse_args()

    return arguments


if __name__ == '__main__':
    args = __parse_arguments()
    run_experiment(args.input, args.boxes, args.output, args.first, args.last, args.visualize2, args.visualize3)
