from inform_or_parser.parse_split_solution_to_pfsp import parse_split_solution_to_pfsp
from inform_or_utility.create_packing_instructions import create_packing_instructions
from inform_or_models.peak_filling_slice_push_heuristic import PeakFillingSlicePush
from inform_or_utility.draw_packing_instructions import draw_packing_instructions
from inform_or_parser.parse_ep_solution_to_pfsp import parse_ep_solution_to_pfsp
from inform_or_utility.draw_3D_packing_instructions import plot_solution
from inform_or_parser.parse_boxes import parse_boxes
import argparse
import time
import os


def run_experiment(input_file: str, item_sizes_file: str, output: str, container_length: int, container_width: int,
                   container_height: int, container_name: str, vis_2d: bool, vis_3d: bool, ep: bool):
    """ This function starts the peak filling slice push heuristic with the given arguments.
        Furthermore, computes runtime statistics and writes the solution to files.

    Parameters
    ----------
    input_file: str
        Path to the file containing the solution created by the split model.

    item_sizes_file: str
        Path to a csv File containing the dimensions of all items to be placed in the specified container.

    output: str
        Folder path to store all created solution outputs.

    container_length: int
        Length of the container to place all items.

    container_width: int
        Width of the container to place all items.

    container_height: int
        Height of the container to place all items.

    container_name: str
        Name of the container. Used for drawing of the packing instructions.

    vis_2d: bool
        Specifies if the packing should be saved as a 2 dimensional top down view.

    vis_3d: bool
        Specifies if the packing should be animated in a 3 dimensional view after solving the problem.

    Returns
    -------
    """
    if ep:
        items_to_pack = parse_ep_solution_to_pfsp(input_file)
    else:
        items_to_pack = parse_split_solution_to_pfsp(input_file)
    num_items = len(items_to_pack)
    boxes = parse_boxes(item_sizes_file)
    heuristic = PeakFillingSlicePush(boxes, container_length, container_width, container_height, container_name)
    start = time.perf_counter()
    packing = heuristic.process(items_to_pack)
    end = time.perf_counter()
    solution = {'P': packing}

    if vis_2d:
        for order, packing in solution.items():
            picture_folder = os.path.join(output, order)
            os.makedirs(picture_folder, exist_ok=True)
            draw_packing_instructions({order: packing}, {container_name: container_length},
                                      {container_name: container_width}, picture_folder)

    if vis_3d:
        for order, packing in solution.items():
            print(f'Animating 3D packing of order {order}')
            plot_solution({order: packing}, {container_name: container_length}, {container_name: container_width},
                          {container_name: container_height})

    create_packing_instructions(solution, output, 'pfsp', True)
    num_containers = len(packing)
    print(f'\nTime elapsed:\t{round(end - start)} seconds\n')
    print(f'Packed {num_items} items into {num_containers} boxes.')


def __parse_arguments():
    """ Function to process the commandline arguments.

    Returns
    -------
    argparse.Namespace:
          An object containing the values for each commandline argument.
    """
    parser = argparse.ArgumentParser()

    # Mandatory Arguments
    parser.add_argument('items_to_pack', help='path to the file defining the items to be packed', type=str)
    parser.add_argument('item_sizes', help='path to the file defining the sizes of all items', type=str)
    parser.add_argument('output', help="path to the folder to store all output files. Expected to end on '\\' or '/' depending on the operating system", type=str)

    # Optional Arguments
    parser.add_argument('-cl', '--container_length', help='The length of the container to place all items', type=int,
                        default=120)
    parser.add_argument('-cw', '--container_width', help='The width of the container to place all items', type=int,
                        default=80)
    parser.add_argument('-ch', '--container_height', help='The height of the container to place all items', type=int,
                        default=80)
    parser.add_argument('-cn', '--container_name', help='The name of the container to place all items', type=str,
                        default='C')
    parser.add_argument('-vis3', '--visualize3', help='Display packing as a 3D animation.', action='store_true')
    parser.add_argument('-vis2', '--visualize2', help='Save packing as 2D plots.', action='store_true')
    parser.add_argument('-ep', '--extreme_point', help="Necessary if 'items_to_pack' has been created by Extreme Points heuristic", action='store_true')

    # Print Version
    parser.add_argument('--version', action='version', version='%(prog)s - 1.0.0')

    arguments = parser.parse_args()

    return arguments


if __name__ == '__main__':
    args = __parse_arguments()
    run_experiment(args.items_to_pack, args.item_sizes, args.output, args.container_length, args.container_width,
                   args.container_height, args.container_name, args.visualize2, args.visualize3, args.extreme_point)
