from inform_or_utility.create_gabm_packing_instructions import create_gabm_packing_instructions
from inform_or_utility.create_packing_instructions import create_packing_instructions
from inform_or_utility.draw_packing_instructions import draw_packing_instructions
from inform_or_utility.draw_3D_packing_instructions import plot_solution
from inform_or_parser.parse_csv_genetic_approach import parse_generated_csv
from inform_or_models.genetic_algorithm import genetic_algorithm
from inform_or_parser.parse_boxes import parse_boxes
import argparse
import pprint
import os


def output_to_dict(item_list, box_list, order):
    """ This function parses the list of items and boxes of a given order to a dictionary.

    Parameters
    ----------
    item_list
        List of all items to be parsed.

    box_list
        List of all boxes to be parsed.

    order
        ID of the order to be parsed.

    Returns
    -------
        Dictionary containing all input information.
    """
    out_dict = dict()
    out_dict[order] = {}

    for box in box_list:
        out_dict[order][box] = {}
        
        for item in item_list:
            if item.box == box[1]:
                out_dict[order][box][item.name] = (item.min_x, item.min_y, item.min_z, item.max_x, item.max_y,
                                                   item.max_z, item.length, item.width, item.height)

    return out_dict


def run_experiment(input_file_path, boxes_files, output_file_path, first, last, generations, chromosomes, probc, probt,
                   E, considered_items, consideredEMS, vis3D, vis2D, text):
    """ This function will start the entire optimisation process with the given arguments and creates all human
        readable outputs.

    Parameters
    ----------
    input_file_path
        Path to the csv file containing all orders. (Created by the INFORM Generator)

    boxes_files
        Path to the csv file specifying all boxes that can be used to pack items.

    output_file_path
        Directory path to store all output files.

    first
        ID of the first order to be considered from the input file.

    last
        ID of the last order to be considered from the input file.

    generations
        Number of generations to be created by the genetic algorithm.

    chromosomes
        Number of chromosomes in each generation.

    probc
        Probability prob_c.

    probt
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    considered_items
        Number of items considered in each step of the best match heuristic.

    consideredEMS
        Number of EMS considered in each iteration.

    vis3D
        Whether a 3D visualisation/animation of the optimal solution plotting should be created.

    vis2D
        Whether a 2D visualisation of the optimal solution should be created.

    text
        Whether a textual packing instruction should be printed to standard output.

    Returns
    -------
    """
    boxes_dict = parse_boxes(boxes_files)

    boxes = []
    length_box = {}
    width_box = {}
    height_box = {}
    
    for box in boxes_dict:
        boxes.append(box)
        length_box[box] = boxes_dict[box][0]
        width_box[box] = boxes_dict[box][1]
        height_box[box] = boxes_dict[box][2]

    orders, length_item, width_item, height_item, weight_item = parse_generated_csv(input_file_path, ";", "utf-16", first, last)

    complete_solution_dictionary = dict()

    for order in orders:
        while (int(order.split('O')[1]) >= first) and (int(order.split('O')[1]) <= last):

            items = list()
            for items_mult in orders[order]:
                for i in range(items_mult[1]):
                    items.append((items_mult[0], i+1))

            if len(items) >= 2:
                best_chromosome, item_list, box_list, elapsedTime = genetic_algorithm(generations=generations,
                                                                                      population_size=chromosomes,
                                                                                      probc=probc,
                                                                                      probt=probt,
                                                                                      E=E,
                                                                                      considered_items=considered_items,
                                                                                      consideredEMS=consideredEMS,
                                                                                      order=order,
                                                                                      items=items,
                                                                                      boxes=boxes,
                                                                                      length_item=length_item,
                                                                                      width_item=width_item,
                                                                                      height_item=height_item,
                                                                                      length_box=length_box,
                                                                                      width_box=width_box,
                                                                                      height_box=height_box,
                                                                                      weight_item=weight_item,
                                                                                      orders=orders)

                item_list.sort(key=lambda x: x.min_z)

                if best_chromosome[0].packing_sequence is None:
                    print("Did not find a feasible packing solution.")

                else:
                    out_dict = output_to_dict(item_list, box_list, order)
                    complete_solution_dictionary.update(out_dict)

                    if text is True:
                        create_gabm_packing_instructions(best_chromosome=best_chromosome[0],
                                                         generations=generations,
                                                         population_size=chromosomes,
                                                         order=order,
                                                         prob_c=probc,
                                                         prob_t=probt,
                                                         E=E,
                                                         considered_items=considered_items,
                                                         considered_EMS=consideredEMS,
                                                         length_box=length_box,
                                                         width_box=width_box,
                                                         height_box=height_box,
                                                         length_item=length_item,
                                                         width_item=width_item,
                                                         height_item=height_item,
                                                         weight_item=weight_item,
                                                         used_time=elapsedTime)

                    if vis3D:
                        plot_solution(packing=out_dict,
                                      length_box=length_box,
                                      width_box=width_box,
                                      height_box=height_box,
                                      color='#ffbf00')
                    if vis2D:
                        picture_folder = os.path.join(output_file_path, order)
                        os.makedirs(picture_folder, exist_ok=True)
                        draw_packing_instructions(packing=out_dict, box_length=length_box, box_width=width_box,
                                                  output=picture_folder)
                        out_dict.clear()


                    instruction_folder = os.path.join(output_file_path, order)
                    create_gabm_packing_instructions(best_chromosome=best_chromosome[0],
                                                     generations=generations,
                                                     population_size=chromosomes,
                                                     order=order,
                                                     prob_c=probc,
                                                     prob_t=probt,
                                                     E=E,
                                                     considered_items=considered_items,
                                                     considered_EMS=consideredEMS,
                                                     length_box=length_box,
                                                     width_box=width_box,
                                                     height_box=height_box,
                                                     length_item=length_item,
                                                     width_item=width_item,
                                                     height_item=height_item,
                                                     weight_item=weight_item,
                                                     used_time=elapsedTime,
                                                     output_file_path=instruction_folder)

            else:
                print(order, 'contains too few items! Skipping this order!')

            del item_list[:], box_list[:]

            break

    create_packing_instructions(complete_solution_dictionary, output_file_path, 'gabm', False)

    with open(os.path.join(output_file_path, 'complete_gabm_dict.txt'), 'w') as solution_file:
        pprint.pprint(complete_solution_dictionary, solution_file)


def __parse_arguments():
    """ Function to process the commandline arguments.

    Returns
    -------
        An object containing the values for each commandline argument.
    """

    parser = argparse.ArgumentParser()

    # Mandatory Arguments
    parser.add_argument('inputFile', help='path to the .csv file containing the orders.', type=str)
    parser.add_argument('boxFile', help='path to the csv file containing the dimensions of all boxes', type=str)
    parser.add_argument('outputDirectory', help='path to the output directory that will hold the packing instructions. For each order a new directory will be automatically created.', type=str)

    # Optional Arguments
    parser.add_argument('-f', '--first', help='ID of the first order to be processed.', type=int, default=None)
    parser.add_argument('-l', '--last', help='ID of the last order to be processed', type=int, default=None)
    parser.add_argument('-g', '--generations', help='number of generations for each order', type=int, default=20)
    parser.add_argument('-c', '--chromosomes', help='number of chromosomes in each generation', type=int, default=50)
    parser.add_argument('-vis3', '--visualize3', help='Display packing as a 3D animation.', action='store_true')
    parser.add_argument('-vis2', '--visualize2', help='Save packing as 2D plots.', action='store_true')
    parser.add_argument('-info', '--textinfo', help='Write textual packing to console.', action='store_true')
    parser.add_argument('-pc', '--probc', help='probability that a pair of parents directly go into the next generation', type=float, default=0.75)
    parser.add_argument('-pt', '--probt', help='probability that the better one (larger fitness) of two chromosomes is added into mating pool', type=float, default=0.85)
    parser.add_argument('-E', help='number of best chromosomes that directly advance to the next generation.', type=int, default=10)
    parser.add_argument('-ci', '--considered_items', help='number of considered items in each iteration of the best match heuristic.', type=int, default=3)
    parser.add_argument('-ce', '--considered_EMS', help='number of EMS considered in each iteration of the best match heuristic.', type=int, default=3)

    # Print Version
    parser.add_argument('-v', '--version', action='version', version='%(prog)s - 1.0.0')

    arguments = parser.parse_args()

    return arguments


if __name__ == '__main__':
    args = __parse_arguments()
    print(args.visualize3)
    print(args.visualize2)
    run_experiment(args.inputFile, args.boxFile, args.outputDirectory, args.first, args.last, args.generations,
                   args.chromosomes, args.probc, args.probt, args.E, args.considered_items, args.considered_EMS,
                   args.visualize3, args.visualize2, args.textinfo)
