# -*- coding: utf-8 -*-
"""Module for running the EP heuristic from the runner directory."""

#  Copyright (c) 2020.  Yordan Manolov <yordan DOT manolov AT rwth DASH aachen DOT de >
#  Copyright (c) 2020.  Moritz Dederichs

import argparse
import logging
import time
from statistics import mean
from typing import List
import pprint
import os

from inform_or_ep_objects.box import Box
from inform_or_models.extreme_points_heuristic import extreme_points, \
    print_instruction_description_to_file
from inform_or_parser import parse_csv_extreme_points

logging.basicConfig(filename="../../Latex/presentation/benchmarks/ep/ep_debug.log",
                    filemode="a",
                    format="%(asctime)s %(message)s",
                    level="DEBUG")


def run_experiment(arguments: argparse.Namespace):
    """ This function starts the current experiment using the extreme points heuristic with the given arguments.

    Parameters
    ----------
    arguments: argparse.Namespace
        The parsed commandline arguments given by script start.

    Returns
    -------
    """
    instruction = os.path.join(arguments.output, 'instructions_ep.txt')
    print_instruction_description_to_file(instruction)
    processed_orders = list()
    start_all_orders = time.perf_counter()

    for order in parse_csv_extreme_points.parse_file_as_objects(arguments):
        start_current_order = time.perf_counter()
        result = extreme_points(order, instruction, arguments)
        processed_orders.append(result)
        end_current_order = time.perf_counter()
        time_current_order = end_current_order - start_current_order
        stat_me("Order no. " + order.identifier,
                1,
                time_current_order,
                result)

    box_list = [box.standard_size for order in processed_orders for box in order]
    with open(os.path.join(arguments.output, 'complete_ep_list.txt'), 'w') as file:
        pprint.pprint(box_list, file)

    end_all_orders = time.perf_counter()
    time_all_orders = end_all_orders - start_all_orders
    stat_me("Category from " + str(args.first) +
            " to " + str(args.last),
            abs(args.first - args.last),
            time_all_orders,
            processed_orders)


def __parse_arguments() -> argparse.Namespace:
    """ Function to process the given commandline arguments

    Returns
    -------
    argparse.Namespace:
        An object containing the values of each commandline argument.
    """
    parser = argparse.ArgumentParser()

    # Mandatory Arguments
    parser.add_argument('input', help='path to the input file to be read',
                        type=str)
    parser.add_argument('output',
                        help="path to the output directory to store all packing instructions.",
                        type=str)

    # Optional Arguments
    parser.add_argument('-m', '--merit_function',
                        help="merit function to use for EP-BFD.",
                        choices=['fv', 'mp', 'lev', 'rs'],
                        default=None)
    parser.add_argument('-s', '--sorting',
                        help="criterion for sorting the items.",
                        choices=["area", 'area_height', "height",
                                 'height_area', 'height_volume', 'volume',
                                 'volume_height'],
                        default='height_area')
    parser.add_argument('-f', '--first',
                        help='order ID of the first order in the input file to be packed. If none is given the packing will start with the first order in the input file',
                        type=int, default=None)
    parser.add_argument('-l', '--last',
                        help="order ID of the last order in the input file to be packed. If none is given the packing will end with the last order in the input file",
                        type=int, default=None)
    return parser.parse_args()


def stat_me(descriptor: str,
            length: int = 1,
            elapsed_time: float = 0,
            packing_result: List[Box] or List[List[Box]] = None):
    """

    Parameters
    ----------
    descriptor What kind of input is benchmarked, e.g. single order, category of orders etc
    length The length of the input for which packing_result was computed
    elapsed_time The time it took to produce packing_result
    packing_result The output of the Extreme Points-based heuristic with input as described in descriptor

    Returns
    -------


    """
    if not isinstance(packing_result[0], Box):
        packing_result_flattened = [x for y in packing_result for x in y]
    else:
        packing_result_flattened = packing_result
    if "Category" in descriptor:
        logging.debug("Details for %s : \n"
                      "Total time (in seconds) : %s \n"
                      "Number of boxes : %s \n"
                      "Avg. number of boxes : %s \n"
                      "Free volume (total) : %s \n"
                      "Avg. free volume: %s percent",
                      descriptor,
                      elapsed_time,
                      len(packing_result_flattened),
                      divmod(len(packing_result_flattened),
                             length)[0],
                      sum([x.get_free_volume() for x in
                           packing_result_flattened]),
                      mean([x.get_free_volume_ratio() for x in
                            packing_result_flattened])
                      )


if __name__ == "__main__":
    args = __parse_arguments()
    run_experiment(args)
