# -*- coding: utf-8 -*-
"""
Provides functions which sort items according to different criteria.

Intended for use as a library for multiple heuristics.
"""

#  Copyright (c) 2020.  Yordan Manolov <yordan DOT manolov AT rwth DASH aachen DOT de >
#

from operator import attrgetter
from typing import List

from inform_or_ep_objects.item import Item


def item_sort_volume(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to volume in descending order.

    Parameters
    ----------
    items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items, key=lambda x: x.volume, reverse=True)


def item_sort_volume_height(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to volume, then height in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("volume", "height"),
                  reverse=True)


def item_sort_height_volume(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to height, then volume in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("height", "volume"),
                  reverse=True)


def item_sort_area(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to area in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("area"),
                  reverse=True)


def item_sort_area_height(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to area, then height in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("area", "height"),
                  reverse=True)


def item_sort_height(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to height in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("height"),
                  reverse=True)


def item_sort_height_area(items: List[Item]) -> List[Item]:
    """
    Sorts a list of items according to area, then height in descending order.

    Parameters
    ----------
        items : List[Item]
        The list of items to sort.

    Returns
    -------
    List[Item]
        The resulting sorted list.
    """
    return sorted(items,
                  key=attrgetter("height", "area"),
                  reverse=True)
