# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Janos Piddubnij

from matplotlib.patches import Rectangle
from matplotlib import pyplot as plt
from collections import defaultdict
from matplotlib.pyplot import Axes
import os


def __plot_packing_layer(items: dict, colours: dict, axes: Axes):
    """ Plot a packing level containing the given items.

    Parameters
    ----------
    items: dict
        A dictionary containing all items and their position inside the box.
        x-,y-coordinates of the lower left corner and x-,y-coordinates of the top right corner.

    colours: dict
        A dictionary assigning each item type to a colour.

    axes: Axes
        The axes object to add the plot.

    Returns
    -------
    """
    for item, placement in items.items():
        item = item[0]
        rectangle_width = placement[3] - placement[0]
        rectangle_height = placement[4] - placement[1]
        obj = Rectangle((placement[0], placement[1]), width=rectangle_width, height=rectangle_height, facecolor=colours[item], edgecolor='k')
        axes.add_patch(obj)
        annotation_x = (placement[0] + placement[3])/2
        annotation_y = (placement[1] + placement[4])/2
        axes.text(annotation_x, annotation_y, item, color='w', weight='bold', ha='center', va='center', fontsize=16)


def __draw_box_packing(order: str, num_box: int, box: str, box_length: float, box_width: float, items: dict, output: str):
    """ Creates a packing plot for each packing level of the given box.

    Parameters
    ----------
    order: str
        The order that is currently packed.

    num_box: int
        The number of the box within the order.

    box: str
        The box type to be packed.

    box_length: float
        The length of the specified box.

    box_width: float
        The width of the specified box.

    items: dict
        A dictionary of all items placed in the box defining their placements inside the box.

    output: str
        The folder to store all packing plots.

    Returns
    -------
    """
    layers = defaultdict(list)
    for key, value in items.items():
        layers[value[2]].append(key)
    colours = {k[0]: 'y' for k, v in items.items()}
    index = 0
    for layer, objects in layers.items():
        __items = {obj: items[obj] for obj in objects}
        fig, ax = plt.subplots(1, figsize=(13, 10))
        ax.set_xlim(0, box_length)
        ax.set_ylim(0, box_width)
        __plot_packing_layer(__items, colours, ax)
        fig.suptitle(f'Order {order}. Box {box}-{num_box+1}. Layer {index+1}')
        plt.savefig(os.path.join(output, f'{order}_{box}_{num_box+1}_{index+1}.png'))
        plt.close(fig)
        index += 1


def draw_packing_instructions(packing: dict, box_length: dict, box_width: dict, output: str):
    """ Creates a visual packing instruction for all passed orders.

    Parameters
    ----------
    packing: dict
        A dictionary specifying the boxes and item placements within each box for all orders.

    box_length: dict
        A dictionary defining the box length of each box type.

    box_width: dict
        A dictionary defining the box width of each box type.

    output: str
        The folder where all instructions should be stored.
        Assumed to end on '\'.

    Returns
    -------
    """
    for order, boxes in packing.items():
        num_box = 0
        for box, items in boxes.items():
            box = box[0]
            __draw_box_packing(order, num_box, box, box_length[box], box_width[box], items, output)
            num_box += 1
