# Copyright (c) 2020 Janos Piddubnij 
# Copyright (c) 2020 Moritz Dederichs


from matplotlib import pyplot as plt
from mpl_toolkits.mplot3d.art3d import Poly3DCollection


def plot_solution(packing: dict, length_box: dict, width_box: dict, height_box: dict, color: str = '#ffbf00',
                  interval: float = 0.25):
    """ This function plots a 3D visualization/animation of the packing solution.

    Parameters
    ----------
    packing: dict
        Dictionary containing the item placements inside each box.

    length_box: dict
        Dictionary containing the length of all box types.

    width_box: dict
        Dictionary containing the width of all box types.

    height_box: dict
        Dictionary containing the height of all box types.

    color: str
        Hexadecimal colour code used to colour all items.

    interval: float
        The time to wait between plotting to subsequent items, i.e. the 'animation' speed.

    Returns
    -------
    """
    for order, boxes in packing.items():
        for box, items in boxes.items():
            fig = plt.figure()
            ax = fig.add_subplot(111, projection='3d')
            ax.set_title(box)
            ax.invert_zaxis()

            for item, placement in items.items():
                if interval > 0:
                    plt.pause(interval)
                x_min, y_min, z_min, x_max, y_max, z_max = placement[:6]

                # Plot all item sides
                __plot_top_side(ax, color, x_max, x_min, y_max, y_min, z_max)

                __plot_bottom_side(ax, color, x_max, x_min, y_max, y_min, z_min)

                __plot_left_side(ax, color, x_min, y_max, y_min, z_max, z_min)

                __plot_right_side(ax, color, x_max, y_max, y_min, z_max, z_min)

                __plot_front_side(ax, color, x_max, x_min, y_min, z_max, z_min)

                __plot_back_side(ax, color, x_max, x_min, y_max, z_max, z_min)

                ax.set_xlim(length_box[box[0]], 0)
                ax.set_ylim(0, width_box[box[0]])
                ax.set_zlim(0, height_box[box[0]])

    print("Done!")
    plt.show()


def __plot_back_side(ax, color, x_max, x_min, y_max, z_max, z_min):
    """ This function plots the back side of a 3D rectangle specified by the given coordinates.

    Parameters
    ----------
    ax
        Axes object to plot the rectangle side.

    color
        Colour of the rectangle side to plot.

    x_max
        The maximal x coordinate of the 3D rectangle to plot.

    x_min
        The minimal x coordinate of the 3D rectangle to plot.

    y_max
        The maximal y coordinate of the 3D rectangle to plot.

    z_max
        The maximal z coordinate of the 3D rectangle to plot.

    z_min
        The minimal z coordinate of the 3D rectangle to plot.

    Returns
    -------
    """
    x_coord = [x_min, x_min, x_max, x_max]
    y_coord = [y_max, y_max, y_max, y_max]
    z_coord = [z_min, z_max, z_max, z_min]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_front_side(ax, color, x_max, x_min, y_min, z_max, z_min):
    """ This function plots the front side of a 3D rectangle specified by the given coordinates.

        Parameters
        ----------
        ax
            Axes object to plot the rectangle side.

        color
            Colour of the rectangle side to plot.

        x_max
            The maximal x coordinate of the 3D rectangle to plot.

        x_min
            The minimal x coordinate of the 3D rectangle to plot.

        y_min
            The minimal y coordinate of the 3D rectangle to plot.

        z_max
            The maximal z coordinate of the 3D rectangle to plot.

        z_min
            The minimal z coordinate of the 3D rectangle to plot.

        Returns
        -------
        """
    x_coord = [x_min, x_min, x_max, x_max]
    y_coord = [y_min, y_min, y_min, y_min]
    z_coord = [z_min, z_max, z_max, z_min]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_right_side(ax, color, x_max, y_max, y_min, z_max, z_min):
    """ This function plots the right side of a 3D rectangle specified by the given coordinates.

        Parameters
        ----------
        ax
            Axes object to plot the rectangle side.

        color
            Colour of the rectangle side to plot.

        x_max
            The maximal x coordinate of the 3D rectangle to plot.

        y_max
            The maximal y coordinate of the 3D rectangle to plot.

        y_min
            The minimal y coordinate of the 3D rectangle to plot.

        z_max
            The maximal z coordinate of the 3D rectangle to plot.

        z_min
            The minimal z coordinate of the 3D rectangle to plot.

        Returns
        -------
        """
    x_coord = [x_max, x_max, x_max, x_max]
    y_coord = [y_min, y_min, y_max, y_max]
    z_coord = [z_min, z_max, z_max, z_min]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_left_side(ax, color, x_min, y_max, y_min, z_max, z_min):
    """ This function plots the left side of a 3D rectangle specified by the given coordinates.

        Parameters
        ----------
        ax
            Axes object to plot the rectangle side.

        color
            Colour of the rectangle side to plot.

        x_min
            The minimal x coordinate of the 3D rectangle to plot.

        y_max
            The maximal y coordinate of the 3D rectangle to plot.

        y_min
            The minimal y coordinate of the 3D rectangle to plot.

        z_max
            The maximal z coordinate of the 3D rectangle to plot.

        z_min
            The minimal z coordinate of the 3D rectangle to plot.

        Returns
        -------
        """
    x_coord = [x_min, x_min, x_min, x_min]
    y_coord = [y_min, y_min, y_max, y_max]
    z_coord = [z_min, z_max, z_max, z_min]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_bottom_side(ax, color, x_max, x_min, y_max, y_min, z_min):
    """ This function plots the bottom side of a 3D rectangle specified by the given coordinates.

        Parameters
        ----------
        ax
            Axes object to plot the rectangle side.

        color
            Colour of the rectangle side to plot.

        x_max
            The maximal x coordinate of the 3D rectangle to plot.

        x_min
            The minimal x coordinate of the 3D rectangle to plot.

        y_max
            The maximal y coordinate of the 3D rectangle to plot.

        y_min
            The minimal y coordinate of the 3D rectangle to plot.

        z_min
            The minimal z coordinate of the 3D rectangle to plot.

        Returns
        -------
        """
    x_coord = [x_min, x_min, x_max, x_max]
    y_coord = [y_min, y_max, y_max, y_min]
    z_coord = [z_min, z_min, z_min, z_min]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_top_side(ax, color, x_max, x_min, y_max, y_min, z_max):
    """ This function plots the top side of a 3D rectangle specified by the given coordinates.

        Parameters
        ----------
        ax
            Axes object to plot the rectangle side.

        color
            Colour of the rectangle side to plot.

        x_max
            The maximal x coordinate of the 3D rectangle to plot.

        x_min
            The minimal x coordinate of the 3D rectangle to plot.

        y_max
            The maximal y coordinate of the 3D rectangle to plot.

        y_min
            The minimal y coordinate of the 3D rectangle to plot.

        z_max
            The maximal z coordinate of the 3D rectangle to plot.

        Returns
        -------
        """
    x_coord = [x_min, x_min, x_max, x_max]
    y_coord = [y_min, y_max, y_max, y_min]
    z_coord = [z_max, z_max, z_max, z_max]
    __plot_rect(x_coord, y_coord, z_coord, color, ax)


def __plot_rect(x_coord, y_coord, z_coord, color, ax):
    """ This function plots a rectangle into the given axes object.

    Parameters
    ----------
    x_coord
        List of all x coordinates used for the four edges of the rectangle.

    y_coord
        List of all y coordinates used for the four edges of the rectangle.

    z_coord
        List of all z coordinates used for the four edges of the rectangle.

    color
        String specifying the colour of the rectangle as hex code.

    ax
        The axes object to plot the rectangle.

    Returns
    -------
    """
    vertices = [list(zip(x_coord, y_coord, z_coord))]
    surface = Poly3DCollection(vertices, edgecolors='k')
    surface.set_facecolor(color)
    ax.add_collection3d(surface)
