# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Janos Piddubnij

from typing import TextIO, Tuple
import os


def __write_item_alignment(file: TextIO, placement: Tuple, obj: str, container: str):
    """ Writes the alignment of an object within its container.

    Parameters
    ----------
    file: TextIO
        The file to write the alignment.

    placement: Tuple
        The placement of the item within the box.

    obj: str
        The object type to be packed (item or box).

    container: str
        The container type to contain all objects (box or pallet).

    Returns
    -------
    """
    alignment = {
        'H': 'height',
        'L': 'length',
        'W': 'width'
    }
    file.write(f'Align the {alignment.get(placement[6])} of the {obj} along the length of the {container}\n')
    file.write(f'Align the {alignment.get(placement[7])} of the {obj} along the width of the {container}\n\n')


def __write_item_placement(file: TextIO, item: str, placement: Tuple, obj: str):
    """ Writes the object placement within a container.

    Parameters
    ----------
    file: TextIO
        The file to write the placement.

    item: str
        The item type to place.

    placement: Tuple
        The placement coordinates within the box.

    obj: str
        The object type to be packed (item or box).

    Returns
    -------
    """
    file.write(f'Place {obj} of type {item} at position x: {abs(placement[0])}, y: {abs(placement[1])}\n')


def __write_prepare_message(file: TextIO, pallet: bool):
    """ Writes what type of objects need to be prepared.

    Parameters
    ----------
    file: TextIO
        The file to write the placement.

    pallet: bool
        Specifies if boxes need to be prepared instead of items.

    Returns
    -------
    """
    if pallet:
        file.write('Prepare the following boxes:\n')
    else:
        file.write(f'Prepare the following items:\n')


def __write_general_instruction(file: TextIO, pallet: bool):
    """ Writes the general packing instructions.

    Parameters
    ----------
    file: TextIO
        The file to write the placement.

    pallet: bool
        Specifies if boxes should be packed instead of items.

    Returns
    -------
    """
    if pallet:
        file.write('Place each box at the specified location on top of all previously packed boxes\n')
    else:
        file.write('Place each item at the specified location on top of all previously packed items\n')


def __write_error(file: TextIO, pallet: bool):
    """ Writes that the objects could not be packed within the container.

    Parameters
    ----------
    file: TextIO
        The file to write the placement.

    pallet: bool
        Specifies if boxes are packed onto pallets instead of items into boxes.

    Returns
    -------
    """
    if pallet:
        file.write('The boxes cannot be packed onto the available pallets')
    else:
        file.write('The items cannot be packed into the available boxes\n')


def create_packing_instructions(packing: dict, output: str, alg: str, pallet: bool):
    """ Creates text file declaring the packing instructions for each order.

    Parameters
    ----------
    packing: dict
        A dictionary stating for each object in an order the container and location within the container.

    output: str
        Path to the output file.

    alg: str
        The algorithm used to compute the object placements inside each container.

    pallet: bool
        Specifies if boxes are packed onto pallets instead of items into boxes.

    Returns
    -------
    """
    if pallet:
        obj = 'box'
        container = 'pallet'
    else:
        obj = 'item'
        container = 'box'

    with open(os.path.join(output, f'instructions_{alg}.txt'), mode='w', encoding='utf-16') as file:
        for order, boxes in packing.items():
            file.write(f'Packing instructions for order {order}\n')
            if not boxes:
                __write_error(file, pallet)
            __write_general_instruction(file, pallet)
            file.write(55 * '-' + '\n')
            for box, items in boxes.items():
                __write_prepare_message(file, pallet)
                item_names = [i[0] for i in items.keys()]
                item_counts = dict([i, item_names.count(i)] for i in sorted(set(item_names)))
                for item, item_count in item_counts.items():
                    file.write(f'{item_count} x {item}\n')
                file.write(f'\nUse a {container} of type {box[0]}\n')
                file.write(f'Rotate the {container} such that the length side faces you\n')
                file.write(55 * '-' + '\n')
                sorted_items = [key for key, values in sorted(items.items(), key=lambda i: [i[1][2], i[1][1], i[1][0]])]
                for item in sorted_items:
                    __write_item_placement(file, item[0], items[item], obj)
                    __write_item_alignment(file, items[item], obj, container)
            file.write(55 * '-' + '\n\n')
