# Copyright (c) 2020 Luca Koczula

from .calculate_gabm_packing_statistics import calculate_packing_statistics
import os


def create_gabm_packing_instructions(best_chromosome, generations, population_size, order, prob_c, prob_t, E,
                                     considered_items, considered_EMS, length_box, width_box, height_box, length_item,
                                     width_item, height_item, weight_item, used_time='N/A', output_file_path=None):
    """ This function will print the Genetic Algorithm Best Match packing instructions either to the standard output
        or to a file.

    Parameters
    ----------
    best_chromosome
        The best chromosome computed during the Genetic Algorithm Best Match optimisation.

    generations
        Number of generations.

    population_size
        Number of chromosomes in each generation.

    order
        ID of the order to generate instructions for.

    prob_c
        Probability prob_c.

    prob_t
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    considered_items
        Number of items considered in each step of the heuristic.

    considered_EMS
        Number of EMS considered in each iteration.

    length_box
        Dictionary containing the length of all box types.

    width_box
        Dictionary containing the width of all box types.

    height_box
        Dictionary containing the height of all box types.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    weight_item
        Dictionary containing the weight of all items.

    used_time: optional
        Time needed for the calculation of the order.

    output_file_path: optional
        Path to a text file, if the instructions should be stored in an output file.

    Returns
    -------
    """
    instruction = __create_instruction_string(best_chromosome, generations, population_size, order, prob_c, prob_t,
                                              E, considered_items, considered_EMS, length_box, width_box, height_box,
                                              length_item, width_item, height_item, weight_item, used_time)
    if output_file_path:
        os.makedirs(output_file_path, exist_ok=True)
        file_path = os.path.join(output_file_path, f"instructions_{order}.txt")
        with open(file_path, 'w', encoding='utf-16') as file:
            file.write(instruction)
    else:
        print(instruction)


def __create_instruction_string(best_chromosome, generations, population_size, order, prob_c, prob_t, E,
                                considered_items, considered_EMS, length_box, width_box, height_box, length_item,
                                width_item, height_item, weight_item, used_time='N/A'):
    """ This function will create the packing instruction string.

    Parameters
    ----------
    best_chromosome
        The best chromosome computed during the Genetic Algorithm Best Match optimisation.

    generations
        Number of generations.

    population_size
        Number of chromosomes in each generation.

    order
        ID of the order to generate instructions for.

    prob_c
        Probability prob_c.

    prob_t
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    considered_items
        Number of items considered in each step of the heuristic.

    considered_EMS
        Number of EMS considered in each iteration.

    length_box
        Dictionary containing the length of all box types.

    width_box
        Dictionary containing the width of all box types.

    height_box
        Dictionary containing the height of all box types.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    weight_item
        Dictionary containing the weight of all items.

    used_time: optional
        Time needed for the calculation of the order.

    Returns
    -------
    """
    used_boxes, max_box, max_items, volume_ratios = calculate_packing_statistics(best_chromosome=best_chromosome,
                                                                                 length_box=length_box,
                                                                                 width_box=width_box,
                                                                                 height_box=height_box,
                                                                                 length_item=length_item,
                                                                                 width_item=width_item,
                                                                                 height_item=height_item,
                                                                                 weight_item=weight_item)
    instruction = '\n '
    instruction += __header()
    instruction += '\n'
    instruction += __setting(order, generations, population_size, prob_c, prob_t, E, considered_items, considered_EMS)
    instruction += '\n'
    instruction += __table_header()
    for i in range(len(best_chromosome.BPS)):
        instruction += __packing(best_chromosome, i)
    instruction += __separator()
    instruction += __results(best_chromosome, used_time, used_boxes, max_items, max_box)
    instruction += __empty_line()
    total_weight_of_order = 0
    for box in volume_ratios:
        instruction += __volume_ratios(box, volume_ratios)
        total_weight_of_order += volume_ratios[box][1]
    instruction += __separator()
    instruction += __total_order_stats(total_weight_of_order, best_chromosome)
    instruction += __separator()
    return instruction


def __header():
    """ This function creates the instruction header.

    Returns
    -------
    """
    return f"{'+++++ BEST MATCH HEURISTIC RESULTS: +++++':^115}\n"


def __setting(order, generations, population_size, prob_c, prob_t, E, considered_items, considered_EMS):
    """ This function creates the general algorithm settings.

    Parameters
    ----------
    order
        ID of the order to generate instructions for.

    generations
        Number of generations.

    population_size
        Number of chromosomes in each generation.

    prob_c
        Probability prob_c.

    prob_t
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    considered_items
        Number of items considered in each step of the heuristic.

    considered_EMS
        Number of EMS considered in each iteration.

    Returns
    -------
    """
    string = f"+{12 * '-'}+{40 * '-'} Selected Parameters: {38 * '-'}+\n"
    algorithm = f"| Order: {str(order)}"
    algorithm += f" | #Generations: {str(generations)}"
    algorithm += f" | #Chromosomes: {str(population_size)}"
    algorithm += f" | prob_c: {str(prob_c)}"
    algorithm += f" | prob_t: {str(prob_t)}"
    algorithm += f" | E: {str(E)} |"
    heuristic = f"| Considered Items: {str(considered_items)}"
    heuristic += f" | Considered EMS: {str(considered_EMS)} |"
    string += f"| ALGORITHM: |{algorithm:^100}|\n"
    string += f"| HEURISTIC: |{heuristic:^100}|\n"
    string += f"+{12 * '-'}+{100 * '-'}+\n"
    return string


def __table_header():
    """ This function creates the general table header for the actual packing instructions.

    Returns
    -------
    """
    string = f"+{8 * '-'} Input BPS: {8 * '-'}+"
    string += f"{8 * '-'} Input CLS: {8 * '-'}+ "
    string += f" +{12 * '-'} Output Packing Assignment: {12 * '-'}+\n"
    string += f"|{28 * ' '}|{28 * ' '}|  |{52 * ' '}|\n"
    return string


def __packing(best_chromosome, index):
    """ This function creates the packing instruction for each item.

    Parameters
    ----------
    best_chromosome
        The best chromosome computed during the Genetic Algorithm Best Match optimisation.

    index
        The index of the current item to pack.

    Returns
    -------
    """
    string = f"| {str(best_chromosome.BPS[index]):^26} "
    string += f"| {str(best_chromosome.CLS[index]):^26} | "
    instruction = f"Place item {str(best_chromosome.packing_sequence[index][0])} "
    instruction += f"into box {str(best_chromosome.CLS[best_chromosome.packing_sequence[index][1] - 1])}."
    string += f" | {instruction:^50} |\n"
    return string


def __separator():
    """ This function creates a separator line.

    Returns
    -------
    """
    return f"+{57 * '-'}+  +{52 * '-'}+\n"


def __empty_line():
    """ This function creates an empty line.

    Returns
    -------
    """
    return f"|  {53 * '-'}  |  |  {48 * '-'}  |\n"


def __results(best_chromosome, used_time, used_boxes, max_items, max_box):
    """ This function creates the result statistics for the best solution.

    Parameters
    ----------
    best_chromosome
        The best chromosome computed during the Genetic Algorithm Best Match optimisation.

    used_time
        Time needed for the calculation of the order.

    used_boxes
        Boxes used for the optimal packing solution.

    max_items
        Maximal number of items packed into a single box.

    max_box
        ID of the box containing the most items within the optimal solution.

    Returns
    -------
    """
    generation = f"Generation of best found solution: {str(best_chromosome.generation)}"
    fitness = f"Overall Fitness of Packing Sequence: {str(round(best_chromosome.fitness, 8))}"
    stats = f"Needed {round(used_time, 4)} s to place {len(best_chromosome.BPS)} items into {len(set(used_boxes))} boxes."
    top = f"Most items ({str(max_items)}) in box {str(best_chromosome.CLS[max_box - 1])}"
    string = f"|{generation:^57}|  |{fitness:^52}|\n"
    string += f"|{stats:^57}|  |{top:^52}|\n"
    return string


def __volume_ratios(box, volume_ratios):
    """ This function writes the volume ratio for each box.

    Parameters
    ----------
    box
        The box to consider when creating the volume ratio.

    volume_ratios
        Dictionary containing the volume ratio and total weight of the specified box.

    Returns
    -------
    """
    weight_of_box = f"Total weight of box {str(box)}: {str(round(volume_ratios[box][1] / 1000, 3))} kg"
    box_ratio = f"Volume ratio of box {str(box)}: {str(round(volume_ratios[box][0], 4))}"
    string = f"|{weight_of_box:^57}|  |{box_ratio:^52}|\n"
    return string


def __total_order_stats(total_weight_of_order, best_chromosome):
    """ This function writes the total weight and cost of the order.

    Parameters
    ----------
    total_weight_of_order
        The total weight of the order.

    best_chromosome
        The chromosome that contains the packing optimal solution.

    Returns
    -------
    """
    weight = f"Total weight of order: {str(round(total_weight_of_order / 1000, 3))} kg"
    cost = f"Total cost of order (DHL pricing): EUR {str(round(best_chromosome.cost, 2))}"
    string = f"|{weight:^57}|  |{cost:^52}|\n"
    return string
