# Copyright (c) 2020 Luca Koczula

from collections import defaultdict
from inform_or_gabm_objects.chromosome import Chromosome


def calculate_packing_statistics(best_chromosome: Chromosome, length_box: dict, width_box: dict, height_box: dict,
                                 length_item: dict, width_item: dict, height_item: dict,
                                 weight_item: dict) -> (list, tuple, int, dict):
    """ This function calculates some packing statistics for the Genetic Algorithm Best Match Heuristic.

    Parameters
    ----------
    best_chromosome: Chromosome
        The chromosome object to use for the calculations.

    length_box: dict
        Dictionary containing the length of all boxes.

    width_box: dict
        Dictionary containing the width of all boxes.

    height_box: dict
        Dictionary containing the height of all boxes.

    length_item: dict
        Dictionary containing the length of all items.

    width_item: dict
        Dictionary containing the width of all items.

    height_item: dict
        Dictionary containing the height of all items.

    weight_item: dict
        Dictionary containing the width of all items.

    Returns
    -------
    list:
        The list of all boxes used for the optimal solution.
    tuple:
        A tuple with the type and ID of the box containing the most items.
    int:
        The maximal number of items packed into one box.
    dict:
        A dictionary specifying the volume ratio of every box used in the optimal solution.
    """

    # calculate max number of items in a box
    used_boxes = []
    volume_ratios = {}
    items_per_box = defaultdict(list)

    for i in range(len(best_chromosome.packing_sequence)):
        # append all used boxes to the list 'usedBoxes'
        used_boxes.append(best_chromosome.packing_sequence[i][1])

        # create dictionary with all items in each used box
        items_per_box[best_chromosome.packing_sequence[i][1]].append(best_chromosome.packing_sequence[i][0][0])

    # determine box with the most items
    max_box = max(set(used_boxes), key=used_boxes.count)
    # determine number of boxes in that box
    max_items = used_boxes.count(max_box)

    for box in items_per_box:
        # get current box name and calculate the volume
        box_tuple = best_chromosome.CLS[box - 1]
        box_volume = (length_box[box_tuple[0]] * width_box[box_tuple[0]] * height_box[box_tuple[0]])

        # calculate the combined item volume in the current box
        item_volume = 0
        item_weight = 0
        for i in items_per_box[box]:
            item_volume += (length_item[i] * width_item[i] * height_item[i])
            item_weight += weight_item[i]

        vol_ratio = item_volume / box_volume

        # create a dictionary containing all used boxes and their individual volume ratio
        volume_ratios[box_tuple] = vol_ratio, item_weight

    return used_boxes, max_box, max_items, volume_ratios
