# Copyright (c) 2020 Yordan Manolov
# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Janos Piddubnij

import csv
from typing import Tuple, Dict, Any, DefaultDict
from collections import defaultdict


def name_to_number(order_name: str) -> int:
    """

    Parameters
    ----------
    order_name
    The full name of the current order

    Returns
    -------
    The name without non-numerals, stripped of leading zeroes

    """
    return int("".join(c for c in order_name if c.isdigit()))


def parse_generated_csv(csv_file: str,
                        delimiter: str = ';',
                        encoding: str = 'utf-16',
                        first: int = None,
                        last: int = None) -> Tuple[DefaultDict[Any, list],
                                                   Dict[str, int],
                                                   Dict[str, int],
                                                   Dict[str, int],
                                                   Dict[str, int]]:
    """ This functions reads the given csv file and returns a gurobi like data collection to use in the optimisation process

    Parameters
    ----------
    csv_file: str
        The CSV file to be processed.
        The file needs to have the same data format as produced by Generator.py provided by INFORM.

    delimiter: str, optional
        The column separator used within the CSV file.
        If not provided, this parameter will default to ';'.

    encoding: str, optional
        The encoding used to store/write the provided CSV file.
        IF not provided, this parameter will default to 'utf-16'.

    Returns
    -------
    Tuple[dict, dict, dict, dict, dict]
        5 shallow dicts containing the order and item information contained in the CSV file.
        The first dict has the order number as key while all other dicts have specific item names as keys.
        1. A dict containing all items of an order. Each dict value is a list of tuples of (item type, number of items).
        2. A dict containing the length of each individual item.
        3. A dict containing the width of each individual item.
        4. A dict containing the height of each individual item.
        5. A dict containing the weight of each individual item.
    """
    orders = defaultdict(list)
    length_items = dict()
    width_items = dict()
    height_items = dict()
    weight_items = dict()
    with open(csv_file, 'r', encoding=encoding) as file:
        reader = csv.reader(file, delimiter=delimiter)
        next(reader)
        for row in reader:
            order_name = row[0].strip()
            order_nr = name_to_number(order_name)
            item_name = row[6].strip()

            if first and last:
                if order_nr < first or order_nr > last:
                    continue
            orders[order_name].append((item_name, int(row[12])))
            length_items[item_name] = int(row[8])
            width_items[item_name] = int(row[9])
            height_items[item_name] = int(row[10])
            weight_items[item_name] = int(row[11])
    return orders, length_items, width_items, height_items, weight_items
