# -*- coding: utf-8 -*-

"""
Parser specific to the EP heuristic.
"""

#  Copyright (c) 2020.  Yordan Manolov <yordan DOT manolov AT rwth DASH aachen DOT de >
#  Copyright (c) 2020.  Moritz Dederichs

import argparse
import csv
from typing import List, Generator

from inform_or_ep_objects.item import Item
from inform_or_ep_objects.order import Order
from inform_or_parser.parse_csv_genetic_approach import parse_generated_csv

ORDER_COLUMN = "Order ID"
ARTICLE_COLUMN = "Article ID"
ARTICLE_COLUMN_COUNT = "Units"
ARTICLE_COLUMN_LENGTH = "Length (cm)"
ARTICLE_COLUMN_HEIGHT = "Height (cm)"
ARTICLE_COLUMN_WEIGHT = "Weight (g)"
ARTICLE_COLUMN_WIDTH = "Width (cm)"


def verify_input(name, delimiter=';', encoding='utf-16') -> bool:
    """

    Parameters
    ----------
    name
    delimiter
    encoding

    Returns
    -------
    True if the specified file is a correct CSV file with complete information.
    """
    required_fields = [ORDER_COLUMN, ARTICLE_COLUMN, ARTICLE_COLUMN_COUNT,
                       ARTICLE_COLUMN_LENGTH, ARTICLE_COLUMN_HEIGHT,
                       ARTICLE_COLUMN_WEIGHT, ARTICLE_COLUMN_WIDTH]
    try:
        with open(name, mode="r", encoding=encoding) as csv_file:
            csv_reader = csv.DictReader(csv_file, delimiter=delimiter)
            for field in required_fields:
                if field in csv_reader.fieldnames:
                    pass
    except TypeError:
        raise Exception("""Nice try, Joker.
                     Now give me a valid CSV file with orders.""")
    else:
        return True


def parse_file_as_objects(args: argparse.Namespace,
                          delimiter: str = ";",
                          encoding: str = "utf-16") -> Generator[Order, None, None]:
    """
    Wrapper for the original parser.
    Outputs objects instead of dictionaries. Fast.

    Parameters
    ----------
    args:
        Basic arguments, especially file name, first and last order to process
    delimiter : str, optional
        The default is ";".
    encoding : str, optional
        The default is "utf-16"

    Returns
    -------
    Generator[Order]
        Orders (objects 1 by 1)
    """
    name = args.input
    file_beginning = args.first
    file_end = args.last
    verify_input(name, delimiter, encoding)
    orders, lengths, widths, heights, weights = parse_generated_csv(name,
                                                                    delimiter,
                                                                    encoding,
                                                                    file_beginning,
                                                                    file_end)

    for key, item_list in orders.items():
        order = Order(key, list())
        for item_id, count in item_list:
            order.itemlist.append(
                Item(item_id, count, lengths[item_id], widths[item_id],
                     heights[item_id], weights[item_id]))
        yield order

