# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Janos Piddubnij

from gurobipy import Model, GRB, quicksum
from typing import Dict, Tuple, Any


def __termination_criterion(model, where):
    """ This termination criterion ensures that the optimizer will not get stuck trying all optimal solutions if there
        are multiple possible solutions.

    Parameters
    ----------
    model
    where

    Returns
    -------
    """
    if where == GRB.Callback.MIP:
        if model._last_sol_time is not None:
            now = model.cbGet(GRB.Callback.RUNTIME)
            if now - model._last_sol_time > 5:
                model.terminate()

    if where == GRB.Callback.MIPSOL:
        if not model._time_limit_disabled:
            model.params.TimeLimit = GRB.INFINITY
            model._time_limit_disabled = True
        now = model.cbGet(GRB.Callback.RUNTIME)
        model._last_sol_time = now


def __formalise_solution(model: Model, items: list, width_items: dict, height_items: dict, length_items: dict) -> Dict[Tuple, Tuple]:
    """ This function transform the optimal solution of a model into a processable and readable dictionary.

    Parameters
    ----------
    model: gurobipy.Model
        The model whose solution should be formalised.

    items: list
        The list of items that have been packed in the given box.

    width_items: dict
        A dictionary specifying the width of each item type.

    height_items: dict
        A dictionary specifying the height of each item type.

    length_items: dict
        A dictionary specifying the length of each item type.

    Returns
    -------
    Dict[Tuple, Tuple]
        The dictionary specifying the placement coordinates and orientation of each item within the box.
    """
    formalised_solution = dict()
    if model.status in (GRB.status.OPTIMAL, GRB.status.INTERRUPTED):
        align = {
            1: 'W',
            2: 'H',
            3: 'L'
        }
        for i in items:
            x_begin = model.getVarByName(f'x_{i}')
            y_begin = model.getVarByName(f'y_{i}')
            z_begin = model.getVarByName(f'z_{i}')
            w_x = model.getVarByName(f'w_x_{i}')
            w_y = model.getVarByName(f'w_y_{i}')
            w_z = model.getVarByName(f'w_z_{i}')
            h_x = model.getVarByName(f'h_x_{i}')
            h_y = model.getVarByName(f'h_y_{i}')
            h_z = model.getVarByName(f'h_z_{i}')
            l_x = model.getVarByName(f'l_x_{i}')
            l_y = model.getVarByName(f'l_y_{i}')
            l_z = model.getVarByName(f'l_z_{i}')
            x_end = x_begin.x + w_x.x * width_items[i[0]] + h_x.x * height_items[i[0]] + l_x.x * length_items[i[0]]
            y_end = y_begin.x + w_y.x * width_items[i[0]] + h_y.x * height_items[i[0]] + l_y.x * length_items[i[0]]
            z_end = z_begin.x + w_z.x * width_items[i[0]] + h_z.x * height_items[i[0]] + l_z.x * length_items[i[0]]
            x_align = 1 * w_x.x + 2 * h_x.x + 3 * l_x.x
            y_align = 1 * w_y.x + 2 * h_y.x + 3 * l_y.x
            z_align = 1 * w_z.x + 2 * h_z.x + 3 * l_z.x
            formalised_solution[i] = (x_begin.x, y_begin.x, z_begin.x, x_end, y_end, z_end, align.get(x_align), align.get(y_align), align.get(z_align))
    return formalised_solution


def place_items_in_box(items: list, width_items: Dict[str, int], height_items: Dict[str, int],
                       length_items: Dict[str, int], width_box: int, height_box: int,
                       length_box: int) -> Dict[Any, Tuple[float, float, float, str, str, str]]:
    """ This function tries to place a given set of items into a given box.
        The minimisation criterion for this model is to reduce the overall sum of z-coordinates (height)
        to simulate gravity.

    Parameters
    ----------
    items: list
        The list of items that should be packed in the given box.

    width_items: Dict[str, int]
        A dictionary defining the width of each item type.

    height_items: Dict[str, int]
        A dictionary defining the height of each item type.

    length_items: Dict[str, int]
        A dictionary defining the length of each item type.

    width_box: int
        The width of the box to pack.

    height_box: int
        The height of the box to pack.

    length_box: int
        The length of the box to pack.

    Returns
    -------
    Dict[Any, Tuple[float, float, float, str, str, str]]
        A dictionary specifying for each item the exact coordinates and the item orientation.
        1. x-coordinate of the item.
        2. y-coordinate of the item.
        3. z-coordinate of the item.
        4. Whether height (h), length (l) or width (w) of the item is aligned to the length of the box.
        5. Whether height (h), length (l) or width (w) of the item is aligned to the width of the box.
        6. Whether height (h), length (l) or width (w) of the item is aligned to the height of the box.
    """
    model = Model('Single_Box_INFORM')
    model.params.TimeLimit = 15

    # Calculating the upper bound for the item coordinates within a box
    big_m = 0
    if height_box > big_m:
        big_m = height_box
    if length_box > big_m:
        big_m = length_box
    if width_box > big_m:
        big_m = width_box
    # big_m = max([width_box, height_box, length_box])

    # Variables

    # Variables representing the x-, y- and z-coordinates of item i within the box
    x = {}
    y = {}
    z = {}
    # Variables indicating whether the width of the item is aligned to the x-, y- or z-side of the box
    w_x = {}
    w_y = {}
    w_z = {}
    # Variables indicating whether the height of the item is aligned to the x-, y- or z-side of the box
    h_x = {}
    h_y = {}
    h_z = {}
    # Variables indicating whether the length of the item is aligned to the x-, y- or z-side of the box
    l_x = {}
    l_y = {}
    l_z = {}
    for i in items:
        x[i] = model.addVar(name=f'x_{i}', vtype=GRB.INTEGER, lb=0, ub=big_m)
        y[i] = model.addVar(name=f'y_{i}', vtype=GRB.INTEGER, lb=0, ub=big_m)
        z[i] = model.addVar(name=f'z_{i}', vtype=GRB.INTEGER, lb=0, ub=big_m)
        w_x[i] = model.addVar(name=f'w_x_{i}', vtype=GRB.BINARY)
        w_y[i] = model.addVar(name=f'w_y_{i}', vtype=GRB.BINARY)
        w_z[i] = model.addVar(name=f'w_z_{i}', vtype=GRB.BINARY)
        h_x[i] = model.addVar(name=f'h_x_{i}', vtype=GRB.BINARY)
        h_y[i] = model.addVar(name=f'h_y_{i}', vtype=GRB.BINARY)
        h_z[i] = model.addVar(name=f'h_z_{i}', vtype=GRB.BINARY)
        l_x[i] = model.addVar(name=f'l_x_{i}', vtype=GRB.BINARY)
        l_y[i] = model.addVar(name=f'l_y_{i}', vtype=GRB.BINARY)
        l_z[i] = model.addVar(name=f'l_z_{i}', vtype=GRB.BINARY)

    # Variables indicating the relative position of two items i and j to each other
    a = {}
    b = {}
    c = {}
    d = {}
    e = {}
    f = {}
    for i in items:
        for j in items:
            if i != j:
                a[i, j] = model.addVar(name=f'a_{i}_{j}', vtype=GRB.INTEGER)
                b[i, j] = model.addVar(name=f'b_{i}_{j}', vtype=GRB.INTEGER)
                c[i, j] = model.addVar(name=f'c_{i}_{j}', vtype=GRB.INTEGER)
                d[i, j] = model.addVar(name=f'd_{i}_{j}', vtype=GRB.INTEGER)
                e[i, j] = model.addVar(name=f'e_{i}_{j}', vtype=GRB.INTEGER)
                f[i, j] = model.addVar(name=f'f_{i}_{j}', vtype=GRB.INTEGER)

    # Constraints

    # Items are not allowed to overlap
    # Additionally items must have at least one positional relation to any other item
    for i in items:
        for j in items:
            if i != j:
                model.addConstr(x[i] + width_items[i[0]] * w_x[i] + height_items[i[0]] * h_x[i] + length_items[i[0]] * l_x[i] <= x[j] + (1 - a[i, j]) * big_m)
                model.addConstr(x[j] + width_items[j[0]] * w_x[j] + height_items[j[0]] * h_x[j] + length_items[j[0]] * l_x[j] <= x[i] + (1 - b[i, j]) * big_m)
                model.addConstr(y[i] + width_items[i[0]] * w_y[i] + height_items[i[0]] * h_y[i] + length_items[i[0]] * l_y[i] <= y[j] + (1 - c[i, j]) * big_m)
                model.addConstr(y[j] + width_items[j[0]] * w_y[j] + height_items[j[0]] * h_y[j] + length_items[j[0]] * l_y[j] <= y[i] + (1 - d[i, j]) * big_m)
                model.addConstr(z[i] + width_items[i[0]] * w_z[i] + height_items[i[0]] * h_z[i] + length_items[i[0]] * l_z[i] <= z[j] + (1 - e[i, j]) * big_m)
                model.addConstr(z[j] + width_items[j[0]] * w_z[j] + height_items[j[0]] * h_z[j] + length_items[j[0]] * l_z[j] <= z[i] + (1 - f[i, j]) * big_m)
                model.addConstr(a[i, j] + b[i, j] + c[i, j] + d[i, j] + e[i, j] + f[i, j] >= 1)

    # No item i may be packed over the boundaries of the box
    for i in items:
        model.addConstr(x[i] + width_items[i[0]] * w_x[i] + height_items[i[0]] * h_x[i] + length_items[i[0]] * l_x[i] <= length_box)
        model.addConstr(y[i] + width_items[i[0]] * w_y[i] + height_items[i[0]] * h_y[i] + length_items[i[0]] * l_y[i] <= width_box)
        model.addConstr(z[i] + width_items[i[0]] * w_z[i] + height_items[i[0]] * h_z[i] + length_items[i[0]] * l_z[i] <= height_box)

    # Each side of an item i may only be aligned to one side of the box
    for i in items:
        model.addConstr(w_x[i] + w_y[i] + w_z[i] == 1)
        model.addConstr(h_x[i] + h_y[i] + h_z[i] == 1)
        model.addConstr(l_x[i] + l_y[i] + l_z[i] == 1)
        model.addConstr(w_x[i] + h_x[i] + l_x[i] == 1)
        model.addConstr(w_y[i] + h_y[i] + l_y[i] == 1)
        model.addConstr(w_z[i] + h_z[i] + l_z[i] == 1)

    model._last_sol_time = None
    model._time_limit_disabled = False

    model.setObjective(quicksum(z[i] for i in items), GRB.MINIMIZE)

    model.optimize(__termination_criterion)

    return __formalise_solution(model, items, width_items, height_items, length_items)
