# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Akshay Ganesh
# Copyright (c) 2020 Clinton Charles

from collections import defaultdict
from typing import List


class PeakFillingSlicePush:

    def __init__(self, item_sizes: dict, container_length: int, container_width: int, container_height: int,
                 container_name: str):
        """ Creates a scenario for the Peak-Filling Slice Push heuristic.
            Used to initialise all necessary class variables.

        Parameters
        ----------
        item_sizes: dict
            Dictionary defining the dimensions for all items to be packed in the current scenario.

        container_length: int
            Length of the container to pack all items.

        container_width: int
            Width of the container to pack all items.

        container_height: int
            Height of the container to pack all items.

        container_name: str
            Name of the container.
        """
        self.item_sizes = item_sizes
        self.container_height = container_height
        self.container_width = container_width
        self.container_length = container_length
        self.container_name = container_name

    @classmethod
    def __init_optimisation(cls) -> dict:
        """ Initialise the numeric variables that are used for the heuristic.

        Returns
        -------
        dict
            Dictionary containing all numeric variables necessary for the heuristic.
        """
        parameters = dict()
        parameters['num_pallets'] = 0
        parameters['num_item'] = 0
        parameters['x_slice'] = 0
        parameters['x_filled'] = 0
        parameters['y_filled'] = 0
        parameters['z_filled'] = 0
        parameters['x_rectangle'] = 0
        parameters['y_rectangle'] = 0
        parameters['x'] = 0
        parameters['y'] = 0
        return parameters

    @classmethod
    def __get_new_container(cls, params: dict):
        """ Create a new container to pack items.

        Parameters
        ----------
        params: dict
            Dictionary containing the numeric variables of the heuristic.

        Returns
        -------
        """
        params['num_pallets'] += 1
        params['num_item'] = 0
        print(f"Opening new pallet {params['num_pallets']}.")
        params['x_slice'] = 0
        params['x_filled'] = 0
        params['y_filled'] = 0
        params['z_filled'] = 0
        params['x_rectangle'] = 0
        params['y_rectangle'] = 0
        params['x'] = 0
        params['y'] = 0

    def __create_new_slice(self, items_to_pack: List[str], params: dict) -> bool:
        """ Tries to find a new slice within the current container.

        Parameters
        ----------
        items_to_pack: List[str]
            List of all items that remain to be packed.

        params: dict
            Dictionary containing the numeric variables of the heuristic.

        Returns
        -------
        bool
            True if a new slice could be found in the current container.
            False if it is not possible to create a new slice.
        """
        print('Searching for new Slice.')
        for item in items_to_pack:
            item_length = self.item_sizes[item][0]
            item_width = self.item_sizes[item][1]
            if item_length <= self.container_length - params['x_filled']:
                params['x'] = params['x_filled']
                params['y'] = 0
                params['x_slice'] = item_length
                params['x_filled'] += params['x_slice']
                print(f"Slice found at {params['x_rectangle']}, {params['x_slice']}, {params['y_rectangle']}")
                params['x_rectangle'] = item_length
                params['y_rectangle'] = item_width
                params['y_filled'] = params['y_rectangle']
                params['z_filled'] = 0
                return True
        return False

    def __pack_item_in_slice(self, item: str, item_length: int, item_width: int, item_height: int, params: dict,
                             packing: dict) -> bool:
        """ Pack the item in the current container slice.

        Parameters
        ----------
        item: str
            Name of the item to pack.

        item_length: int
            Length of the item to pack.

        item_width: int
            Width of the item to pack.

        item_height: int
            Height of the item to pack.

        params: dict
            Dictionary containing the numeric variables of the heuristic.

        packing: dict
            Dictionary containing the exact packing specifications for each item.

        Returns
        -------
        bool
            Returns True if the item could be packed into the current slice.
            False if the item could not be packed.
        """
        if params['z_filled'] + item_height <= self.container_height:
            print(f'Packing item of type {item}')
            x_start = params['x']
            y_start = params['y']
            z_start = params['z_filled']
            x_end = x_start + item_length
            y_end = y_start + item_width
            z_end = z_start + item_height
            placement = (x_start, y_start, z_start, x_end, y_end, z_end, 'L', 'W', 'H')
            packing[(self.container_name, params['num_pallets'])][(item, params['num_item'])] = placement
            params['x_rectangle'] = item_length
            params['y_rectangle'] = item_width
            params['z_filled'] = z_end
            params['num_item'] += 1
            return True
        return False

    def __create_new_rectangle(self, items_to_pack: List[str], params: dict) -> bool:
        """ Create a new rectangle on top of the current slice.

        Parameters
        ----------
        items_to_pack: List[str]
            List of all items that remain to be packed.

        params: dict
            Dictionary containing the numeric variables of the heuristic.

        Returns
        -------
        bool
            True if a new rectangle could be created.
            False if not.
        """
        for item in items_to_pack:
            item_length = self.item_sizes[item][0]
            item_width = self.item_sizes[item][1]
            if item_length <= params['x_slice'] and item_width <= self.container_width - params['y_filled']:
                params['x_rectangle'] = item_length
                params['y_rectangle'] = item_width
                params['y'] = params['y_filled']
                params['y_filled'] += item_width
                params['z_filled'] = 0
                return True
        return False

    def process(self, items_to_pack: List[str]) -> dict:
        """ Pack all items into as few containers as possible utilizing the Peak-Filling Slice Push heuristic.

        Parameters
        ----------
        items_to_pack: List[str]
            List of all items that have to be packed.

        Returns
        -------
        dict
            Dictionary containing the exact packing specifications for each item.
        """
        new_pallet = True
        parameters = self.__init_optimisation()
        packing = defaultdict(dict)
        while items_to_pack:
            if new_pallet:
                self.__get_new_container(parameters)
                new_pallet = False
            else:
                if not parameters['x_slice']:
                    if not self.__create_new_slice(items_to_pack, parameters):
                        print(f"Pallet {parameters['num_pallets']} is full")
                        new_pallet = True
                else:
                    print('Entering Slice.')
                    start_new_slice = True
                    for index, item in enumerate(items_to_pack):
                        item_length = self.item_sizes[item][0]
                        item_width = self.item_sizes[item][1]
                        item_height = self.item_sizes[item][2]
                        if item_length <= parameters['x_rectangle'] and item_width <= parameters['y_rectangle']:
                            if self.__pack_item_in_slice(
                                    item, item_length, item_width, item_height, parameters, packing):
                                del items_to_pack[index]
                                start_new_slice = False
                                break
                            if self.__create_new_rectangle(items_to_pack, parameters):
                                start_new_slice = False
                                break
                    if start_new_slice:
                        parameters['x_slice'] = 0
        return packing
